/*
 * Copyright (c) 2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: AbstractStoreStrategy.java 3118 2006-08-30 14:38:36Z jmettraux $
 */

//
// CachedWorkItemStorage.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org.worklist.impl.store;

import openwfe.org.MapUtils;
import openwfe.org.AbstractService;
import openwfe.org.ServiceException;
import openwfe.org.ApplicationContext;
import openwfe.org.xml.XmlUtils;
import openwfe.org.misc.Cache;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.expressions.FlowExpressionId;
import openwfe.org.worklist.store.StoreException;
import openwfe.org.worklist.store.WorkItemStorage;


/**
 * An intermediate storage, with a cache.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author$
 * <br>$Id$ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class CachedWorkItemStorage

    extends AbstractService

    implements WorkItemStorage

{

    /*
     *
     * made in Japan
     *
     */

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(CachedWorkItemStorage.class.getName());

    //
    // CONSTANTS & co

    /**
     * 'storage' : a CachedWorkItemStorage is put in front of another storage 
     * to do its caching job; This parameter points at that other storage.
     */
    public final static String P_STORAGE
        = "storage";

    /**
     * 'cacheSize' : how many workitems per store should be cached in memory ?
     * The default value is 1'000.
     */
    public final static String P_CACHE_SIZE
        = "cacheSize";

    private final static int DEFAULT_CACHE_SIZE = 1000;

    //
    // FIELDS

    private int cacheSize = -1;
    private String storageName = null;

    private java.util.Map workitemCaches = new java.util.HashMap();

    //
    // CONSTRUCTORS

    public void init 
        (final String serviceName, 
         final ApplicationContext context, 
         final java.util.Map serviceParams)
    throws 
        ServiceException
    {
        super.init(serviceName, context, serviceParams);

        //
        // determine which is the storage that is going to be cached

        this.storageName =
            MapUtils.getMandatoryString(serviceParams, P_STORAGE);

        log.info
            ("init() '"+serviceName+"' caches for '"+this.storageName+"'");

        //
        // how many workitmes should be cached for each store ?

        this.cacheSize = 
            MapUtils.getAsInt(serviceParams, P_CACHE_SIZE, DEFAULT_CACHE_SIZE);

        log.info
            ("init() '"+serviceName+
             "' will hold up to "+this.cacheSize+" workitems per store");
    }

    //
    // METHODS from WorkItemStorage

    /**
     * Inserts a workitem into the storage for given store.
     */
    public void storeWorkItem 
        (final String storeName, final InFlowWorkItem wi)
    throws 
        StoreException
    {
        if (log.isDebugEnabled())
            log.debug("storeWorkItem() '"+storeName+"'  "+wi.getId());

        //
        // store in cache

        getWorkitemCache(storeName).cacheWorkitem(wi);

        //
        // store in the real storage

        getCachedStorage().storeWorkItem(storeName, wi);
    }

    /**
     * Removes the workitem from the storage for a given store.
     */
    public void removeWorkItem 
        (final String storeName, final FlowExpressionId fei)
    throws 
        StoreException
    {
        if (log.isDebugEnabled())
            log.debug("removeWorkItem() '"+storeName+"'  "+fei);

        //
        // remove from cache

        getWorkitemCache(storeName).uncache(fei);

        //
        // remove from real storage

        getCachedStorage().removeWorkItem(storeName, fei);
    }

    /**
     * Retrieves a workitem given the storename and the workitem id.
     */
    public InFlowWorkItem retrieveWorkItem 
        (final String storeName, final FlowExpressionId fei)
    throws 
        StoreException
    {
        final StoreCache cache = getWorkitemCache(storeName);

        /*
        if (log.isDebugEnabled())
        {
            log.debug
                ("retrieveWorkItem() "+cache.size()+
                 " wis in cache for "+storeName);
        }
        */

        InFlowWorkItem wi = cache.getWorkitem(fei);

        if (wi != null)
        {
            if (log.isDebugEnabled())
            {
                log.debug
                    ("retrieveWorkItem() "+storeName+"  found in cache  "+fei);
            }

            return wi;
        }

        //
        // retrieve from underlying storage and cache

        wi = getCachedStorage().retrieveWorkItem(storeName, fei);

        if (log.isDebugEnabled())
            log.debug("retrieveWorkItem() "+storeName+"  had to load  "+fei);

        cache.cacheWorkitem(wi);

        return wi;
    }

    /**
     * Returns the number of workitems in a store.
     */
    public int countWorkItems 
        (final String storeName)
    throws 
        StoreException
    {
        return getCachedStorage().countWorkItems(storeName);
    }

    /**
     * Returns a list of workitems from a store.
     */
    public java.util.List listWorkItems 
        (final String storeName, final int limit)
    throws 
        StoreException
    {
        if (log.isDebugEnabled())
            log.debug("listWorkItems() "+storeName+" limit "+limit);

        final StoreCache cache = getWorkitemCache(storeName);

        synchronized(cache)
        {
            if ( ! cache.isInSync())
            {
                final java.util.List list = getCachedStorage()
                    .listWorkItems(storeName, limit);

                cache.cacheAll(list);
                cache.setInSync(true);
            }

            return cache.asList(limit);
        }
    }

    //
    // METHODS from Service

    /**
     * Status is outputted as a JDOM element. The status is various
     * information about a service activities and state.
     */
    public org.jdom.Element getStatus ()
    {
        final org.jdom.Element result = new org.jdom.Element(getName());

        result.addContent(XmlUtils.getClassElt(this));
        result.addContent(XmlUtils.getRevisionElt("$Id: FileWorkItemStorage.java 3118 2006-08-30 14:38:36Z jmettraux $"));

        final java.util.Iterator it = this.workitemCaches.values().iterator();
        while (it.hasNext())
        {
            final StoreCache sc = (StoreCache)it.next();

            final org.jdom.Element e = new org.jdom.Element("cache");
            e.setAttribute("store", sc.getStoreName());
            e.setAttribute("cacheSize", ""+sc.maxSize());
            e.setAttribute("size", ""+sc.size());
        }

        return result;
    }

    //
    // METHODS

    /**
     * Returns the storage object that this instance caches.
     */
    protected WorkItemStorage getCachedStorage ()
    {
        return (WorkItemStorage)this.getContext().get(this.storageName);
    }

    protected synchronized StoreCache getWorkitemCache (final String storeName)
    {
        StoreCache c = (StoreCache)this.workitemCaches.get(storeName);

        if (c == null)
        {
            c = new StoreCache(storeName, this.cacheSize);
            this.workitemCaches.put(storeName, c);
        }

        return c;
    }

    //
    // STATIC METHODS

    //
    // INNER CLASSES

    protected static class StoreCache 
        extends Cache
    {
        private String storeName = null;
        private boolean inSync = false;

        public StoreCache (final String storeName, final int size)
        {
            super(size);

            this.storeName = storeName;

            //if (log.isDebugEnabled())
            //    log.debug("new StoreCache() for "+this.storeName);
        }

        public String getStoreName ()
        {
            return this.storeName;
        }

        public boolean isInSync ()
        {
            return this.inSync;
        }

        public void setInSync (final boolean b)
        {
            this.inSync = b;
        }

        public InFlowWorkItem getWorkitem (final FlowExpressionId fei)
        {
            //debugGet(fei);
            return (InFlowWorkItem)this.get(fei.toString());
        }

        /*
        private void debugGet (final FlowExpressionId fei)
        {
            final String sFei = fei.toString();

            int i = 0;

            final java.util.Iterator it = this.keySet().iterator();
            while (it.hasNext())
            {
                final String key = (String)it.next();

                log.debug
                    ("?= "+i+
                     "\n   c "+key+
                     "\n   g "+sFei+
                     "\n   "+(key.equals(sFei)));

                i++;
            }
        }
        */

        public void cacheWorkitem (final InFlowWorkItem wi)
        {
            this.put(wi.getId().toString(), wi);

            /*
            if (log.isDebugEnabled())
            {
                log.debug
                    ("cacheWorkitem() ("+this.storeName+
                     " "+this.size()+")  "+wi.getId());
            }
            */
        }

        public void uncache (final FlowExpressionId fei)
        {
            this.remove(fei.toString());

            /*
            if (log.isDebugEnabled())
            {
                log.debug
                    ("uncache() ("+this.storeName+
                     " "+this.size()+")  "+fei);
            }
            */
        }

        public void cacheAll (final java.util.List workitems)
        {
            final java.util.Iterator it = workitems.iterator();
            while (it.hasNext())
            {
                final InFlowWorkItem wi = (InFlowWorkItem)it.next();

                this.cacheWorkitem(wi);
            }
        }

        private java.util.List asList ()
        {
            return new java.util.ArrayList(this.values());
        }

        public java.util.List asList (final int maxSize)
        {
            final java.util.List l = asList();

            if (l.size() <= maxSize) return l;

            return l.subList(0, maxSize);
        }

        public synchronized String toString ()
        {
            final StringBuffer sb = new StringBuffer();

            //sb.append("\n");

            sb
                .append("--- ")
                .append(this.storeName)
                .append(" ")
                .append(this.size())
                .append(" ---\n");

            final java.util.Iterator it = this.entrySet().iterator();
            while (it.hasNext())
            {
                final java.util.Map.Entry e = 
                    (java.util.Map.Entry)it.next();

                sb
                    .append("  ")
                    .append(e.getKey().toString())
                    //.append(" --> ")
                    //.append(((InFlowWorkItem)e.getValue()).getId().toString())
                    .append("\n");
            }

            sb.append("--- . ---");

            //sb.append("\n");

            return sb.toString();
        }
    }

}
