/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: BasicAuditLogger.java 3118 2006-08-30 14:38:36Z jmettraux $
 */

//
// BasicAuditLogger.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org.worklist.impl.audit;

import javax.security.auth.Subject;

import openwfe.org.MapUtils;
import openwfe.org.AbstractService;
import openwfe.org.ServiceException;
import openwfe.org.ApplicationContext;
import openwfe.org.auth.BasicPrincipal;
import openwfe.org.engine.workitem.Attribute;
import openwfe.org.engine.workitem.WorkItem;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.worklist.audit.AuditLogger;


/**
 * A very simple implementation of an AuditLogger.
 *
 * This code was sponsored by Obinary AG, Basel, Switzerland.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: BasicAuditLogger.java 3118 2006-08-30 14:38:36Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class BasicAuditLogger

    extends AbstractService

    implements AuditLogger

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(BasicAuditLogger.class.getName());

    //
    // CONSTANTS & co

    /**
     * The parameter 'logFile' is used to tell this worksession where it should
     * log its audit records. 
     * This parameter defaults to "audit.log".
     */
    public final static String P_LOG_FILE = "logFile";

    private final static String DEFAULT_LOG_FILE = "audit.log";

    /**
     * If the parameter 'append' is set to 'true', old logs won't get overriden
     * at engine restart time, the default value for this parameter is 'true'.
     */
    public final static String P_APPEND = "append";

    /**
     * Use 'authorField' to tell this worksession where the username/authorname
     * can be found in the workitem when logging (for audit purpose) the 
     * changes made to the workitem.
     */
    public final static String P_AUTHOR_FIELD = "authorField";

    /**
     * Use 'idField' to tell this worksession if there is a business-dependent
     * id for the workitem to display when logging (for audit purpose) the
     * changes made to the workitem.
     */
    public final static String P_ID_FIELD = "idField";

    /**
     * Use this parameter 'separator' to tell this service how the columns
     * should be separated in its log file.
     * The default value for the separator is "::".
     */
    public final static String P_SEPARATOR = "separator";

    private final static String DEFAULT_SEPARATOR = "::";


    private final static String C_REMOVED = "REM";
    private final static String C_MODIFIED = "MOD";
    private final static String C_NEW = "NEW";

    //
    // FIELDS

    private java.io.PrintWriter logWriter = null;

    private String authorFieldName = null;
    private String idFieldName = null;

    private String separator = null;

    //
    // CONSTRUCTORS

    public void init 
        (final String serviceName,
         final ApplicationContext context, 
         final java.util.Map serviceParams)
    throws 
        ServiceException
    {
        super.init(serviceName, context, serviceParams);

        final boolean shouldAppend = 
            MapUtils.getAsBoolean(serviceParams, P_APPEND, true);

        final String logFilePath = 
            MapUtils.getAsString(serviceParams, P_LOG_FILE, DEFAULT_LOG_FILE);

        try
        {
            this.logWriter = new java.io.PrintWriter
                (new java.io.FileWriter(logFilePath, shouldAppend));
        }
        catch (final java.io.IOException ie)
        {
            throw new ServiceException
                ("Couldn't open file '"+logFilePath+"' for writing audit logs");
        }

        this.authorFieldName = 
            MapUtils.getAsString(serviceParams, P_AUTHOR_FIELD);

        this.idFieldName = 
            MapUtils.getAsString(serviceParams, P_ID_FIELD);

        this.separator =
            MapUtils.getAsString(serviceParams, P_SEPARATOR, DEFAULT_SEPARATOR);
    }

    //
    // METHODS from AuditLogger

    /**
     * Logs when a field got removed from a workitem.
     */
    public void logFieldRemoval 
        (final Subject s,
         final InFlowWorkItem currentItem, 
         final String action,
         final String fieldName)
    {
        doLog
            (C_REMOVED,
             determineAuthorName(s, currentItem),
             action,
             determineItemId(currentItem),
             fieldName,
             null,
             null);
    }

    /**
     * Logs a modification to an already existing field.
     */
    public void logFieldModification
        (final Subject s,
         final InFlowWorkItem currentItem, 
         final String action,
         final String fieldName, 
         final Attribute original, 
         final Attribute modified)
    {
        doLog
            (C_MODIFIED,
             determineAuthorName(s, currentItem),
             action,
             determineItemId(currentItem),
             fieldName,
             ""+original,
             ""+modified);
    }

    /**
     * Logs the apparition of a new field in the workitem.
     */
    public void logNewField
        (final Subject s,
         final WorkItem currentItem, 
         final String action,
         final String fieldName, 
         final Attribute attribute)
    {
        doLog
            (C_NEW,
             determineAuthorName(s, currentItem),
             action,
             determineItemId(currentItem),
             fieldName,
             ""+attribute,
             null);
    }

    //
    // METHODS from Service

    /**
     * This override takes care of flushing and closing the logWriter used by
     * this class.
     */
    public void stop ()
        throws ServiceException
    {
        try
        {
            this.logWriter.flush();
            this.logWriter.close();
        }
        catch (final Throwable t)
        {
            // ignore
        }

        super.stop();
    }

    //
    // METHODS

    /**
     * This method extracts the author name from the workitem and if it doesn't
     * find it, it will take the subject username.
     */
    protected String determineAuthorName 
        (final Subject s, final WorkItem item)
    {
        String authorName = null;

        if (this.authorFieldName != null)
            authorName = item.getAttributes().sget(this.authorFieldName);

        if (authorName == null)
        {
            final BasicPrincipal bp = BasicPrincipal.getBasicPrincipal(s);

            if (bp != null) authorName = bp.getName();
        }

        if (authorName == null)
            authorName = "_unknown_";

        return authorName;
    }

    /**
     * This method returns the item id that should be used for the log.
     */
    protected String determineItemId (final WorkItem item)
    {
        StringBuffer id = new StringBuffer();

        if (this.idFieldName != null)
            id.append(item.getAttributes().sget(this.idFieldName));

        if (item instanceof InFlowWorkItem)
        {
            final InFlowWorkItem ifwi = (InFlowWorkItem)item;

            if (id.length() > 0) 
            {
                id
                    .insert(0, this.separator)
                    .insert(0, ifwi.getId().getWorkflowInstanceId());
            }
            else
            {
                id = new StringBuffer(ifwi.getId().toString());
            }
        }

        if (id.length() < 1)
            return "_new_";

        return id.toString();
    }

    /**
     * This method does the job of logging the audit records.
     */
    protected void doLog 
        (final String code, 
         final String author, 
         final String action,
         final String itemId, 
         final String fieldName,
         final String oldValue,
         final String newValue)
    {
        final StringBuffer sb = new StringBuffer()
            .append(new java.util.Date())
            .append(this.separator)
            .append(itemId)
            .append(this.separator)
            .append(action)
            .append(this.separator)
            .append(code)
            .append(this.separator)
            .append(author)
            .append(this.separator)
            .append(fieldName);

        if (oldValue != null)
        {
            sb
                .append(this.separator)
                .append("'")
                .append(oldValue)
                .append("'");
        }

        if (newValue != null)
        {
            sb
                .append(this.separator)
                .append("'")
                .append(newValue)
                .append("'");
        }

        this.logWriter.println(sb);
        this.logWriter.flush();
    }

    //
    // STATIC METHODS

}
