/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: AbstractWorkItemStore.java 3118 2006-08-30 14:38:36Z jmettraux $
 */

//
// AbstractWorkItemStore.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 20.11.2001 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.worklist.impl;

import javax.security.auth.Subject;

import openwfe.org.MapUtils;
import openwfe.org.Application;
import openwfe.org.AbstractService;
import openwfe.org.ServiceException;
import openwfe.org.ApplicationContext;
import openwfe.org.time.Time;
import openwfe.org.misc.Cache;
import openwfe.org.auth.BasicPrincipal;
import openwfe.org.engine.workitem.CancelItem;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.workitem.WorkItemCoder;
import openwfe.org.engine.workitem.WorkItemCoderLoader;
import openwfe.org.engine.dispatch.DispatchingException;
import openwfe.org.engine.expressions.FlowExpressionId;
import openwfe.org.engine.participants.Participant;
import openwfe.org.engine.participants.ParticipantMap;
import openwfe.org.worklist.Header;
import openwfe.org.worklist.CachedHeaders;
import openwfe.org.worklist.HeaderFactory;
import openwfe.org.worklist.WorkListException;
import openwfe.org.worklist.store.Lock;
import openwfe.org.worklist.store.WorkItemStore;
import openwfe.org.worklist.store.StoreException;
import openwfe.org.worklist.auth.StorePermission;


/**
 * This abstract workitem store serves as a foundation for the 
 * SimpleWorkitemStore implementation.
 * It mainly contains a cache for headers.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2006-08-30 16:38:36 +0200 (Wed, 30 Aug 2006) $
 * <br>$Id: AbstractWorkItemStore.java 3118 2006-08-30 14:38:36Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public abstract class AbstractWorkItemStore

    extends AbstractService

    implements WorkItemStore

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(AbstractWorkItemStore.class.getName());

    //
    // CONSTANTS (definitions)

    /**
     * The parameter 'cacheSize' defines how many workitems should be cached
     * in memory. The bigger the size the faster workitem retrieval may
     * be.
     */
    public final static String P_CACHE_SIZE 
        = "cacheSize";

    /**
     * Constant for the parameter named 'headerFactory' : a header is a 
     * summary of a workitem for display in a selection list, a header
     * factory is a piece of code that creates a header for a workitem. Use
     * this parameter to indicate the workitem store which header factory
     * (an OpenWFE like an other) it should use.
     */
    public final static String P_HEADER_FACTORY
        = "headerFactory";

    /**
     * Constant for the time parameter named 'lockTimeout' : how many
     * time an idle lock should be maintained on a workitem ?
     */
    public final static String P_LOCK_TIMEOUT
        = "lockTimeout";

    /**
     * Constant for the time parameter named 'unlockFrequency' : how 
     * frequently should the workitem store check for workitem whose
     * locks did time out ?
     */
    public final static String P_UNLOCK_FREQUENCY
        = "unlockFrequency";

    /**
     * Constant for the time parameter named 'maxStorageThreads' : after how
     * many concurrent storage thread should storage be performed sequentially
     * (lower response time) to avoid a JVM crash ?
     */
    public final static String P_MAX_STORAGE_THREADS
        = "maxStorageThreads";

    /**
     * The parameter 'workItemCoder' should point to the name of a workitem 
     * coder as defined by the 'workItemCoderLoader' service.
     */
    public final static String P_WORKITEM_CODER
        = "workItemCoder";

    /**
     * If the parameter 'default' for this store is set to 'true' or 'yes', 
     * the store will be a default store.
     */
    public final static String P_DEFAULT
        = "default";

    /** 
     * Set to 15m
     */
    public final static String DEFAULT_LOCK_TIMEOUT
        = "15m";

    /**
     * Set to 7m
     */
    public final static String DEFAULT_UNLOCK_FREQUENCY
        = "7m";

    /**
     * Set to 100 (workitems)
     */
    public final static int DEFAULT_CACHE_SIZE
        = 100;

    /**
     * Set to 100 (concurrent storage threads)
     */
    public final static int DEFAULT_MAX_STORAGE_THREADS
        = 100;

    //
    // FIELDS

    private Cache cache = null;

    private HeaderFactory headerFactoryService = null;

    private CachedHeaders cachedHeaders = null;
        // no headers determined for the moment

    private String[] acceptedParticipants = null;

    private java.util.Map lockMap = new java.util.HashMap();
    private long lockTimeout = 0;
    private UnlockDaemon unlockDaemon = null;

    private int maxStorageThreads = 30;
    private int activeStorageThreads = 0;

    private String workItemCoderName = null;

    //
    // CONSTRUCTORS

    public void init 
        (final String serviceName, 
         final ApplicationContext context, 
         final java.util.Map serviceParams)
    throws 
        ServiceException
    {
        super.init(serviceName, context, serviceParams);

        //
        // init cache

        final int cacheSize = MapUtils.getAsInt
            (getParams(), P_CACHE_SIZE, DEFAULT_CACHE_SIZE);

        this.cache = new Cache(cacheSize);

        log.info("init() cache size set to "+this.cache.size());

        //
        // determine max storage threads
        
        this.maxStorageThreads = MapUtils.getAsInt
            (getParams(), 
             P_MAX_STORAGE_THREADS, 
             DEFAULT_MAX_STORAGE_THREADS);

        log.info("init() max storage threads set to "+this.maxStorageThreads);

        //
        // determine workItemCoder
        
        this.workItemCoderName = MapUtils.getAsString
            (getParams(),
             P_WORKITEM_CODER);

        log.info("init() workItemCoder set to '"+this.workItemCoderName+"'");

        //
        // init accepted participants
        
        initAcceptedParticipants(serviceParams);

        //
        // init UnlockDaemon
        
        initUnlockDaemon();

        //
        // display OpenWFE version in logs
        
        log.info
            ("OpenWFE version : "+
             openwfe.org.engine.Definitions.OPENWFE_VERSION);
    }

    //
    // GETTERS and SETTERS

    //
    // METHODS

    /**
     * Loads the workitem coder
     */
    public WorkItemCoder getCoder ()
    {
        final WorkItemCoderLoader coderLoader = openwfe.org.engine.Definitions
            .getWorkItemCoderLoader(getContext());

        if (this.workItemCoderName != null)
            return coderLoader.getCoder(this.workItemCoderName);

        return coderLoader.getDefaultCoder();
    }

    //
    // METHODS from WorkItemStore

    /**
     * Returns true if the given workitem is locked.
     */
    public boolean isLocked (final FlowExpressionId workitemId)
    {
        return this.lockMap.keySet().contains(workitemId);
    }

    /**
     * Returns the lock set on the workitem, will return null if no lock is set
     * on the workitem.
     */
    public Lock getLock (final FlowExpressionId workitemId)
    {
        return (Lock)this.lockMap.get(workitemId);
    }

    public boolean isDefaultStore ()
    {
        return MapUtils.getAsBoolean(getParams(), P_DEFAULT, false);
    }

    /**
     * Adds the given workitem to this expression store.
     */
    public void store (final InFlowWorkItem wi)
        throws StoreException
    {
        if (log.isDebugEnabled())
        {
            log.debug
                ("store() in '"+getName()+"' for "+wi.getLastExpressionId());
        }

        //
        // cache workitem

        this.cache.put(wi.getLastExpressionId(), wi);

        //logDebugCache();

        //
        // uncache headers

        this.cachedHeaders = null;

        //
        // 'physical' storage

        if (this.activeStorageThreads >= this.maxStorageThreads)
            //
            // unthreaded storage
        {
            storeWorkItem(wi);
        }
        else
            //
            // threaded storage
        {
            this.activeStorageThreads++;

            if (log.isDebugEnabled())
            {
                log.debug
                    ("store() this.activeStorageThreads = "+
                     this.activeStorageThreads);
            }

            (new Thread()
             {
                 public void run ()
                 {
                     final FlowExpressionId fei = wi.getLastExpressionId();
                     this.setName
                        ("storage thread for "+fei.getWorkflowInstanceId()+
                         " exp "+fei.getExpressionId());

                     try
                     {
                         AbstractWorkItemStore.this.storeWorkItem(wi);
                     }
                     catch (final Throwable t)
                     {
                         //log.warn
                         //    ("store() failed to store workitem "+
                         //     wi.getLastExpressionId()+
                         //     " : "+t);
                         log.error
                             ("store() failed to store workitem "+
                              wi.getLastExpressionId(), 
                              t);
                     }

                     AbstractWorkItemStore.this.activeStorageThreads--;
                 }
             }).start();
         }

    }

    public void cancel (final CancelItem ci)
        throws StoreException
    {
        final FlowExpressionId id = ci.getId();

        this.cache.remove(id);
        this.cachedHeaders = null;

        removeWorkItem(id);
    }

    /**
     * Updates a workitem in the store
     */
    public void save (final Subject s, final InFlowWorkItem wi)
        throws StoreException
    {
        java.security.AccessController.checkPermission
            (StorePermission.newStorePermission(getName(), "write"));

        synchronized (this) // check impact on perf
        {
            final FlowExpressionId id = wi.getLastExpressionId();

            //
            // is the item locked ?
            
            Lock lock = (Lock)this.lockMap.get(id);
            if (lock != null)
            {
                Subject locker = lock.getLocker();

                //
                // is the subject the locker ?
                
                if ( ! s.equals(locker))
                {
                    throw new StoreException
                        ("WorkItem '"+id+
                         "' is already locked by someone else.");
                }
            }
            else
            {
                //log.debug("Saving a workitem that was not locked before...");

                throw new StoreException
                    ("Cannot save workitem : you have no lock on it.");
            }

            //
            // proceed

            wi.touch();
                // update wi.lastModified

            store(wi);

            //
            // notify : the headers for this participant
            // are not up to date anymore

            this.cachedHeaders = null;

            // leave lock untouched
        }
    }

    /**
     * Delegate works to this workitem store
     */
    public void delegate (InFlowWorkItem wi)
        throws StoreException
    {
        java.security.AccessController.checkPermission
            (StorePermission.newStorePermission(getName(), "delegate"));

        if (log.isDebugEnabled())
        {
            log.debug
                ("'"+getName()+"' Received delegated WI "+
                 wi.getLastExpressionId());
        }

        store(wi);
    }

    public void delegateToParticipant 
        (Subject s, InFlowWorkItem wi, String participantName)
    throws 
        StoreException
    {
        //
        // find participant
        
        ParticipantMap pMap = openwfe.org.engine.Definitions
            .getParticipantMap(getContext());

        Participant p = pMap.get(participantName);

        if (p == null)
        {
            throw new StoreException
                ("Cannot delegate to unknown participant '"+
                 participantName+"'");
        }

        //
        // remove from store

        this.remove(s, wi.getLastExpressionId());

        //
        // tag history

        wi.addHistoryItem
            ("subject::"+BasicPrincipal.getBasicPrincipal(s).getName(),
             "Delegated to participant::"+participantName);

        //
        // dispatch
        
        try
        {
            p.dispatch(getContext(), wi);
        }
        catch (DispatchingException de)
        {
            throw new StoreException
                ("Failed to delegate to participant '"+participantName+"'", de);
        }
    }

    public void remove (final Subject s, final FlowExpressionId expressionId)
        throws StoreException
    {
        //openwfe.org.Utils.debugMap(log, this.lockMap); // DEBUG

        java.security.AccessController.checkPermission
            (StorePermission.newStorePermission(getName(), "write"));

        synchronized (this) // check impact on perf
        {
            //
            // is the item locked ?
            
            Lock lock = (Lock)this.lockMap.get(expressionId);

            if (lock == null)
            {
                throw new StoreException
                    ("You cannot remove a workitem when there "+
                     "is no lock on it. Use 'getAndLock()'");
            }

            Subject locker = lock.getLocker();

            //
            // is the subject the locker ?
            
            if ( ! s.equals(locker))
            {
                throw new StoreException
                    ("WorkItem '"+expressionId+
                     "' is already locked by someone else.");
            }

            //
            // find item

            final InFlowWorkItem wi = get(s, expressionId);

            //
            // do it

            removeWorkItem(expressionId);

            //
            // remove lock

            this.lockMap.remove(expressionId);

            if (log.isDebugEnabled())
                log.debug(getName()+" remove() removed "+expressionId);

            //
            // notify : the headers for this participant
            // are not up to date anymore

            this.cachedHeaders = null;
        }
    }

    public InFlowWorkItem get 
        (final Subject s, 
         final FlowExpressionId expressionId)
    throws 
        StoreException
    {
        //log.debug(getName()+" get() for      "+expressionId);
        //logDebugCache();

        java.security.AccessController.checkPermission
            (StorePermission.newStorePermission(getName(), "read"));

        //
        // is workitem cached ?
        
        InFlowWorkItem wi = (InFlowWorkItem)this.cache.get(expressionId);

        if (wi != null)
        {
            //log.debug
            //    (getName()+" get() was cached : "+wi.getLastExpressionId());

            return wi;
        }

        //
        // no, load it

        wi = retrieveWorkItem(s, expressionId);

        //log.debug
        //    (getName()+" get() had to retrieve : "+wi.getLastExpressionId());

        //
        // cache it
        
        this.cache.put(expressionId, wi);

        //log.debug
        //    (getName()+" get() caching \n"+expressionId+
        //     "\n ---> \n"+wi.getLastExpressionId());

        //
        // return it

        return wi;
    }

    /**
     * fetches and lock a workitem from the store : ticket...
     */
    public InFlowWorkItem getAndLock 
        (final Subject s, 
         final FlowExpressionId expressionId)
    throws 
        StoreException
    {
        if (log.isDebugEnabled())
            log.debug(getName()+" getAndLock() for   "+expressionId);

        //openwfe.org.Utils.debugMap(log, this.lockMap); // DEBUG

        java.security.AccessController.checkPermission
            (StorePermission.newStorePermission(getName(), "write"));

        synchronized (this) // check impact on perf
        {
            //
            // get item
            
            final InFlowWorkItem wi = get(s, expressionId);

            //log.debug
            //    (getName()+" getAndLock() found "+wi.getLastExpressionId());

            // (caching is handled by getWorkItem)

            //
            // lock it
            
            this.lockMap.put(expressionId, new Lock(s, this.lockTimeout));

            //log.debug(getName()+" getAndLock() locked "+expressionId);

            //
            // 'uncache' headers

            this.cachedHeaders = null;

            //
            // return it
            
            return wi;
        }
    }

    public void release (final Subject s, final FlowExpressionId workItemId)
        throws StoreException
    {

        java.security.AccessController.checkPermission
            (StorePermission.newStorePermission(getName(), "write"));

        //FlowExpressionId id = wi.getLastExpressionId();

        synchronized (this)
        {
            //
            // find lock

            //Lock lock = (Lock)this.lockMap.get(id);
            final Lock lock = getLock(workItemId);

            if (lock == null)
            {
                throw new StoreException
                    ("WorkItem is not locked. No need to release it.");
            }

            Subject locker = lock.getLocker();

            if (locker == null) return; // success by default

            if ( ! s.equals(locker))
            {
                throw new StoreException
                    ("WorkItem is locked by someone else.");
            }

            //
            // remove lock

            this.lockMap.remove(workItemId);

            if (log.isDebugEnabled())
                log.debug(getName()+" release() released "+workItemId);

            //
            // notify : the headers for this participant
            // are not up to date anymore

            this.cachedHeaders = null;
        }
    }

    /**
     * Returns the workitem belonging to a certain workflow instance
     */
    public java.util.List findFlowInstance
        (final Subject s,
         final String workflowInstanceId)
    throws 
        StoreException
    {
        log.debug("findFlowInstance()");

        final int limit = 1500;
        final java.util.List result = new java.util.ArrayList(10);

        final java.util.Iterator it = listWorkItems(s, limit).iterator();
        while (it.hasNext())
        {
            final InFlowWorkItem item = (InFlowWorkItem)it.next();
            final FlowExpressionId fei = item.getLastExpressionId();

            if (fei.getWorkflowInstanceId().equals(workflowInstanceId))
                result.add(item.getLastExpressionId());
        }

        return result;
    }

    /**
     * Returns headers of workitem. Younger workitems first.
     */
    public java.util.List getHeaders 
        (final Subject s, 
         final int limit)
    throws 
        StoreException
    {
        return getHeaders(s, limit, null);
    }

    /**
     * Returns headers of workitem. 
     * You can specify the comparator first.
     */
    public java.util.List getHeaders 
        (final Subject s, 
         final int limit, 
         java.util.Comparator comparator)
    throws 
        StoreException
    {
        java.security.AccessController.checkPermission
            (StorePermission.newStorePermission(getName(), "browse"));

        if (comparator == null)
            comparator = LastModifiedHeaderComparator.YOUNGER_FIRST;

        if (shouldReloadHeaders(limit, comparator))
        {
            log.debug ("getHeaders() reloading headers");

            loadHeaders(s, limit, comparator);
        }

        return this.cachedHeaders.getHeaders();
    }

    /**
     * Implementing classes may override this method to have a
     * different headers reload rythm
     */
    protected boolean shouldReloadHeaders
        (int limit, java.util.Comparator comparator)
    {
        return
            (this.cachedHeaders == null ||
             //this.cachedHeaders.getLimit() < limit ||
             ! this.cachedHeaders.getComparator().equals(comparator));
    }

    public int countWorkItems (Subject s)
        throws StoreException
    {
        java.security.AccessController.checkPermission
            (StorePermission.newStorePermission(getName(), "browse"));

        return doCountWorkItems(s);
    }

    public boolean acceptsWorkItemsFor (String participantName)
    {
        if (log.isDebugEnabled())
        {
            log.debug
                ("acceptsWorkItemsFor() (store '"+getName()+
                 "') ? >"+participantName+"<");
        }

        for (int i=0; i<this.acceptedParticipants.length; i++)
        {
            if (log.isDebugEnabled())
            {
                log.debug
                    ("acceptsWorkItemsFor() does '"+participantName+
                     "' matches '"+this.acceptedParticipants[i]+"' ?");
            }

            if (participantName.matches(this.acceptedParticipants[i]))
            {
                log.debug("acceptsWorkItemsFor() true.");
                return true;
            }

            log.debug("acceptsWorkItemsFor() false");
        }

        log.debug("acceptsWorkItemsFor() no success : returning false.");

        return false;
    }

    public boolean checkPermission (final String action)
    {
        if (log.isDebugEnabled())
        {
            log.debug
                ("checkPermission() checking '"+getName()+
                 "' for permission '"+action+"'");
        }

        try
        {
            java.security.AccessController.checkPermission
                (StorePermission.newStorePermission(getName(), action));
        }
        catch (final java.security.AccessControlException ace)
        {
            log.debug("checkPermission() returning false : permission denied");
            //log.debug("checkPermission() returning false", ace);
            return false;
        }
        log.debug("checkPermission() returning true");
        return true;
    }

    //
    // METHODS from Service

    public void stop ()
        throws ServiceException
    {
        super.stop();

        this.unlockDaemon.cancel();
        log.info("UnlockDaemon stopped.");

        log.info("Service '"+getName()+"' stopped.");
    }

    //
    // ABSTRACT METHODS

    protected abstract void storeWorkItem (InFlowWorkItem wi)
        throws StoreException;

    protected abstract void removeWorkItem (FlowExpressionId fei)
        throws StoreException;

    protected abstract InFlowWorkItem retrieveWorkItem 
        (Subject s, FlowExpressionId fei)
    throws 
        StoreException;

    protected abstract int doCountWorkItems (Subject s)
        throws StoreException;

    protected abstract java.util.List listWorkItems (Subject s, int limit)
        throws StoreException;

    //
    // LOCAL METHODS

    //
    // Tue Oct 29 09:01:21 CET 2002
    // je suis dans un wagon restaurant 'Blaise Cendrars'. Est-ce
    // l'unique wagon-restaurant ddi  cet crivain ?
    //
    // "Wagon-restaurant. On ne distingue rien dehors. 
    // Il fait nuit noire."
    // Blaise Cendras (aka Frdric Sauser)
    //
    // Fait plaisir de croiser ce Suisse-l.
    //
    // Envie de relire le 'transsibrien'
    //

    protected HeaderFactory getHeaderFactory ()
    {
        if (this.headerFactoryService == null)
        {
            String serviceName = MapUtils.getAsString
                (getParams(), 
                 P_HEADER_FACTORY, 
                 openwfe.org.worklist.Definitions.S_HEADER_FACTORY);

            this.headerFactoryService = (HeaderFactory)getContext()
                .get(serviceName);
        }

        return this.headerFactoryService;
    }

    protected void initHeaderFactory (java.util.Map serviceParams)
        throws ServiceException
    {
        String headerFactoryClass = 
            (String)serviceParams.get(P_HEADER_FACTORY);
    }

    public boolean hasLockOn (Subject s, FlowExpressionId expressionId)
    {
        if (log.isDebugEnabled())
            log.debug(getName()+" hasLockOn() "+expressionId);

        final Lock lock = (Lock)this.lockMap.get(expressionId);

        if (log.isDebugEnabled())
            log.debug(getName()+" hasLockOn() found lock "+lock);

        if (lock == null) return false;

        //log.debug("hasLockOn() subject  >"+s+"<");
        //log.debug("hasLockOn() locker   >"+lock.getLocker()+"<");

        if ( ! s.equals(lock.getLocker())) return false;

        lock.touch(); 
            // ensure that the dispatching (that happens next) will be ok

        return true;
    }

    protected void initUnlockDaemon ()
    {
        this.lockTimeout = MapUtils.getAsTime
            (getParams(), P_LOCK_TIMEOUT, DEFAULT_LOCK_TIMEOUT);

        log.info("lockTimeout set to "+this.lockTimeout+" ms");

        final long unlockFreq = MapUtils.getAsTime
            (getParams(), P_UNLOCK_FREQUENCY, DEFAULT_UNLOCK_FREQUENCY);

        this.unlockDaemon = new UnlockDaemon();

        Application.getTimer().schedule
            (this.unlockDaemon, 60, unlockFreq);

        log.info("UnlockDaemon invoked. Will wake up every "+unlockFreq+" ms");
    }

    /**
     * The header loading bulk work.
     */
    protected void loadHeaders 
        (final Subject s, 
         final int limit, 
         final java.util.Comparator headerComparator)
    throws 
        StoreException
    {
        //final long startTime = System.currentTimeMillis();

        int capacity = limit;
        if (capacity < 0) capacity = 1000;

        java.util.List result = new java.util.ArrayList(capacity);

        java.util.List items = listWorkItems(s, limit);

        if (log.isDebugEnabled())
            log.debug("loadHeaders() found "+items.size()+" workitems");

        java.util.Iterator it = items.iterator();
        while (it.hasNext())
        {
            final InFlowWorkItem wi = (InFlowWorkItem)it.next();

            //log.debug("loadHeaders() Building header...");

            try
            {
                Header header = getHeaderFactory()
                    //.buildHeader(wi, isLocked(wi));
                    .buildHeader(wi, getLockerName(wi));

                result.add(header);
            }
            catch (final WorkListException wle)
            {
                throw new StoreException
                    ("Failed to loadHeaders from store '"+getName()+"'", wle);
            }
        }
        //log.debug("loadHeaders() Built headers");

        ((java.util.ArrayList)result).trimToSize(); 
            // less bandwidth required

        java.util.Collections.sort(result, headerComparator);

        if (log.isDebugEnabled())
            log.debug("loadHeaders() loaded "+result.size()+" headers.");

        this.cachedHeaders = 
            new CachedHeaders(limit, headerComparator, result);

        //log.debug
        //    ("loadHeaders() took "+
        //     (System.currentTimeMillis()-startTime)+" ms");
    }

    /**
     * Returns true if the given workitem is locked.
     */
    public boolean isLocked (final InFlowWorkItem wi)
    {
        /*
        synchronized (this.lockMap)
        {
            final FlowExpressionId id = wi.getLastExpressionId();
            return this.lockMap.keySet().contains(id);
        }
        */
        return isLocked(wi.getLastExpressionId());
    }

    /**
     * Returns the name of the principal that locked this workitem (or
     * null if it's not locked).
     */
    protected String getLockerName (final InFlowWorkItem wi)
    {
        final Lock lock = getLock(wi.getLastExpressionId());

        if (lock == null) return null;

        return BasicPrincipal.getBasicPrincipal(lock.getLocker()).getName();
    }

    /**
     * Establishes a list of participant names that this workitem
     * stores is going to accept when the worklist will iterator
     * among its stores.
     */
    protected void initAcceptedParticipants 
        (final java.util.Map serviceParams)
    {
        String rawString = (String)serviceParams.get(P_PARTICIPANTS);
        
        if (rawString == null) 
        {
            this.acceptedParticipants = new String[] {};

            log.warn
                ("initAcceptedParticipants() "+
                 "no participants specified for the store '"+
                 this.getName()+"'");
        }
        else
        {
            this.acceptedParticipants = rawString.split(", *");

            log.info("initAcceptedParticipants() '"+rawString+"'");
        }
    }

    //
    // INNER CLASSES

    class UnlockDaemon extends java.util.TimerTask
    {
        private final org.apache.log4j.Logger log = 
            org.apache.log4j.Logger
                .getLogger(UnlockDaemon.class.getName());

        public void run ()
        {
            log.debug("waking up for "+AbstractWorkItemStore.this.getName());

            synchronized (AbstractWorkItemStore.this.lockMap)
            {
                java.util.List toRemove = new java.util.ArrayList();

                java.util.Iterator it = AbstractWorkItemStore.this.lockMap
                    .keySet().iterator();
                while (it.hasNext())
                {
                    FlowExpressionId itemId = (FlowExpressionId)it.next();

                    Lock lock = (Lock)AbstractWorkItemStore.this.lockMap
                        .get(itemId);

                    if (lock.shouldBeUnlocked())
                    {
                        toRemove.add(itemId);
                    }
                }

                it = toRemove.iterator();
                while (it.hasNext())
                {
                    FlowExpressionId itemId = (FlowExpressionId)it.next();
                    AbstractWorkItemStore.this.lockMap.remove(itemId);
                    AbstractWorkItemStore.this.cache.remove(itemId);
                    AbstractWorkItemStore.this.cachedHeaders = null;

                    log.debug
                        ("In "+AbstractWorkItemStore.this.getName()+
                         ", unlocked and uncached "+itemId);
                }
            }

            log.debug("going to sleep.");
        }
    }

    //
    // MISC

    /*
    private void logDebugCache ()
    {
        final StringBuffer sb = new StringBuffer();

        sb.append("\n*** in cache :\n");

        final java.util.Iterator it = this.cache.keySet().iterator();
        while (it.hasNext())
        {
            final FlowExpressionId key = (FlowExpressionId)it.next();
            final InFlowWorkItem wi = (InFlowWorkItem)this.cache.get(key);

            sb.append(key.toString());
            sb.append("\n   ---> ");
            sb.append(wi.getLastExpressionId().toString());
            sb.append("\n");
        }
        sb.append("***\n");

        log.debug(sb.toString());
    }
    */

}
