/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: Launchable.java 3118 2006-08-30 14:38:36Z jmettraux $
 */

//
// Launchable.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 31.10.2002 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.worklist;

import openwfe.org.ApplicationContext;
import openwfe.org.time.Time;
import openwfe.org.engine.workitem.LaunchItem;
import openwfe.org.engine.workitem.WorkItemCoder;
import openwfe.org.engine.workitem.WorkItemCoderLoader;
import openwfe.org.engine.workitem.StringMapAttribute;
import openwfe.org.engine.impl.launch.SimpleXmlLauncher;


/**
 * Something you can launch, either a flow, either an item
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2006-08-30 16:38:36 +0200 (Wed, 30 Aug 2006) $
 * <br>$Id: Launchable.java 3118 2006-08-30 14:38:36Z jmettraux $ </font>
 *
 * @author jmettraux@openwfe.org
 */
public class Launchable

    implements java.io.Serializable

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(Launchable.class.getName());

    //
    // CONSTANTS & co 

    private final static String ROOT_ELT_NAME = "root.element.name";
        // see applic/openwfe.org.Utils.extractXmlDescription(docUrl);

    private final static String LAUNCHITEM = "launchitem";

    //
    // FIELDS

    private transient ApplicationContext applicationContext = null;
    private transient java.util.Map serviceParams = null;

    private String engineId = null;
    private String url = null;
    private java.util.Map descriptionMap = null;
    private boolean isItem = false;

    private LaunchItem launchitem = null;

    //
    // CONSTRUCTORS

    /**
     * Instantiates a Launchable.
     *
     * @param context As a launchable is supposed to be built inside a woklist, 
     * this points to the application context holding the worklist
     * @param serviceParams 
     * @param engineId The name (as found in the participant map) of the engine
     * supposed to run the flow to be launched.
     * @param url The URL of the workflow definition.
     */
    public Launchable 
        (final ApplicationContext context, 
         final java.util.Map serviceParams,
         final String engineId, 
         final String url)
    throws 
        LaunchException
    {
        this.applicationContext = context;
        this.serviceParams = serviceParams;

        this.engineId = engineId;
        this.url = url;
        this.descriptionMap = SimpleXmlLauncher.extractXmlDescription(url);
        
        String rootElementName = 
            (String)this.descriptionMap.get(ROOT_ELT_NAME);
        this.isItem = 
            (rootElementName != null && rootElementName.equals(LAUNCHITEM));

        this.launchitem = buildLaunchitem();
    }

    //
    // METHODS

    /**
     * Returns the participant name (engine id) of the engine supposed to
     * host the launched flow.
     */
    public String getEngineId () { return this.engineId; }

    /**
     * Returns the URL of this launchable.
     */
    public String getUrl () { return this.url; }

    /**
     * If false is returned, then the launchable points
     * to a flow directly
     */
    public boolean isLaunchItem () { return this.isItem; }

    /**
     * Returns the default description (english?) of this launchable.
     */
    public String getDefaultDescription ()
    {
        return (String)this.descriptionMap.get("default");
    }

    /**
     * Returns the description for a given language.
     * If the language has no description, null will be returned.
     */
    public String getDescription (final String language)
    {
        if (language == null) return getDefaultDescription();

        return 
            (String)this.descriptionMap.get(language);
    }

    /**
     * Returns the launchitem (if any) wrapped in this launchable.
     */
    public LaunchItem getLaunchItem ()
    {
        return this.launchitem;
    }

    /**
     * Returns a string representation of this launchable.
     */
    public String toString ()
    {
        StringBuffer sb = new StringBuffer();

        sb.append("<launchable\n");
        sb.append("  engineId=\""); sb.append(this.engineId); sb.append("\"\n");
        sb.append("  url=\""); sb.append(this.url); sb.append("\"\n");
        sb.append("  isLaunchItem=\""); sb.append(this.isItem); sb.append("\"\n");
        sb.append("> \n");

        java.util.Iterator it = this.descriptionMap.keySet().iterator();
        while (it.hasNext())
        {
            String language = (String)it.next();

            sb.append("  <description \n");
            sb.append("    language=\""); sb.append(language); sb.append("\" \n");
            sb.append("  >");
            sb.append((String)this.descriptionMap.get(language));
            sb.append("</description>\n");
        }

        sb.append("</launchable>");

        return sb.toString();
    }

    /**
     * Builds a launchitem for this Launchable instance.
     */
    protected LaunchItem buildLaunchitem ()
        throws LaunchException
    {
        if (this.isItem)
        {
            final WorkItemCoderLoader coderLoader = 
                openwfe.org.engine.Definitions
                    .getWorkItemCoderLoader(this.applicationContext);

            if (coderLoader == null)
            {
                throw new LaunchException
                    ("A CoderLoader service named '"+
                     "workItemCoderLoader' or "+
                     "'EngineContext.workItemCoderLoader"+
                     "' is required for the proper operation of the class '"+
                     this.getClass().getName()+"'");
            }

            final WorkItemCoder coder = coderLoader.getXmlCoder();

            try
            {
                LaunchItem li = (LaunchItem)coder.decode
                    (new java.net.URL(this.url), 
                     this.applicationContext, 
                     this.serviceParams);

                li.setLastModified(Time.toIsoDate());

                return li;
            }
            catch (Exception e)
            {
                log.debug
                    ("Failed to build launchitem at '"+this.url+"'", e);
                throw new LaunchException
                    ("Failed to build launchitem at '"+this.url+"'", e);
            }
        }

        //
        // just a regular flow

        LaunchItem li = new LaunchItem();

        li.setWorkflowDefinitionUrl(this.url);
        li.setAttributes(new StringMapAttribute());
        li.setLastModified(Time.toIsoDate());

        return li;
    }

}
