/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: WfCmdEdit.java 3118 2006-08-30 14:38:36Z jmettraux $
 */

package openwfe.org.wlshell;

import java.io.BufferedReader;
import java.io.InputStreamReader;

import openwfe.org.shell.CmdHandler;
import openwfe.org.engine.workitem.AtomicAttribute;
import openwfe.org.engine.workitem.AttributeException;
import openwfe.org.engine.workitem.BooleanAttribute;
import openwfe.org.engine.workitem.DoubleAttribute;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.workitem.IntegerAttribute;
import openwfe.org.engine.workitem.LongAttribute;
import openwfe.org.engine.workitem.StringAttribute;
import openwfe.org.engine.workitem.StringMapAttribute;
import openwfe.org.engine.expressions.FlowExpressionId;
import openwfe.org.worklist.Header;
import openwfe.org.worklist.WorkSession;

/**
 * Provide a command line editor for an OpenWFE work item.
 *  
 * @author Peter Mayne
 *
 */
public class WfCmdEdit
{
    private WorkSession session;
    private String storeName;
    private Header header;
    private FlowExpressionId fei;
    private BufferedReader reader;

    private InFlowWorkItem ifItem;
    boolean modified;
        
    private String prompt;
    
    private static final int ADD_ATTRIBUTE = 0;
    private static final int SET_ATTRIBUTE = 1;
    private static final int DEL_ATTRIBUTE = 2;
    
    // When setting a workflow item, values of these types can be set.
    // Only fundamental types can be modified from the command line.
    //
    final static String[] types = {"boolean", "double", "integer", "long", "string"};

    public WfCmdEdit(WorkSession session, String storeName, Header header, BufferedReader reader)
    {
        this.session = session;
        this.storeName = storeName;
        this.header = header;
        this.reader = reader;

        fei = header.getExpressionId();
        try
        {
            ifItem = session.getAndLock(storeName, fei);
        }
        catch (Exception e)
        {
            throw new IllegalArgumentException("Can't get FlowExpresssionId " + fei.getWorkflowInstanceId() + ": " + e);
        }

        StringMapAttribute sma = header.getAttributes();
        prompt = sma.get(Header.WF_DEFINITION_NAME) + " " + sma.get(Header.WF_DEFINITION_REVISION) + ": " + sma.get(Header.WF_INSTANCE_ID) + "> ";
//        prompt = "> ";

        modified = false;
    }
    
    public void help_add()
    {
        System.out.println("add [<attribute type>] <field name> <field value>");
        System.out.println("  Adds the specified attribute with the specified value. Use double quotes for values with spaces.");
        System.out.println("  By default, the type of the attribute is string.");
        System.out.print("  Possible atomic attribute types are:");
        for(int i=0; i<types.length; i++)
        {
            System.out.print(" " + types[i]);
        }
        System.out.println();
    }
    
    public Boolean do_add(String[] args)
    {
        return modifyAttribute(args, ADD_ATTRIBUTE);
    }
    
    public void help_delegate()
    {
        System.out.println("delegate <store|participant> <name>");
        System.out.println("  Delegate this workflow item to another store or participant");
    }
    
    public Boolean do_delegate(String[] args)
    {
        if(args.length!=2)
        {
            System.out.println("Wrong number of arguments.");
            return CmdHandler.DO_NOT_EXIT;
        }
        String delegateTo = args[0].toLowerCase();
        if(!"store".startsWith(delegateTo) && !"participant".startsWith(delegateTo))
        {
            System.out.println("The second argument must specify \"store\" or \"particpant\".");
            return CmdHandler.DO_NOT_EXIT;
        }
        String delegateName = args[1];
        
        // If there are problems saving/forwarding, don't return.
        //
        Boolean doReturn = CmdHandler.DO_EXIT;
        try
        {
            if(modified)
            {
                session.save(storeName, ifItem);
                modified = false;
            }    
            if("store".startsWith(delegateTo))
            {
                session.delegate(storeName, ifItem, delegateName);
            }
            else
            {
                session.delegateToParticipant(storeName, ifItem, delegateName);
            }    
        }
        catch(Exception e)
        {
            System.err.println("Can't delegate workflow item: " + e);
            doReturn = CmdHandler.DO_NOT_EXIT;
        }

        return doReturn;
    }
    
    public void help_display()
    {
        System.out.println("display");
        System.out.println("  Display the current workflow item");
    }
    
    public Boolean do_display(String[] args)
    {
        if(args.length!=0)
        {
            System.out.println("Wrong number of arguments.");
            return CmdHandler.DO_NOT_EXIT;
        }
        
        WfCmd.displayWorkItem(header, ifItem);
        
        return CmdHandler.DO_NOT_EXIT;
    }

    public void help_proceed()
    {
        System.out.println("proceed");
        System.out.println("  Save any modifications and proceed to the next workflow step.");
    }
    
    public Boolean do_proceed(String[] args)
    {
        if(args.length!=0)
        {
            System.out.println("Wrong number of arguments.");
            return CmdHandler.DO_NOT_EXIT;
        }
        
        // If there are problems saving/forwarding, don't return.
        //
        Boolean doReturn = CmdHandler.DO_EXIT;
        try
        {
            if(modified)
            {
                session.save(storeName, ifItem);
                modified = false;
            }
            session.forward(storeName, ifItem);
        }
        catch(Exception e)
        {
            System.err.println("Can't forward workflow item: " + e);
            doReturn = CmdHandler.DO_NOT_EXIT;
        }

        return doReturn;
    }
    
    public void help_release()
    {
        System.out.println("release");
        System.out.println("Finish editing this workflow item. Do not save any unsaved changes.");
    }
    
    public Boolean do_release(String[] args)
    {
        if(args.length!=0)
        {
            System.out.println("Wrong number of arguments.");
            return CmdHandler.DO_NOT_EXIT;
        }
        
        try
        {
            session.release(storeName, ifItem);
        }
        catch (Exception e)
        {
            System.err.println("Can't release FlowExpressionId " + fei.getWorkflowInstanceId() + ": " + e);
        }

        return CmdHandler.DO_EXIT;
    }
    
    public void help_save()
    {
        System.out.println("save");
        System.out.println("  Save the current workflow item");
    }
    
    public Boolean do_save(String[] args)
    {
        if(args.length!=0)
        {
            System.out.println("Wrong number of arguments.");
            return CmdHandler.DO_NOT_EXIT;
        }
        
        try
        {
            session.save(storeName, ifItem);
        }
        catch(Exception e)
        {
            System.err.println("Cannot save workitem: " + e);
        }
        
        return CmdHandler.DO_NOT_EXIT;
    }
    
    public void help_set()
    {
        System.out.println("set [<atomic attribute type>] <field name> <field value>");
        System.out.println("  Sets the specified attribute to the specified value. Use double quotes for values with spaces.");
        System.out.println("  By default, the type of the attribute is string.");
        System.out.print("  Allowed atomic attribute types are:");
        for(int i=0; i<types.length; i++)
        {
            System.out.print(" " + types[i]);
        }
        System.out.println();
    }
    
    public Boolean do_set(String[] args)
    {
        return modifyAttribute(args, SET_ATTRIBUTE);
    }
    
    public void help_unset()
    {
        System.out.println("unset <field name>");
        System.out.println("  Removes the specified field from this work item.");
    }

    public Boolean do_unset(String args[])
    {
        if(args.length!=1)
        {
            System.out.println("Wrong number of arguments");
            return CmdHandler.DO_NOT_EXIT;
        }
        
        return modifyAttribute(args, DEL_ATTRIBUTE);
    }

    /**
     * Modify an attribute: add, set, delete.
     * 
     * @param args
     * @param modification What kind of modification?
     * 
     * @return
     */    
    private Boolean modifyAttribute(String[] args, int modification)
    {
        String fieldType = null, fieldName = null, fieldValue = null;
        AtomicAttribute attr = null;
        
        if(modification==DEL_ATTRIBUTE)
        {
            if(args.length==1)
            {
                fieldName = args[0];
            }
            else
            {
                System.out.println("Wrong number of arguments.");
                return CmdHandler.DO_NOT_EXIT;
            }    
        }
        else
        {
            if(args.length==2)
            {
                fieldType = "string";
                fieldName = args[0];
                fieldValue = args[1];
            }
            else if(args.length==3)
            {
                fieldType = args[0].toLowerCase();
                fieldName = args[1];
                fieldValue = args[2];
            }
            else
            {
                System.out.println("Wrong number of arguments.");
                return CmdHandler.DO_NOT_EXIT;
            }

            int type = -1;
            for(int i=0; i<types.length; i++)
            {
                if(types[i].startsWith(fieldType.toLowerCase()))
                {
                    type = i;
                    break;
                }
            }
            if(type==-1)
            {
                System.out.println("Unknown type \"" + fieldType + "\" specified.");
                return CmdHandler.DO_NOT_EXIT;
            }

            if(types[type].equals("boolean"))
            {
                attr = new BooleanAttribute("true".startsWith(fieldValue.toLowerCase()));
            }
            else if(types[type].equals("double"))
            {
                attr = new DoubleAttribute(Double.parseDouble(fieldValue));
            }
            else if(types[type].equals("integer"))
            {
                attr = new IntegerAttribute(Integer.parseInt(fieldValue));
            }
            else if(types[type].equals("long"))
            {
                attr = new LongAttribute(Long.parseLong(fieldValue));
            }
            else
            {
                attr = new StringAttribute(fieldValue);
            }
        }
        
        switch (modification)
        {
            case ADD_ATTRIBUTE:
                try
                {
                    ifItem.addAttribute(fieldName, attr);
                    modified = true;
                }
                catch (AttributeException ae)
                {
                    System.err.println("\nCan't add attribute: " + ae);
                }                        
                break;

            case SET_ATTRIBUTE:
                try
                {
                    ifItem.setAttribute(fieldName, attr);
                    modified = true;
                }
                catch (AttributeException ae)
                {
                    System.err.println("\nCan't set attribute: " + ae);
                }
                break;
            
            case DEL_ATTRIBUTE:
                ifItem.removeAttribute(fieldName);
                modified = true;
                break;
        }
        
        return CmdHandler.DO_NOT_EXIT;            
    }
    
    public void run()
    {
        final CmdHandler parser = new CmdHandler(this);
        
        parser.commandLoop(prompt, reader);

    }

    public static void main(String[] args)
    {
        BufferedReader reader = new BufferedReader(new InputStreamReader(System.in));
        new WfCmdEdit(null, null, null, reader).run();
    }
}
