/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: WfCmd.java 3118 2006-08-30 14:38:36Z jmettraux $
 */

package openwfe.org.wlshell;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.rmi.Naming;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import javax.security.auth.Subject;

import openwfe.org.shell.CmdHandler;
import openwfe.org.rmi.session.WorkSessionServer;
import openwfe.org.engine.workitem.Attribute;
import openwfe.org.engine.workitem.HistoryItem;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.workitem.LaunchItem;
import openwfe.org.engine.workitem.LongAttribute;
import openwfe.org.engine.workitem.StringAttribute;
import openwfe.org.engine.workitem.StringMapAttribute;
import openwfe.org.engine.expressions.FlowExpressionId;
import openwfe.org.worklist.Header;
import openwfe.org.worklist.Launchable;
import openwfe.org.worklist.WorkSession;

/**
 * An RMI shell for tinkering with a worklist
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2006-08-30 16:38:36 +0200 (Wed, 30 Aug 2006) $
 * <br>$Id: WfCmd.java 3118 2006-08-30 14:38:36Z jmettraux $ </font>
 *
 * @author Peter Mayne
 * @author jmettraux@openwfe.org
 */
public class WfCmd
{
    public final static String BANNER = "\n" + WfCmd.class.getName() + "\nBy Peter Mayne\n$Id: WfCmd.java 3118 2006-08-30 14:38:36Z jmettraux $\n";

    // The default maximum number of workitems that will be displayed.
    //    
    final static int DEFAULT_LIMIT = 1000;

    private String url;
    private WorkSession session;
    private String prompt;
    private String language = null;
    private BufferedReader reader;

    public WfCmd(String url, WorkSession session, BufferedReader reader)
    {
        this.url = url;
        this.session = session;
        prompt = url + "> ";
        this.reader = reader;
    }
    
    public void help_count()
    {
        System.out.println("count [<storeName>]");
        System.out.println("  Returns the count of workitems in the store. If you don't provide a");
        System.out.println("  storeName, it will be set to 'default'.");
    }

    /**
     * Display the number of workflow items in a store.
     * 
     * @param args
     * @return
     * @throws Exception
     */
    public Boolean do_count(String[] args) throws Exception
    {
        String storeName = "default";

        if (args.length > 0)
        {    
            storeName = args[0];
        }

        System.out.print("WorkItems in store '" + storeName + "': ");
        int count = session.countWorkItems(storeName);
        System.out.println(count);
        
        return CmdHandler.DO_NOT_EXIT;
    }

    public void help_display()
    {
        System.out.println("display <store> <workflow item id>");
        System.out.println("  Display the specified workflow item.");
    }

    /**
     * Show a workflow item.
     * 
     * @param args
     * @return
     * @throws Exception
     */        
    public Boolean do_display (final String[] args) 
        throws Exception
    {
        if (args.length!=2)
        {
            System.out.println("Required parameters not specified");
            return CmdHandler.DO_NOT_EXIT;
        }
        
        String storeName = args[0];
        String id = args [1];

        Header header = getHeader(session, storeName, id);
        
        if (header!=null)
        {
            FlowExpressionId fei = header.getExpressionId();
            InFlowWorkItem wi = session.get(storeName, fei);
            displayWorkItem(header, wi);
            
        }
        else
        {
            System.out.println("No such work item.");
        }
        
        return CmdHandler.DO_NOT_EXIT;            
    }
    
    public void help_edit()
    {
        System.out.println("edit <store> <workflow item id>");
        System.out.println("  Edit the specified workflow item.");
    }
    
    /**
     * Edit a workflow item.
     * 
     * @param args
     * @return
     * @throws Exception
     */
    public Boolean do_edit (String[] args) throws Exception
    {
        if(args.length!=2)
        {
            System.out.println("Required parameters not specified");
            return CmdHandler.DO_NOT_EXIT;
        }
        
        String storeName = args[0];
        String id = args[1];

        Header header = getHeader(session, storeName, id);
        
        // Start a new workitem editor.
        //
        new WfCmdEdit(session, storeName, header, reader).run();

        return CmdHandler.DO_NOT_EXIT;
    }
    
    public void help_headers ()
    {
        System.out.println("headers [<storeName> [<limit>]]");
        System.out.println("  Returns headers of the workitem in a given store (defaults to");
        System.out.println("  store 'default'). The 'limit' parameters defaults to " + DEFAULT_LIMIT + " and allows you");
        System.out.println("  to specify how may headers you want to receive at max.");
    }

    public Boolean do_headers (String[] args) throws Exception
    {
        String storeName = "default";
        int limit = DEFAULT_LIMIT;

        String sLimit = null;

        if (args.length > 0)
        {
            storeName = args[0];
        }
        if (args.length > 1)
        {
            sLimit = args[1];
        }

        if (sLimit != null)
        {
            try
            {
                limit = Integer.parseInt(sLimit);
            }
            catch (NumberFormatException nfe)
            {
                System.out.println("Cannot read '" + sLimit + "': setting limit to " + limit);
            }
        }

        List headers = session.getHeaders(storeName, limit);

        // Display the headers.
        //

        System.out.println();

        final Iterator it = headers.iterator();
        while (it.hasNext())
        {
            final Header header = (Header)it.next();
            final StringMapAttribute sma = header.getAttributes();
            
            System.out.println("Workflow " + sma.get(Header.WF_DEFINITION_NAME) + " " + sma.get(Header.WF_DEFINITION_REVISION) + " : " + sma.get(Header.WF_INSTANCE_ID));
            
            StringAttribute subject = (StringAttribute)sma.get(Header.SUBJECT);
            System.out.println("\"" + (subject!=null ? subject.toString() : "") + "\"");
            System.out.println("Dispatch time : " + sma.get(Header.DISPATCH_TIME));
            System.out.println("Last modified : " + header.getLastModified());
            System.out.println("Participant   : " + sma.get(Header.PARTICIPANT_NAME));
            System.out.println(".");

            System.out.println("headers 'en vrac' :");
            
            final java.util.Iterator jt = sma.stringKeySet().iterator();
            while (jt.hasNext())
            {
                final String key = (String)jt.next();
                System.out.println(" * '"+key+"': '"+sma.get(key)+"'");
            }

            if (header.isLocked()) System.out.println("Locked");
            
            System.out.println();
        }
        
        return CmdHandler.DO_NOT_EXIT;
    }
    
    public void help_launch()
    {
        System.out.println("launch <url>"); // | flow <name> <version>");
        System.out.println("  Launch the specified workflow.");
//        System.out.println("  Specify the URL of the workflow."); //, or its name and version.");
    }
    
    public Boolean do_launch(String[] args)
    {
        if(args.length!=1)
        {
            System.out.println("Incorrect parameters specified");
            return CmdHandler.DO_NOT_EXIT;
        }

        String url = args[0];
        
        try
        {
            Launchable launch = null;
            LaunchItem li = null;
            for(Iterator it = session.getLaunchables().iterator(); it.hasNext();)
            {
                launch = (Launchable)it.next();
                if(launch.getUrl().equalsIgnoreCase(url))
                {
                    li = launch.getLaunchItem();
                    break;
                }
            }

            if(li!=null)
            {
                // Add the subject.
                //
                System.out.print("Subject: ");
                String line = null;
                try
                {
                    line = reader.readLine().trim();
                }
                catch(Exception e)
                {
                }
                if(line !=null && line.length()>0)
                {
                    li.addAttribute(Header.SUBJECT, new StringAttribute(line));
                }
                
                // Optionally add some StringAttributes.
                //
                do
                {
                    System.out.print("Attribute (leave empty to finish): ");
                    String attr = reader.readLine().trim();
                    if(attr.length()==0)
                    {
                        break;
                    }
                    
                    System.out.print("Value: ");
                    String value = reader.readLine().trim();
                    if(li.getAttribute(attr)==null)
                    {
                        li.addAttribute(attr, new StringAttribute(value));
                    }
                    else
                    {
                        li.setAttribute(attr, new StringAttribute(value));
                    }    
                } while (true);
                
                session.launch(launch.getEngineId(), li);
                return CmdHandler.DO_NOT_EXIT;
            }    
        }
        catch(Exception e)
        {
            System.err.println("Can't launch workflow: " + e);
        }
        
        return CmdHandler.DO_NOT_EXIT;
    }

    public Boolean do_exit (final String[] args)
    {
        return CmdHandler.DO_EXIT;
    }

    public Boolean do_quit (final String[] args)
    {
        return CmdHandler.DO_EXIT;
    }
    
    public void help_stores()
    {
        System.out.println("stores");
        System.out.println("  Lists all of the stores in the worklist you are connected to. (It doesn't");
        System.out.println("  tell you whether or not you have access to each of these stores.)");
    }

    /**
     * List stores.
     * 
     * @param args
     * @return
     * @throws Exception
     */
    public Boolean do_stores(String[] args) throws Exception
    {
        for(Iterator it = session.getStoreNames().iterator(); it.hasNext();)
        {
            String name = (String)it.next();
            System.out.println("   - '" + name + "'");
        }
        
        return CmdHandler.DO_NOT_EXIT;
    }

    public void help_user()
    {
        System.out.println("user [<username> <password>]");
        System.out.println("  If not parameters are specified, display the current user.");
        System.out.println("  Otherwise, start a new work session as this user and logout from the current session.");
        System.out.println("  If unsuccessful, the current session will continue to be used.");
    }
    
    public Boolean do_user(String[] args)
    {
        if(args.length==0)
        {
            try
            {
                Subject subject = session.getSubject();
                System.out.println(subject);
            }
            catch(Exception e)
            {
                System.err.println("Can't get Subject: " + e);
            }
            
            return CmdHandler.DO_NOT_EXIT;
        }
        
        if(args.length!=2)
        {
            System.out.println("Required parameters not specified");
            return CmdHandler.DO_NOT_EXIT;
        }
        
        String username = args[0];
        String password = args[1];
        
        try
        {
            WorkSessionServer sessionServer = 
                (WorkSessionServer)Naming.lookup(url);

            WorkSession session = 
                (WorkSession)sessionServer.login(username, password);
            
            // Switch sessions.
            //
            //this.session.release();
            this.session.close();
            this.session = session;
        }
        catch(Exception e)
        {
            System.out.println("Login not successful: " + e);
        }
        
        return CmdHandler.DO_NOT_EXIT;
    }
    
    public void help_workflows()
    {
        System.out.println("workflows");
        System.out.println("  Lists all of the workflows that can be launched.");
    }

    /**
     * List launchable workflows.
     * 
     * @param args
     * @return
     * @throws Exception
     */
    public Boolean do_workflows(String[] args) throws Exception
    {
        for(Iterator it = session.getLaunchables().iterator(); it.hasNext();)
        {
            Launchable launch = (Launchable)it.next();
            String description = language==null ? launch.getDefaultDescription() : launch.getDescription(language);
            System.out.println("Workflow url " + launch.getUrl());
            System.out.println("  \"" + description + "\"");
            System.out.println("  Root element name: " + launch.getDescription("root.element.name"));
            System.out.println("  Engine Id: " + launch.getEngineId());
            System.out.println("  isLaunchItem: " + launch.isLaunchItem());

            System.out.println();
        }
        
        return CmdHandler.DO_NOT_EXIT;
    }

    public static Header getHeader
        (final WorkSession session, final String storeName, final String id) 
    throws 
        Exception
    {
        List headers = session.getHeaders(storeName, DEFAULT_LIMIT);
        Header header = null;
        FlowExpressionId fei = null;
        for (Iterator i = headers.iterator(); i.hasNext();)
        {
            header = (Header)i.next();
            if (header.getExpressionId().getWorkflowInstanceId().equals(id))
            {
                return header;
            }
        }
        
        return null;
    }

    public static void displayAttributes(StringMapAttribute sma)
    {
        for(Iterator i = sma.stringKeySet().iterator(); i.hasNext();)
        {
            String key = (String)i.next();
            Attribute attr = (Attribute)sma.get(key);
            String attrType = attr.getClass().getName();
            attrType = attrType.substring(attrType.lastIndexOf(".")+1);
            System.out.print("\n" + key + " [" + attr + "] (" + attrType + ")");
        }
        
    }

    public static void displayWorkItem(Header header, InFlowWorkItem wi)
    {
        StringMapAttribute wsma = header.getAttributes();
        String title = "Workflow " + wsma.get(Header.WF_DEFINITION_NAME) + " " + wsma.get(Header.WF_DEFINITION_REVISION) + ": " + wsma.get(Header.WF_INSTANCE_ID);
        System.out.println("\n" + title);
        for(int i=0; i<title.length(); i++)
        {
            System.out.print("=");
        }
        System.out.println();
            
        FlowExpressionId feId = wi.getLastExpressionId();

        System.out.println("Last modified:   " + wi.getLastModified());
        System.out.println("Dispatch time:   " + wi.getDispatchTime());
        System.out.println("Participant:     " + wi.getParticipantName());
        System.out.println("Engine:          " + feId.getEngineId());
        System.out.println("WF def name:     " + feId.getWorkflowDefinitionName());
        System.out.println("WF def revision: " + feId.getWorkflowDefinitionRevision());
        System.out.println("WF instance id:  " + feId.getWorkflowInstanceId());
        System.out.println("Expression name: " + feId.getExpressionName());
        System.out.println("Expression id:   " + feId.getExpressionId());
            
        displayAttributes(wi.getAttributes());
            
        if(wi.getHistory() != null)
        {
            System.out.println("\n\nHistory\n=======");

            for(Iterator i = wi.getHistory().iterator(); i.hasNext();)
            {
                HistoryItem hi = (HistoryItem)i.next();
                System.out.println("Date:   " + hi.getDate());
                System.out.println("Author: " + hi.getAuthor());
                System.out.println("Host:   " + hi.getHost());
                System.out.println("Text:   " + hi.getText());
                System.out.println("----");
            }
            System.out.println();
        }
    }
    
    public void run()
    {
        final CmdHandler parser = new CmdHandler(this);
        parser.commandLoop(prompt, reader);

        System.out.println("Bye.");

        System.exit(0);
    }

    public static void main(String[] args)
    {
        final String DEFAULT_URL = "rmi://localhost:7099/workSessionServer";
        
        System.out.println(BANNER);

        if(args.length>0)
        {
            String arg = args[0].trim().toLowerCase();
            if(arg.equals("-help") || arg.equals("-h") || arg.equals("-?"))
            {
                System.out.println("OpenWFE workflow client");
                System.out.println();
                System.out.println("Usage: " + WfCmd.class.getName() + " [<session server URL> [<username> [<password>]]");
                System.out.println();
                System.out.println("Default URL is " + DEFAULT_URL);
                System.exit(0);
            }
        }

        try
        {
            String url = args.length > 0 ? args[0] : DEFAULT_URL;

            BufferedReader reader = new BufferedReader(new InputStreamReader(System.in));

            // Get the username/password if not specified on the command line.
            //
            
            String username;
            if(args.length>1)
            {
                username = args[1];
            }
            else
            {
                System.out.print("username> ");
                username = reader.readLine();
            }

            String password;
            if(args.length>2)
            {
                password = args[2];
            }
            else
            {
                System.out.print("password> ");
                password = reader.readLine();
            }

            // Login to the session server.
            //

            WorkSessionServer sessionServer = (WorkSessionServer) Naming.lookup(url);

            WorkSession session = (WorkSession) sessionServer.login(username, password);

            new WfCmd(url, session, reader).run();
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }
}
