/*
 * Decompiled with CFR 0.152.
 */
package org.opentripplanner.updater.vehicle_position;

import com.google.common.base.Strings;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.MessageOrBuilder;
import com.google.protobuf.util.JsonFormat;
import com.google.transit.realtime.GtfsRealtime;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Stream;
import org.opentripplanner.framework.geometry.WgsCoordinate;
import org.opentripplanner.service.realtimevehicles.RealtimeVehicleRepository;
import org.opentripplanner.service.realtimevehicles.model.RealtimeVehicle;
import org.opentripplanner.service.realtimevehicles.model.RealtimeVehicleBuilder;
import org.opentripplanner.standalone.config.routerconfig.updaters.VehiclePositionsUpdaterConfig;
import org.opentripplanner.transit.model.framework.FeedScopedId;
import org.opentripplanner.transit.model.framework.Result;
import org.opentripplanner.transit.model.network.TripPattern;
import org.opentripplanner.transit.model.site.StopLocation;
import org.opentripplanner.transit.model.timetable.OccupancyStatus;
import org.opentripplanner.transit.model.timetable.Trip;
import org.opentripplanner.transit.model.timetable.TripTimes;
import org.opentripplanner.updater.spi.ResultLogger;
import org.opentripplanner.updater.spi.UpdateError;
import org.opentripplanner.updater.spi.UpdateResult;
import org.opentripplanner.updater.spi.UpdateSuccess;
import org.opentripplanner.updater.trip.gtfs.GtfsRealtimeFuzzyTripMatcher;
import org.opentripplanner.utils.lang.StringUtils;
import org.opentripplanner.utils.time.ServiceDateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class RealtimeVehiclePatternMatcher {
    private static final Logger LOG = LoggerFactory.getLogger(RealtimeVehiclePatternMatcher.class);
    private final String feedId;
    private final RealtimeVehicleRepository repository;
    private final ZoneId timeZoneId;
    private final Function<FeedScopedId, Trip> getTripForId;
    private final Function<Trip, TripPattern> getStaticPattern;
    private final BiFunction<Trip, LocalDate, TripPattern> getRealtimePattern;
    private final GtfsRealtimeFuzzyTripMatcher fuzzyTripMatcher;
    private final Set<VehiclePositionsUpdaterConfig.VehiclePositionFeature> vehiclePositionFeatures;

    public RealtimeVehiclePatternMatcher(String feedId, Function<FeedScopedId, Trip> getTripForId, Function<Trip, TripPattern> getStaticPattern, BiFunction<Trip, LocalDate, TripPattern> getRealtimePattern, RealtimeVehicleRepository repository, ZoneId timeZoneId, GtfsRealtimeFuzzyTripMatcher fuzzyTripMatcher, Set<VehiclePositionsUpdaterConfig.VehiclePositionFeature> vehiclePositionFeatures) {
        this.feedId = feedId;
        this.getTripForId = getTripForId;
        this.getStaticPattern = getStaticPattern;
        this.getRealtimePattern = getRealtimePattern;
        this.repository = repository;
        this.timeZoneId = timeZoneId;
        this.fuzzyTripMatcher = fuzzyTripMatcher;
        this.vehiclePositionFeatures = vehiclePositionFeatures;
    }

    public UpdateResult applyRealtimeVehicleUpdates(List<GtfsRealtime.VehiclePosition> vehiclePositions) {
        List<Result> matchResults = vehiclePositions.stream().map(vehiclePosition -> this.toRealtimeVehicle(this.feedId, (GtfsRealtime.VehiclePosition)vehiclePosition)).toList();
        ArrayListMultimap vehicles = (ArrayListMultimap)matchResults.stream().filter(Result::isSuccess).map(Result::successValue).collect(Multimaps.toMultimap(PatternAndRealtimeVehicle::pattern, PatternAndRealtimeVehicle::vehicle, ArrayListMultimap::create));
        this.repository.setRealtimeVehiclesForFeed(this.feedId, (Multimap<TripPattern, RealtimeVehicle>)vehicles);
        if (!vehiclePositions.isEmpty() && vehicles.keySet().isEmpty()) {
            LOG.error("Could not match any vehicle positions for feedId '{}'. Are you sure that the updater is using the correct feedId?", (Object)this.feedId);
        }
        List<Result> results = matchResults.stream().map(e -> e.mapSuccess(ignored -> UpdateSuccess.noWarnings())).toList();
        UpdateResult updateResult = UpdateResult.ofResults(new ArrayList<Result<UpdateSuccess, UpdateError>>(results));
        ResultLogger.logUpdateResult(this.feedId, "gtfs-rt-vehicle-positions", updateResult);
        return updateResult;
    }

    private LocalDate inferServiceDate(Trip trip) {
        TripTimes staticTripTimes = this.getStaticPattern.apply(trip).getScheduledTimetable().getTripTimes(trip);
        return RealtimeVehiclePatternMatcher.inferServiceDate(staticTripTimes, this.timeZoneId, Instant.now());
    }

    protected static LocalDate inferServiceDate(TripTimes staticTripTimes, ZoneId zoneId, Instant now) {
        int start = staticTripTimes.getScheduledDepartureTime(0);
        int end = staticTripTimes.getScheduledDepartureTime(staticTripTimes.getNumStops() - 1);
        LocalDate today = now.atZone(zoneId).toLocalDate();
        LocalDate yesterday = today.minusDays(1L);
        LocalDate tomorrow = today.plusDays(1L);
        return Stream.of(yesterday, today, tomorrow).flatMap(day -> {
            Instant startTime = ServiceDateUtils.toZonedDateTime((LocalDate)day, (ZoneId)zoneId, (int)start).toInstant();
            Instant endTime = ServiceDateUtils.toZonedDateTime((LocalDate)day, (ZoneId)zoneId, (int)end).toInstant();
            return Stream.of(Duration.between(startTime, now), Duration.between(endTime, now)).map(Duration::abs).map(duration -> new TemporalDistance((LocalDate)day, duration.toSeconds()));
        }).min(Comparator.comparingLong(TemporalDistance::distance)).map(TemporalDistance::date).orElse(today);
    }

    private RealtimeVehicle mapRealtimeVehicle(GtfsRealtime.VehiclePosition vehiclePosition, List<StopLocation> stopsOnVehicleTrip, Trip trip, Function<Integer, OptionalInt> stopIndexOfGtfsSequence) {
        RealtimeVehicleBuilder newVehicle = RealtimeVehicle.builder();
        if (this.vehiclePositionFeatures.contains((Object)VehiclePositionsUpdaterConfig.VehiclePositionFeature.POSITION) && vehiclePosition.hasPosition()) {
            GtfsRealtime.Position position = vehiclePosition.getPosition();
            newVehicle.withCoordinates(new WgsCoordinate(position.getLatitude(), position.getLongitude()));
            if (position.hasSpeed()) {
                newVehicle.withSpeed(position.getSpeed());
            }
            if (position.hasBearing()) {
                newVehicle.withHeading(position.getBearing());
            }
        }
        if (vehiclePosition.hasVehicle()) {
            GtfsRealtime.VehicleDescriptor vehicle = vehiclePosition.getVehicle();
            FeedScopedId id = new FeedScopedId(this.feedId, vehicle.getId());
            newVehicle.withVehicleId(id).withLabel(Optional.ofNullable(vehicle.getLabel()).orElse(vehicle.getLicensePlate()));
        }
        if (vehiclePosition.hasTimestamp()) {
            newVehicle.withTime(Instant.ofEpochSecond(vehiclePosition.getTimestamp()));
        }
        if (this.vehiclePositionFeatures.contains((Object)VehiclePositionsUpdaterConfig.VehiclePositionFeature.STOP_POSITION)) {
            if (vehiclePosition.hasCurrentStatus()) {
                newVehicle.withStopStatus(RealtimeVehiclePatternMatcher.stopStatusToModel(vehiclePosition.getCurrentStatus()));
            }
            if (vehiclePosition.hasStopId()) {
                List<StopLocation> matchedStops = stopsOnVehicleTrip.stream().filter(stop -> stop.getId().getId().equals(vehiclePosition.getStopId())).toList();
                if (matchedStops.size() == 1) {
                    newVehicle.withStop(matchedStops.get(0));
                } else {
                    LOG.warn("Stop ID {} is not in trip {}. Not setting stopRelationship.", (Object)vehiclePosition.getStopId(), (Object)trip.getId());
                }
            } else if (vehiclePosition.hasCurrentStopSequence()) {
                stopIndexOfGtfsSequence.apply(vehiclePosition.getCurrentStopSequence()).ifPresent(stopIndex -> {
                    if (RealtimeVehiclePatternMatcher.validStopIndex(stopIndex, stopsOnVehicleTrip)) {
                        StopLocation stop = (StopLocation)stopsOnVehicleTrip.get(stopIndex);
                        newVehicle.withStop(stop);
                    }
                });
            }
        }
        newVehicle.withTrip(trip);
        if (this.vehiclePositionFeatures.contains((Object)VehiclePositionsUpdaterConfig.VehiclePositionFeature.OCCUPANCY) && vehiclePosition.hasOccupancyStatus()) {
            newVehicle.withOccupancyStatus(RealtimeVehiclePatternMatcher.occupancyStatusToModel(vehiclePosition.getOccupancyStatus()));
        }
        return newVehicle.build();
    }

    private static boolean validStopIndex(int stopIndex, List<StopLocation> stopsOnVehicleTrip) {
        return stopIndex < stopsOnVehicleTrip.size() - 1;
    }

    private static RealtimeVehicle.StopStatus stopStatusToModel(GtfsRealtime.VehiclePosition.VehicleStopStatus currentStatus) {
        return switch (currentStatus) {
            default -> throw new MatchException(null, null);
            case GtfsRealtime.VehiclePosition.VehicleStopStatus.IN_TRANSIT_TO -> RealtimeVehicle.StopStatus.IN_TRANSIT_TO;
            case GtfsRealtime.VehiclePosition.VehicleStopStatus.INCOMING_AT -> RealtimeVehicle.StopStatus.INCOMING_AT;
            case GtfsRealtime.VehiclePosition.VehicleStopStatus.STOPPED_AT -> RealtimeVehicle.StopStatus.STOPPED_AT;
        };
    }

    private static OccupancyStatus occupancyStatusToModel(GtfsRealtime.VehiclePosition.OccupancyStatus occupancyStatus) {
        return switch (occupancyStatus) {
            default -> throw new MatchException(null, null);
            case GtfsRealtime.VehiclePosition.OccupancyStatus.NO_DATA_AVAILABLE -> OccupancyStatus.NO_DATA_AVAILABLE;
            case GtfsRealtime.VehiclePosition.OccupancyStatus.EMPTY -> OccupancyStatus.EMPTY;
            case GtfsRealtime.VehiclePosition.OccupancyStatus.MANY_SEATS_AVAILABLE -> OccupancyStatus.MANY_SEATS_AVAILABLE;
            case GtfsRealtime.VehiclePosition.OccupancyStatus.FEW_SEATS_AVAILABLE -> OccupancyStatus.FEW_SEATS_AVAILABLE;
            case GtfsRealtime.VehiclePosition.OccupancyStatus.STANDING_ROOM_ONLY -> OccupancyStatus.STANDING_ROOM_ONLY;
            case GtfsRealtime.VehiclePosition.OccupancyStatus.CRUSHED_STANDING_ROOM_ONLY -> OccupancyStatus.CRUSHED_STANDING_ROOM_ONLY;
            case GtfsRealtime.VehiclePosition.OccupancyStatus.FULL -> OccupancyStatus.FULL;
            case GtfsRealtime.VehiclePosition.OccupancyStatus.NOT_ACCEPTING_PASSENGERS -> OccupancyStatus.NOT_ACCEPTING_PASSENGERS;
            case GtfsRealtime.VehiclePosition.OccupancyStatus.NOT_BOARDABLE -> OccupancyStatus.NOT_ACCEPTING_PASSENGERS;
        };
    }

    private static String toString(GtfsRealtime.VehiclePosition vehiclePosition) {
        try {
            return JsonFormat.printer().omittingInsignificantWhitespace().print((MessageOrBuilder)vehiclePosition);
        }
        catch (InvalidProtocolBufferException ignored) {
            return vehiclePosition.toString();
        }
    }

    private GtfsRealtime.VehiclePosition fuzzilySetTrip(GtfsRealtime.VehiclePosition vehiclePosition) {
        GtfsRealtime.TripDescriptor trip = this.fuzzyTripMatcher.match(this.feedId, vehiclePosition.getTrip());
        return vehiclePosition.toBuilder().setTrip(trip).build();
    }

    private Result<PatternAndRealtimeVehicle, UpdateError> toRealtimeVehicle(String feedId, GtfsRealtime.VehiclePosition vehiclePosition) {
        if (!vehiclePosition.hasTrip()) {
            LOG.debug("Realtime vehicle positions {} has no trip ID. Ignoring.", (Object)RealtimeVehiclePatternMatcher.toString(vehiclePosition));
            return Result.failure(UpdateError.noTripId(UpdateError.UpdateErrorType.INVALID_INPUT_STRUCTURE));
        }
        GtfsRealtime.VehiclePosition vehiclePositionWithTripId = this.fuzzyTripMatcher == null ? vehiclePosition : this.fuzzilySetTrip(vehiclePosition);
        String tripId = vehiclePositionWithTripId.getTrip().getTripId();
        if (StringUtils.hasNoValue((String)tripId)) {
            return Result.failure(UpdateError.noTripId(UpdateError.UpdateErrorType.NO_TRIP_ID));
        }
        FeedScopedId scopedTripId = new FeedScopedId(feedId, tripId);
        Trip trip = this.getTripForId.apply(scopedTripId);
        if (trip == null) {
            LOG.debug("Unable to find trip ID in feed '{}' for vehicle position with trip ID {}", (Object)feedId, (Object)tripId);
            return UpdateError.result(scopedTripId, UpdateError.UpdateErrorType.TRIP_NOT_FOUND);
        }
        LocalDate serviceDate = Optional.of(vehiclePositionWithTripId.getTrip().getStartDate()).map(Strings::emptyToNull).flatMap(ServiceDateUtils::parseStringToOptional).orElseGet(() -> this.inferServiceDate(trip));
        TripPattern pattern = this.getRealtimePattern.apply(trip, serviceDate);
        if (pattern == null) {
            LOG.debug("Unable to match OTP pattern ID for vehicle position with trip ID {}", (Object)tripId);
            return UpdateError.result(scopedTripId, UpdateError.UpdateErrorType.NO_SERVICE_ON_DATE);
        }
        TripTimes staticTripTimes = this.getStaticPattern.apply(trip).getScheduledTimetable().getTripTimes(trip);
        if (staticTripTimes == null) {
            return UpdateError.result(scopedTripId, UpdateError.UpdateErrorType.TRIP_NOT_FOUND_IN_PATTERN);
        }
        RealtimeVehicle newVehicle = this.mapRealtimeVehicle(vehiclePositionWithTripId, pattern.getStops(), trip, staticTripTimes::stopIndexOfGtfsSequence);
        return Result.success(new PatternAndRealtimeVehicle(pattern, newVehicle));
    }

    private record PatternAndRealtimeVehicle(TripPattern pattern, RealtimeVehicle vehicle) {
    }

    private record TemporalDistance(LocalDate date, long distance) {
    }
}

