/*
 * Decompiled with CFR 0.152.
 */
package org.opentripplanner.updater.trip.gtfs;

import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.Multimaps;
import com.google.transit.realtime.GtfsRealtime;
import java.text.ParseException;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;
import javax.annotation.Nullable;
import org.opentripplanner.framework.i18n.NonLocalizedString;
import org.opentripplanner.gtfs.mapping.TransitModeMapper;
import org.opentripplanner.model.PickDrop;
import org.opentripplanner.model.RealTimeTripUpdate;
import org.opentripplanner.model.Timetable;
import org.opentripplanner.transit.model.basic.TransitMode;
import org.opentripplanner.transit.model.framework.DataValidationException;
import org.opentripplanner.transit.model.framework.Deduplicator;
import org.opentripplanner.transit.model.framework.FeedScopedId;
import org.opentripplanner.transit.model.framework.Result;
import org.opentripplanner.transit.model.network.Route;
import org.opentripplanner.transit.model.network.RouteBuilder;
import org.opentripplanner.transit.model.network.StopPattern;
import org.opentripplanner.transit.model.network.TripPattern;
import org.opentripplanner.transit.model.organization.Agency;
import org.opentripplanner.transit.model.site.RegularStop;
import org.opentripplanner.transit.model.site.StopLocation;
import org.opentripplanner.transit.model.timetable.RealTimeState;
import org.opentripplanner.transit.model.timetable.RealTimeTripTimes;
import org.opentripplanner.transit.model.timetable.RealTimeTripTimesBuilder;
import org.opentripplanner.transit.model.timetable.Trip;
import org.opentripplanner.transit.model.timetable.TripBuilder;
import org.opentripplanner.transit.model.timetable.TripOnServiceDate;
import org.opentripplanner.transit.model.timetable.TripTimes;
import org.opentripplanner.transit.service.DefaultTransitService;
import org.opentripplanner.transit.service.TimetableRepository;
import org.opentripplanner.transit.service.TransitEditorService;
import org.opentripplanner.updater.spi.DataValidationExceptionMapper;
import org.opentripplanner.updater.spi.ResultLogger;
import org.opentripplanner.updater.spi.UpdateError;
import org.opentripplanner.updater.spi.UpdateResult;
import org.opentripplanner.updater.spi.UpdateSuccess;
import org.opentripplanner.updater.trip.TimetableSnapshotManager;
import org.opentripplanner.updater.trip.UpdateIncrementality;
import org.opentripplanner.updater.trip.gtfs.BackwardsDelayPropagationType;
import org.opentripplanner.updater.trip.gtfs.ForwardsDelayPropagationType;
import org.opentripplanner.updater.trip.gtfs.GtfsRealtimeFuzzyTripMatcher;
import org.opentripplanner.updater.trip.gtfs.StopAndStopTimeUpdate;
import org.opentripplanner.updater.trip.gtfs.TripPatternCache;
import org.opentripplanner.updater.trip.gtfs.TripTimesUpdater;
import org.opentripplanner.updater.trip.gtfs.TripTimesWithStopPattern;
import org.opentripplanner.updater.trip.gtfs.model.AddedRoute;
import org.opentripplanner.updater.trip.gtfs.model.StopTimeUpdate;
import org.opentripplanner.updater.trip.gtfs.model.TripDescriptor;
import org.opentripplanner.updater.trip.gtfs.model.TripTimesPatch;
import org.opentripplanner.updater.trip.gtfs.model.TripUpdate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;

public class GtfsRealTimeTripUpdateAdapter {
    private static final Logger LOG = LoggerFactory.getLogger(GtfsRealTimeTripUpdateAdapter.class);
    private final TripPatternCache tripPatternCache = new TripPatternCache();
    private final ZoneId timeZone;
    private final TransitEditorService transitEditorService;
    private final Deduplicator deduplicator;
    private final Map<FeedScopedId, Integer> serviceCodes;
    private final TimetableSnapshotManager snapshotManager;
    private final Supplier<LocalDate> localDateNow;

    public GtfsRealTimeTripUpdateAdapter(TimetableRepository timetableRepository, TimetableSnapshotManager snapshotManager, Supplier<LocalDate> localDateNow) {
        this.snapshotManager = snapshotManager;
        this.timeZone = timetableRepository.getTimeZone();
        this.localDateNow = localDateNow;
        this.transitEditorService = new DefaultTransitService(timetableRepository, snapshotManager.getTimetableSnapshotBuffer());
        this.deduplicator = timetableRepository.getDeduplicator();
        this.serviceCodes = timetableRepository.getServiceCodes();
    }

    public UpdateResult applyTripUpdates(@Nullable GtfsRealtimeFuzzyTripMatcher fuzzyTripMatcher, ForwardsDelayPropagationType forwardsDelayPropagationType, BackwardsDelayPropagationType backwardsDelayPropagationType, UpdateIncrementality updateIncrementality, List<GtfsRealtime.TripUpdate> updates, String feedId) {
        HashMap<GtfsRealtime.TripDescriptor.ScheduleRelationship, Integer> failuresByRelationship = new HashMap<GtfsRealtime.TripDescriptor.ScheduleRelationship, Integer>();
        ArrayList<Result<UpdateSuccess, UpdateError>> results = new ArrayList<Result<UpdateSuccess, UpdateError>>();
        if (updateIncrementality == UpdateIncrementality.FULL_DATASET) {
            this.snapshotManager.clearBuffer(feedId);
        }
        GtfsRealTimeTripUpdateAdapter.debug(feedId, "message contains {} trip updates", updates.size());
        for (int i = 0; i < updates.size(); ++i) {
            int uIndex = i;
            GtfsRealtime.TripUpdate rawTripUpdate = updates.get(uIndex);
            if (fuzzyTripMatcher != null) {
                GtfsRealtime.TripDescriptor trip = fuzzyTripMatcher.match(feedId, rawTripUpdate.getTrip());
                rawTripUpdate = rawTripUpdate.toBuilder().setTrip(trip).build();
            }
            TripUpdate tripUpdate = new TripUpdate(rawTripUpdate);
            TripDescriptor tripDescriptor = tripUpdate.tripDescriptor();
            tripDescriptor.tripId().map(id -> new FeedScopedId(feedId, (String)id)).ifPresentOrElse(tripId -> {
                Result result;
                LocalDate serviceDate;
                try {
                    serviceDate = tripDescriptor.startDate().orElse(this.localDateNow.get());
                }
                catch (ParseException e) {
                    GtfsRealTimeTripUpdateAdapter.debug(tripId, null, "Failed to parse start date in gtfs-rt trip update: {}", e.getMessage());
                    return;
                }
                GtfsRealtime.TripDescriptor.ScheduleRelationship scheduleRelationship = tripDescriptor.scheduleRelationship();
                if (updateIncrementality == UpdateIncrementality.DIFFERENTIAL) {
                    this.purgePatternModifications(scheduleRelationship, (FeedScopedId)tripId, serviceDate);
                }
                if (LOG.isTraceEnabled()) {
                    GtfsRealTimeTripUpdateAdapter.trace(tripId, serviceDate, "trip update #{} ({} updates): {}", uIndex, updates.size(), tripUpdate);
                } else {
                    GtfsRealTimeTripUpdateAdapter.debug(tripId, serviceDate, "trip update #{} ({} updates)", uIndex, updates.size());
                }
                try {
                    result = switch (scheduleRelationship) {
                        default -> throw new MatchException(null, null);
                        case GtfsRealtime.TripDescriptor.ScheduleRelationship.SCHEDULED -> this.handleScheduledTrip(tripUpdate, (FeedScopedId)tripId, serviceDate, forwardsDelayPropagationType, backwardsDelayPropagationType);
                        case GtfsRealtime.TripDescriptor.ScheduleRelationship.NEW, GtfsRealtime.TripDescriptor.ScheduleRelationship.ADDED -> this.validateAndHandleNewTrip(tripUpdate, (FeedScopedId)tripId, serviceDate);
                        case GtfsRealtime.TripDescriptor.ScheduleRelationship.CANCELED -> this.handleCanceledTrip((FeedScopedId)tripId, serviceDate, CancelationType.CANCEL, updateIncrementality);
                        case GtfsRealtime.TripDescriptor.ScheduleRelationship.DELETED -> this.handleCanceledTrip((FeedScopedId)tripId, serviceDate, CancelationType.DELETE, updateIncrementality);
                        case GtfsRealtime.TripDescriptor.ScheduleRelationship.REPLACEMENT -> this.validateAndHandleReplacementTrip(tripUpdate, (FeedScopedId)tripId, serviceDate);
                        case GtfsRealtime.TripDescriptor.ScheduleRelationship.UNSCHEDULED -> UpdateError.result(tripId, UpdateError.UpdateErrorType.NOT_IMPLEMENTED_UNSCHEDULED);
                        case GtfsRealtime.TripDescriptor.ScheduleRelationship.DUPLICATED -> UpdateError.result(tripId, UpdateError.UpdateErrorType.NOT_IMPLEMENTED_DUPLICATED);
                    };
                }
                catch (DataValidationException e) {
                    result = DataValidationExceptionMapper.toResult(e);
                }
                results.add(result);
                if (result.isFailure()) {
                    GtfsRealTimeTripUpdateAdapter.debug(tripId, serviceDate, "Failed to apply TripUpdate.", new Object[0]);
                    if (failuresByRelationship.containsKey(scheduleRelationship)) {
                        Integer c = (Integer)failuresByRelationship.get(scheduleRelationship);
                        c = c + 1;
                        failuresByRelationship.put(scheduleRelationship, c);
                    } else {
                        failuresByRelationship.put(scheduleRelationship, 1);
                    }
                }
            }, () -> {
                GtfsRealTimeTripUpdateAdapter.debug(feedId, "No trip id found for gtfs-rt trip update: \n{}", tripUpdate);
                results.add(Result.failure(UpdateError.noTripId(UpdateError.UpdateErrorType.INVALID_INPUT_STRUCTURE)));
            });
        }
        UpdateResult updateResult = UpdateResult.ofResults(results);
        if (updateIncrementality == UpdateIncrementality.FULL_DATASET) {
            GtfsRealTimeTripUpdateAdapter.logUpdateResult(feedId, failuresByRelationship, updateResult);
        }
        return updateResult;
    }

    private void purgePatternModifications(GtfsRealtime.TripDescriptor.ScheduleRelationship tripScheduleRelationship, FeedScopedId tripId, LocalDate serviceDate) {
        TripPattern pattern = this.snapshotManager.getNewTripPatternForModifiedTrip(tripId, serviceDate);
        if (!this.isPreviouslyAddedTrip(tripId, pattern, serviceDate) || tripScheduleRelationship != GtfsRealtime.TripDescriptor.ScheduleRelationship.CANCELED && tripScheduleRelationship != GtfsRealtime.TripDescriptor.ScheduleRelationship.DELETED) {
            this.snapshotManager.revertTripToScheduledTripPattern(tripId, serviceDate);
        }
    }

    private boolean isPreviouslyAddedTrip(FeedScopedId tripId, TripPattern pattern, LocalDate serviceDate) {
        if (pattern == null) {
            return false;
        }
        Timetable timetable = this.snapshotManager.resolve(pattern, serviceDate);
        if (timetable == null) {
            return false;
        }
        TripTimes tripTimes = timetable.getTripTimes(tripId);
        if (tripTimes == null) {
            return false;
        }
        return tripTimes.getRealTimeState() == RealTimeState.ADDED;
    }

    private static void logUpdateResult(String feedId, Map<GtfsRealtime.TripDescriptor.ScheduleRelationship, Integer> failuresByRelationship, UpdateResult updateResult) {
        ResultLogger.logUpdateResult(feedId, "gtfs-rt-trip-updates", updateResult);
        if (!failuresByRelationship.isEmpty()) {
            GtfsRealTimeTripUpdateAdapter.info(feedId, "Failures by scheduleRelationship {}", failuresByRelationship);
        }
        ImmutableListMultimap warnings = Multimaps.index(updateResult.warnings(), w -> w);
        warnings.keySet().forEach(key -> {
            int count = warnings.get((Object)key).size();
            GtfsRealTimeTripUpdateAdapter.info(feedId, "{} warnings of type {}", count, key);
        });
    }

    private Result<UpdateSuccess, UpdateError> handleScheduledTrip(TripUpdate tripUpdate, FeedScopedId tripId, LocalDate serviceDate, ForwardsDelayPropagationType forwardsDelayPropagationType, BackwardsDelayPropagationType backwardsDelayPropagationType) {
        TripPattern pattern = this.getPatternForTripId(tripId);
        if (pattern == null) {
            GtfsRealTimeTripUpdateAdapter.debug(tripId, serviceDate, "No pattern found for tripId, skipping TripUpdate.", new Object[0]);
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.TRIP_NOT_FOUND);
        }
        if (tripUpdate.stopTimeUpdates().isEmpty()) {
            GtfsRealTimeTripUpdateAdapter.debug(tripId, serviceDate, "TripUpdate contains no updates, skipping.", new Object[0]);
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.NO_UPDATES);
        }
        FeedScopedId serviceId = this.transitEditorService.getTrip(tripId).getServiceId();
        Set<LocalDate> serviceDates = this.transitEditorService.getCalendarService().getServiceDatesForServiceId(serviceId);
        if (!serviceDates.contains(serviceDate)) {
            GtfsRealTimeTripUpdateAdapter.debug(tripId, serviceDate, "SCHEDULED trip has service date {} for which trip's service is not valid, skipping.", serviceDate.toString());
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.NO_SERVICE_ON_DATE);
        }
        Result<TripTimesPatch, UpdateError> result = TripTimesUpdater.createUpdatedTripTimesFromGtfsRt(pattern.getScheduledTimetable(), tripUpdate, this.timeZone, serviceDate, forwardsDelayPropagationType, backwardsDelayPropagationType);
        if (result.isFailure()) {
            return result.toFailureResult();
        }
        TripTimesPatch tripTimesPatch = result.successValue();
        Map<Integer, PickDrop> updatedPickup = tripTimesPatch.updatedPickup();
        Map<Integer, PickDrop> updatedDropoff = tripTimesPatch.updatedDropoff();
        Map<Integer, String> replacedStopIndices = tripTimesPatch.replacedStopIndices();
        RealTimeTripTimes updatedTripTimes = tripTimesPatch.tripTimes();
        HashMap<Integer, StopLocation> newStops = new HashMap<Integer, StopLocation>();
        for (Map.Entry<Integer, String> entry : replacedStopIndices.entrySet()) {
            RegularStop stop = this.transitEditorService.getRegularStop(new FeedScopedId(tripId.getFeedId(), entry.getValue()));
            if (stop != null) {
                newStops.put(entry.getKey(), stop);
                continue;
            }
            GtfsRealTimeTripUpdateAdapter.debug(tripId, serviceDate, "Graph doesn't contain assigned stop id '{}' at position '{}' for trip '{}' , skipping stop assignment.", entry.getValue(), entry.getKey(), tripId);
        }
        if (!(updatedPickup.isEmpty() && updatedDropoff.isEmpty() && newStops.isEmpty())) {
            StopPattern newStopPattern = pattern.copyPlannedStopPattern().updatePickups(updatedPickup).updateDropoffs(updatedDropoff).replaceStops(newStops).build();
            Trip trip = this.transitEditorService.getTrip(tripId);
            TripPattern newPattern = this.tripPatternCache.getOrCreateTripPattern(newStopPattern, trip, pattern);
            this.cancelScheduledTrip(tripId, serviceDate, CancelationType.DELETE);
            return this.snapshotManager.updateBuffer(new RealTimeTripUpdate(newPattern, updatedTripTimes, serviceDate));
        }
        return this.snapshotManager.updateBuffer(new RealTimeTripUpdate(pattern, updatedTripTimes, serviceDate));
    }

    private Result<UpdateSuccess, UpdateError> validateAndHandleNewTrip(TripUpdate tripUpdate, FeedScopedId tripId, LocalDate serviceDate) {
        if (this.transitEditorService.getScheduledTrip(tripId) != null) {
            GtfsRealTimeTripUpdateAdapter.debug(tripId, serviceDate, "Graph already contains trip id of NEW trip, skipping.", new Object[0]);
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.TRIP_ALREADY_EXISTS);
        }
        TripDescriptor tripDescriptor = tripUpdate.tripDescriptor();
        Optional<Route> optionalRoute = this.getRoute(tripId.getFeedId(), tripDescriptor);
        Route route = optionalRoute.orElseGet(() -> this.createRoute(tripDescriptor, tripId));
        TripBuilder tripBuilder = Trip.of(tripId);
        tripBuilder.withRoute(route);
        Set<FeedScopedId> serviceIds = this.transitEditorService.getCalendarService().getServiceIdsOnDate(serviceDate);
        if (serviceIds.isEmpty()) {
            GtfsRealTimeTripUpdateAdapter.debug(tripId, serviceDate, "NEW trip has service date {} for which no service id is available, skipping.", serviceDate.toString());
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.NO_SERVICE_ON_DATE);
        }
        tripBuilder.withServiceId(serviceIds.iterator().next());
        tripUpdate.tripHeadsign().ifPresent(tripBuilder::withHeadsign);
        tripUpdate.tripShortName().ifPresent(tripBuilder::withShortName);
        Trip trip = (Trip)tripBuilder.build();
        return this.handleNewOrReplacementTrip(trip, tripUpdate, serviceDate, RealTimeState.ADDED, optionalRoute.isEmpty());
    }

    private List<StopAndStopTimeUpdate> matchStopsToStopTimeUpdates(TripUpdate tripUpdate, FeedScopedId tripId, LocalDate serviceDate) {
        return tripUpdate.stopTimeUpdates().stream().flatMap(st -> st.stopId().flatMap(id -> {
            FeedScopedId stopId = new FeedScopedId(tripId.getFeedId(), (String)id);
            RegularStop stop = this.transitEditorService.getRegularStop(stopId);
            if (stop == null) {
                GtfsRealTimeTripUpdateAdapter.debug(tripId, serviceDate, "Stop '{}' not found in graph. Removing from NEW trip.", stopId);
            }
            return stop == null ? Optional.empty() : Optional.of(new StopAndStopTimeUpdate(stop, (StopTimeUpdate)st));
        }).stream()).toList();
    }

    private Result<UpdateSuccess, UpdateError> handleNewOrReplacementTrip(Trip trip, TripUpdate tripUpdate, LocalDate serviceDate, RealTimeState realTimeState, boolean hasANewRouteBeenCreated) {
        FeedScopedId tripId = trip.getId();
        List<StopAndStopTimeUpdate> stopAndStopTimeUpdates = this.matchStopsToStopTimeUpdates(tripUpdate, tripId, serviceDate);
        ArrayList<UpdateSuccess.WarningType> warnings = new ArrayList<UpdateSuccess.WarningType>(0);
        if (stopAndStopTimeUpdates.size() < tripUpdate.stopTimeUpdates().size()) {
            warnings.add(UpdateSuccess.WarningType.UNKNOWN_STOPS_REMOVED_FROM_ADDED_TRIP);
        }
        if (stopAndStopTimeUpdates.size() < 2) {
            GtfsRealTimeTripUpdateAdapter.debug(tripId, serviceDate, "NEW or REPLACEMENT trip has fewer than two known stops, skipping.", new Object[0]);
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.TOO_FEW_STOPS);
        }
        Result<TripTimesWithStopPattern, UpdateError> result = TripTimesUpdater.createNewTripTimesFromGtfsRt(trip, TripTimesUpdater.getWheelchairAccessibility(tripUpdate).orElse(null), stopAndStopTimeUpdates, this.timeZone, serviceDate, realTimeState, tripUpdate.tripHeadsign().orElse(null), this.deduplicator, this.serviceCodes.get(trip.getServiceId()));
        return result.flatMap(value -> this.addNewOrReplacementTripToSnapshot((TripTimesWithStopPattern)value, serviceDate, realTimeState, hasANewRouteBeenCreated)).mapSuccess(s -> s.addWarnings(warnings));
    }

    private Route createRoute(TripDescriptor tripDescriptor, FeedScopedId tripId) {
        Optional<FeedScopedId> routeId = tripDescriptor.routeId().map(id -> new FeedScopedId(tripId.getFeedId(), (String)id));
        return routeId.map(id -> {
            RouteBuilder builder = Route.of(id);
            AddedRoute addedRouteExtension = AddedRoute.ofTripDescriptor(tripDescriptor);
            Agency agency = this.transitEditorService.findAgency(new FeedScopedId(tripId.getFeedId(), addedRouteExtension.agencyId())).orElseGet(() -> this.fallbackAgency(tripId.getFeedId()));
            builder.withAgency(agency);
            builder.withGtfsType(addedRouteExtension.routeType());
            TransitMode mode = TransitModeMapper.mapMode(addedRouteExtension.routeType());
            builder.withMode(mode);
            String name = Objects.requireNonNullElse(addedRouteExtension.routeLongName(), tripId.toString());
            builder.withLongName(new NonLocalizedString(name));
            builder.withUrl(addedRouteExtension.routeUrl());
            return (Route)builder.build();
        }).orElseGet(() -> {
            RouteBuilder builder = Route.of(tripId);
            builder.withAgency(this.fallbackAgency(tripId.getFeedId()));
            builder.withGtfsType(3);
            builder.withMode(TransitMode.BUS);
            NonLocalizedString longName = NonLocalizedString.ofNullable(tripId.getId());
            builder.withLongName(longName);
            return (Route)builder.build();
        });
    }

    private Agency fallbackAgency(String feedId) {
        return (Agency)Agency.of(new FeedScopedId(feedId, "autogenerated-gtfs-rt-added-route")).withName("Agency automatically added by GTFS-RT update").withTimezone(this.transitEditorService.getTimeZone().toString()).build();
    }

    private Optional<Route> getRoute(String feedId, TripDescriptor tripDescriptor) {
        return tripDescriptor.routeId().flatMap(id -> Optional.ofNullable(this.transitEditorService.getRoute(new FeedScopedId(feedId, (String)id))));
    }

    private Result<UpdateSuccess, UpdateError> addNewOrReplacementTripToSnapshot(TripTimesWithStopPattern tripTimesWithStopPattern, LocalDate serviceDate, RealTimeState realTimeState, boolean hasANewRouteBeenCreated) {
        RealTimeTripTimes tripTimes = tripTimesWithStopPattern.tripTimes();
        Trip trip = tripTimes.getTrip();
        if (realTimeState == RealTimeState.MODIFIED) {
            this.cancelScheduledTrip(trip.getId(), serviceDate, CancelationType.DELETE);
        }
        StopPattern stopPattern = tripTimesWithStopPattern.stopPattern();
        TripPattern originalTripPattern = this.transitEditorService.findPattern(trip);
        TripPattern pattern = this.tripPatternCache.getOrCreateTripPattern(stopPattern, trip, originalTripPattern);
        GtfsRealTimeTripUpdateAdapter.trace(trip.getId(), serviceDate, "Trip pattern added with mode {} on {} from {} to {}", trip.getRoute().getMode(), serviceDate, pattern.firstStop().getName(), pattern.lastStop().getName());
        return this.snapshotManager.updateBuffer(new RealTimeTripUpdate(pattern, tripTimes, serviceDate, realTimeState == RealTimeState.ADDED ? (TripOnServiceDate)TripOnServiceDate.of(trip.getId()).withTrip(trip).withServiceDate(serviceDate).build() : null, realTimeState == RealTimeState.ADDED, hasANewRouteBeenCreated));
    }

    private boolean cancelScheduledTrip(FeedScopedId tripId, LocalDate serviceDate, CancelationType cancelationType) {
        boolean success = false;
        TripPattern pattern = this.getPatternForTripId(tripId);
        if (pattern != null) {
            Timetable timetable = pattern.getScheduledTimetable();
            TripTimes tripTimes = timetable.getTripTimes(tripId);
            if (tripTimes == null) {
                GtfsRealTimeTripUpdateAdapter.debug(tripId, serviceDate, "Could not cancel scheduled trip because it's not in the timetable", new Object[0]);
            } else {
                this.cancelTrip(serviceDate, cancelationType, pattern, tripTimes);
                success = true;
            }
        }
        return success;
    }

    private boolean cancelPreviouslyAddedTrip(FeedScopedId tripId, LocalDate serviceDate, CancelationType cancelationType) {
        boolean cancelledAddedTrip = false;
        TripPattern pattern = this.snapshotManager.getNewTripPatternForModifiedTrip(tripId, serviceDate);
        if (this.isPreviouslyAddedTrip(tripId, pattern, serviceDate)) {
            Timetable timetable = this.snapshotManager.resolve(pattern, serviceDate);
            TripTimes tripTimes = timetable.getTripTimes(tripId);
            if (tripTimes == null) {
                GtfsRealTimeTripUpdateAdapter.debug(tripId, serviceDate, "Could not cancel previously added trip on {}", serviceDate);
            } else {
                this.cancelTrip(serviceDate, cancelationType, pattern, tripTimes);
                cancelledAddedTrip = true;
            }
        }
        return cancelledAddedTrip;
    }

    private void cancelTrip(LocalDate serviceDate, CancelationType cancelationType, TripPattern pattern, TripTimes tripTimes) {
        RealTimeTripTimesBuilder builder = tripTimes.createRealTimeFromScheduledTimes();
        switch (cancelationType.ordinal()) {
            case 0: {
                builder.cancelTrip();
                break;
            }
            case 1: {
                builder.deleteTrip();
            }
        }
        this.snapshotManager.updateBuffer(new RealTimeTripUpdate(pattern, builder.build(), serviceDate));
    }

    private Result<UpdateSuccess, UpdateError> validateAndHandleReplacementTrip(TripUpdate tripUpdate, FeedScopedId tripId, LocalDate serviceDate) {
        Objects.requireNonNull(tripUpdate);
        Objects.requireNonNull(serviceDate);
        Trip trip = this.transitEditorService.getTrip(tripId);
        if (trip == null) {
            GtfsRealTimeTripUpdateAdapter.debug(tripId, serviceDate, "Feed does not contain trip id of REPLACEMENT trip, skipping.", new Object[0]);
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.TRIP_NOT_FOUND);
        }
        Set<FeedScopedId> serviceIds = this.transitEditorService.getCalendarService().getServiceIdsOnDate(serviceDate);
        if (!serviceIds.contains(trip.getServiceId())) {
            GtfsRealTimeTripUpdateAdapter.debug(tripId, serviceDate, "REPLACEMENT trip has a service date that is not served by trip, skipping.", new Object[0]);
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.NO_SERVICE_ON_DATE);
        }
        return this.handleNewOrReplacementTrip(trip, tripUpdate, serviceDate, RealTimeState.MODIFIED, false);
    }

    private Result<UpdateSuccess, UpdateError> handleCanceledTrip(FeedScopedId tripId, LocalDate serviceDate, CancelationType cancelationType, UpdateIncrementality incrementality) {
        boolean canceledPreviouslyAddedTrip;
        boolean bl = canceledPreviouslyAddedTrip = incrementality != UpdateIncrementality.FULL_DATASET && this.cancelPreviouslyAddedTrip(tripId, serviceDate, cancelationType);
        if (canceledPreviouslyAddedTrip) {
            return Result.success(UpdateSuccess.noWarnings());
        }
        boolean cancelScheduledSuccess = this.cancelScheduledTrip(tripId, serviceDate, cancelationType);
        if (!cancelScheduledSuccess) {
            GtfsRealTimeTripUpdateAdapter.debug(tripId, serviceDate, "No pattern found for tripId. Skipping cancellation.", new Object[0]);
            return UpdateError.result(tripId, UpdateError.UpdateErrorType.NO_TRIP_FOR_CANCELLATION_FOUND);
        }
        GtfsRealTimeTripUpdateAdapter.debug(tripId, serviceDate, "Canceled trip", new Object[0]);
        return Result.success(UpdateSuccess.noWarnings());
    }

    private TripPattern getPatternForTripId(FeedScopedId tripId) {
        Trip trip = this.transitEditorService.getTrip(tripId);
        return this.transitEditorService.findPattern(trip);
    }

    private static void debug(FeedScopedId id, @Nullable LocalDate serviceDate, String message, Object ... params) {
        GtfsRealTimeTripUpdateAdapter.log(Level.DEBUG, id.getFeedId(), id.getId(), serviceDate, message, params);
    }

    private static void debug(String feedId, String message, Object ... params) {
        GtfsRealTimeTripUpdateAdapter.log(Level.DEBUG, feedId, null, null, message, params);
    }

    private static void trace(FeedScopedId id, @Nullable LocalDate serviceDate, String message, Object ... params) {
        GtfsRealTimeTripUpdateAdapter.log(Level.TRACE, id.getFeedId(), id.getId(), serviceDate, message, params);
    }

    private static void info(String feedId, String message, Object ... params) {
        GtfsRealTimeTripUpdateAdapter.log(Level.INFO, feedId, null, null, message, params);
    }

    private static void log(Level logLevel, String feedId, @Nullable String tripId, @Nullable LocalDate serviceDate, String message, Object ... params) {
        if (LOG.isEnabledForLevel(logLevel)) {
            String m = tripId != null || serviceDate != null ? "[feedId: %s, tripId: %s, serviceDate: %s] %s".formatted(feedId, tripId, serviceDate, message) : "[feedId: %s] %s".formatted(feedId, message);
            LOG.makeLoggingEventBuilder(logLevel).log(m, params);
        }
    }

    private static enum CancelationType {
        CANCEL,
        DELETE;

    }
}

