/*
 * Decompiled with CFR 0.152.
 */
package org.opentripplanner.routing.linking;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Nullable;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.linearref.LinearLocation;
import org.locationtech.jts.linearref.LocationIndexedLine;
import org.opentripplanner.framework.application.OTPFeature;
import org.opentripplanner.framework.geometry.GeometryUtils;
import org.opentripplanner.framework.geometry.SphericalDistanceLibrary;
import org.opentripplanner.routing.graph.Graph;
import org.opentripplanner.routing.linking.DisposableEdgeCollection;
import org.opentripplanner.routing.linking.Scope;
import org.opentripplanner.routing.linking.VisibilityMode;
import org.opentripplanner.street.model.edge.Area;
import org.opentripplanner.street.model.edge.AreaEdge;
import org.opentripplanner.street.model.edge.AreaEdgeBuilder;
import org.opentripplanner.street.model.edge.AreaGroup;
import org.opentripplanner.street.model.edge.Edge;
import org.opentripplanner.street.model.edge.LinkingDirection;
import org.opentripplanner.street.model.edge.SplitStreetEdge;
import org.opentripplanner.street.model.edge.StreetEdge;
import org.opentripplanner.street.model.vertex.IntersectionVertex;
import org.opentripplanner.street.model.vertex.SplitterVertex;
import org.opentripplanner.street.model.vertex.StreetVertex;
import org.opentripplanner.street.model.vertex.TemporarySplitterVertex;
import org.opentripplanner.street.model.vertex.Vertex;
import org.opentripplanner.street.model.vertex.VertexFactory;
import org.opentripplanner.street.search.TraverseMode;
import org.opentripplanner.street.search.TraverseModeSet;
import org.opentripplanner.transit.model.site.AreaStop;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VertexLinker {
    private static final Logger LOG = LoggerFactory.getLogger(VertexLinker.class);
    private static final double DUPLICATE_WAY_EPSILON_DEGREES = SphericalDistanceLibrary.metersToDegrees(0.001);
    private static final double DUPLICATE_NODE_EPSILON_DEGREES_SQUARED = SphericalDistanceLibrary.metersToDegrees(1.0) * SphericalDistanceLibrary.metersToDegrees(1.0);
    private static final double INITIAL_SEARCH_RADIUS_DEGREES = SphericalDistanceLibrary.metersToDegrees(100.0);
    private static final double MAX_SEARCH_RADIUS_DEGREES = SphericalDistanceLibrary.metersToDegrees(1000.0);
    private static final GeometryFactory GEOMETRY_FACTORY = GeometryUtils.getGeometryFactory();
    private static final Set<TraverseMode> NO_THRU_MODES = Set.of(TraverseMode.WALK, TraverseMode.BICYCLE, TraverseMode.CAR);
    private final Graph graph;
    private final VertexFactory vertexFactory;
    private final VisibilityMode visibilityMode;
    private final int maxAreaNodes;

    public VertexLinker(Graph graph, VisibilityMode visibilityMode, int maxAreaNodes) {
        this.graph = Objects.requireNonNull(graph);
        this.vertexFactory = new VertexFactory(graph);
        this.visibilityMode = Objects.requireNonNull(visibilityMode);
        this.maxAreaNodes = maxAreaNodes;
    }

    public void linkVertexPermanently(Vertex vertex, TraverseModeSet traverseModes, LinkingDirection direction, BiFunction<Vertex, StreetVertex, List<Edge>> edgeFunction) {
        this.link(vertex, traverseModes, direction, Scope.PERMANENT, edgeFunction);
    }

    public DisposableEdgeCollection linkVertexForRealTime(Vertex vertex, TraverseModeSet traverseModes, LinkingDirection direction, BiFunction<Vertex, StreetVertex, List<Edge>> edgeFunction) {
        return this.link(vertex, traverseModes, direction, Scope.REALTIME, edgeFunction);
    }

    public DisposableEdgeCollection linkVertexForRequest(Vertex vertex, TraverseModeSet traverseModes, LinkingDirection direction, BiFunction<Vertex, StreetVertex, List<Edge>> edgeFunction) {
        return this.link(vertex, traverseModes, direction, Scope.REQUEST, edgeFunction);
    }

    private void removeEdgeFromIndex(Edge edge, Scope scope) {
        if (edge.getGeometry() != null) {
            this.graph.removeEdge(edge, scope);
        }
    }

    private static double distance(Vertex tstop, StreetEdge edge, double xscale) {
        LineString transformed = VertexLinker.equirectangularProject(edge.getGeometry(), xscale);
        return transformed.distance((Geometry)GEOMETRY_FACTORY.createPoint(new Coordinate(tstop.getLon() * xscale, tstop.getLat())));
    }

    private static LineString equirectangularProject(LineString geometry, double xScale) {
        Coordinate[] coords = new Coordinate[geometry.getNumPoints()];
        for (int i = 0; i < coords.length; ++i) {
            Coordinate c = geometry.getCoordinateN(i);
            c = (Coordinate)c.clone();
            c.x *= xScale;
            coords[i] = c;
        }
        return GEOMETRY_FACTORY.createLineString(coords);
    }

    private DisposableEdgeCollection link(Vertex vertex, TraverseModeSet traverseModes, LinkingDirection direction, Scope scope, BiFunction<Vertex, StreetVertex, List<Edge>> edgeFunction) {
        DisposableEdgeCollection tempEdges = scope != Scope.PERMANENT ? new DisposableEdgeCollection(this.graph, scope) : null;
        try {
            Set<StreetVertex> streetVertices = this.linkToStreetEdges(vertex, traverseModes, direction, scope, INITIAL_SEARCH_RADIUS_DEGREES, tempEdges);
            if (streetVertices.isEmpty() && scope == Scope.REQUEST) {
                streetVertices = this.linkToStreetEdges(vertex, traverseModes, direction, scope, MAX_SEARCH_RADIUS_DEGREES, tempEdges);
            }
            for (StreetVertex streetVertex : streetVertices) {
                List<Edge> edges = edgeFunction.apply(vertex, streetVertex);
                if (tempEdges == null) continue;
                for (Edge edge : edges) {
                    tempEdges.addEdge(edge);
                }
            }
        }
        catch (Exception e) {
            if (tempEdges != null) {
                tempEdges.disposeEdges();
            }
            throw e;
        }
        return tempEdges;
    }

    public Set<StreetVertex> linkToSpecificStreetEdgesPermanently(Vertex vertex, TraverseModeSet traverseModes, LinkingDirection direction, Set<StreetEdge> edges) {
        double xscale = VertexLinker.getXscale(vertex);
        return this.linkToCandidateEdges(vertex, traverseModes, direction, Scope.PERMANENT, null, edges.stream().map(e -> new DistanceTo<StreetEdge>((StreetEdge)e, VertexLinker.distance(vertex, e, xscale))).toList(), xscale);
    }

    private Set<StreetVertex> linkToStreetEdges(Vertex vertex, TraverseModeSet traverseModes, LinkingDirection direction, Scope scope, double radiusDeg, @Nullable DisposableEdgeCollection tempEdges) {
        Envelope env = new Envelope(vertex.getCoordinate());
        double xscale = VertexLinker.getXscale(vertex);
        env.expandBy(radiusDeg / xscale, radiusDeg);
        Collection<Edge> candidateEdges = this.graph.findEdges(env, scope);
        List<DistanceTo<StreetEdge>> candidateDistanceToEdges = candidateEdges.stream().filter(StreetEdge.class::isInstance).map(StreetEdge.class::cast).filter(e -> e.canTraverse(traverseModes) && e.isReachableFromGraph()).map(e -> new DistanceTo<StreetEdge>((StreetEdge)e, VertexLinker.distance(vertex, e, xscale))).filter(ead -> ead.distanceDegreesLat < radiusDeg).toList();
        return this.linkToCandidateEdges(vertex, traverseModes, direction, scope, tempEdges, candidateDistanceToEdges, xscale);
    }

    private static double getXscale(Vertex vertex) {
        return Math.cos(vertex.getLat() * Math.PI / 180.0);
    }

    private Set<StreetVertex> linkToCandidateEdges(Vertex vertex, TraverseModeSet traverseModes, LinkingDirection direction, Scope scope, @Nullable DisposableEdgeCollection tempEdges, List<DistanceTo<StreetEdge>> candidateEdges, double xscale) {
        if (candidateEdges.isEmpty()) {
            return Set.of();
        }
        Set<DistanceTo<StreetEdge>> closestEdges = this.getClosestEdgesPerMode(traverseModes, candidateEdges);
        HashMap linkedAreas = new HashMap();
        return closestEdges.stream().map(ce -> this.snapAndLink(vertex, (StreetEdge)ce.item, xscale, scope, direction, tempEdges, linkedAreas)).filter(Objects::nonNull).collect(Collectors.toSet());
    }

    private Set<DistanceTo<StreetEdge>> getClosestEdgesPerMode(TraverseModeSet traverseModeSet, List<DistanceTo<StreetEdge>> candidateEdges) {
        HashSet<DistanceTo<StreetEdge>> closesEdges = new HashSet<DistanceTo<StreetEdge>>();
        for (TraverseMode mode : traverseModeSet.getModes()) {
            TraverseModeSet modeSet = new TraverseModeSet(mode);
            List<DistanceTo> candidateEdgesForMode = candidateEdges.stream().filter(e -> ((StreetEdge)e.item).canTraverse(modeSet)).toList();
            if (candidateEdgesForMode.isEmpty()) continue;
            double closestDistance = candidateEdgesForMode.stream().mapToDouble(ce -> ce.distanceDegreesLat).min().getAsDouble();
            closesEdges.addAll(candidateEdges.stream().filter(ce -> ce.distanceDegreesLat <= closestDistance + DUPLICATE_WAY_EPSILON_DEGREES).collect(Collectors.toSet()));
        }
        return closesEdges;
    }

    private StreetVertex snapAndLink(Vertex vertex, StreetEdge edge, double xScale, Scope scope, LinkingDirection direction, DisposableEdgeCollection tempEdges, HashMap<AreaGroup, IntersectionVertex> linkedAreas) {
        IntersectionVertex start = null;
        IntersectionVertex split = this.findSplitVertex(vertex, edge, xScale, scope, direction, tempEdges);
        if (this.visibilityMode == VisibilityMode.COMPUTE_AREA_VISIBILITY_LINES && edge instanceof AreaEdge) {
            AreaEdge aEdge = (AreaEdge)edge;
            AreaGroup ag = aEdge.getArea();
            start = linkedAreas.get(ag);
            if (start == null && ag.getGeometry().contains((Geometry)GEOMETRY_FACTORY.createPoint(vertex.getCoordinate()))) {
                if (this.distSquared(vertex, split) <= DUPLICATE_NODE_EPSILON_DEGREES_SQUARED) {
                    start = split;
                } else {
                    IntersectionVertex iv;
                    start = vertex instanceof IntersectionVertex ? (iv = (IntersectionVertex)vertex) : this.createSplitVertex(aEdge, scope, vertex.getLon(), vertex.getLat());
                    linkedAreas.put(ag, start);
                }
            }
            if (start != null && start != split) {
                this.addVisibilityEdges(start, split, ag, scope, tempEdges, true);
            } else {
                start = split;
            }
            if (!ag.visibilityVertices().contains(start)) {
                this.addAreaVertex(start, ag, scope, tempEdges, false);
            }
        } else {
            start = split;
        }
        if (OTPFeature.FlexRouting.isOn()) {
            List<AreaStop> areaStops = Stream.concat(start.getIncoming().stream(), start.getOutgoing().stream()).flatMap(e -> Stream.concat(e.getFromVertex().areaStops().stream(), e.getToVertex().areaStops().stream())).toList();
            start.addAreaStops(areaStops);
        }
        return start;
    }

    private IntersectionVertex findSplitVertex(Vertex vertex, StreetEdge edge, double xScale, Scope scope, LinkingDirection direction, DisposableEdgeCollection tempEdges) {
        LineString geom = edge.getGeometry();
        LineString transformed = VertexLinker.equirectangularProject(geom, xScale);
        LocationIndexedLine il = new LocationIndexedLine((Geometry)transformed);
        LinearLocation ll = il.project(new Coordinate(vertex.getLon() * xScale, vertex.getLat()));
        double length = SphericalDistanceLibrary.length(geom);
        if (ll.getSegmentIndex() == 0 && (ll.getSegmentFraction() < 1.0E-8 || ll.getSegmentFraction() * length < 0.1)) {
            return (IntersectionVertex)edge.getFromVertex();
        }
        if (ll.getSegmentIndex() == geom.getNumPoints() - 1) {
            return (IntersectionVertex)edge.getToVertex();
        }
        if (ll.getSegmentIndex() == geom.getNumPoints() - 2 && (ll.getSegmentFraction() > 0.99999999 || (1.0 - ll.getSegmentFraction()) * length < 0.1)) {
            return (IntersectionVertex)edge.getToVertex();
        }
        return this.split(edge, ll.getCoordinate((Geometry)geom), scope, direction, tempEdges);
    }

    private SplitterVertex split(StreetEdge originalEdge, Coordinate splitPoint, Scope scope, LinkingDirection direction, DisposableEdgeCollection tempEdges) {
        SplitStreetEdge newEdges;
        SplitterVertex v = this.createSplitVertex(originalEdge, scope, splitPoint.x, splitPoint.y);
        SplitStreetEdge splitStreetEdge = newEdges = scope == Scope.PERMANENT ? originalEdge.splitDestructively(v) : originalEdge.splitNonDestructively(v, tempEdges, direction);
        if (scope == Scope.REALTIME || scope == Scope.PERMANENT) {
            if (newEdges.head() != null) {
                this.graph.insert(newEdges.head(), scope);
            }
            if (newEdges.tail() != null) {
                this.graph.insert(newEdges.tail(), scope);
            }
            if (scope == Scope.PERMANENT) {
                this.removeEdgeFromIndex(originalEdge, scope);
                this.graph.removeEdge(originalEdge);
            }
        }
        return v;
    }

    private SplitterVertex createSplitVertex(StreetEdge originalEdge, Scope scope, double x, double y) {
        SplitterVertex v;
        String uniqueSplitLabel = "split_" + this.graph.nextSplitNumber++;
        if (scope != Scope.PERMANENT) {
            TemporarySplitterVertex tsv = new TemporarySplitterVertex(uniqueSplitLabel, x, y, originalEdge);
            tsv.setWheelchairAccessible(originalEdge.isWheelchairAccessible());
            v = tsv;
        } else {
            v = this.vertexFactory.splitter(originalEdge, x, y, uniqueSplitLabel);
        }
        v.addRentalRestriction(originalEdge.getFromVertex().rentalRestrictions());
        v.addRentalRestriction(originalEdge.getToVertex().rentalRestrictions());
        return v;
    }

    public boolean addPermanentAreaVertex(IntersectionVertex newVertex, AreaGroup areaGroup) {
        return this.addAreaVertex(newVertex, areaGroup, Scope.PERMANENT, null, true);
    }

    private double distSquared(Vertex a, Vertex b) {
        Coordinate aco = a.getCoordinate();
        Coordinate bco = b.getCoordinate();
        aco.x -= bco.x;
        aco.y -= bco.y;
        return aco.x * aco.x + aco.y * aco.y;
    }

    private boolean addAreaVertex(IntersectionVertex newVertex, AreaGroup areaGroup, Scope scope, DisposableEdgeCollection tempEdges, boolean force) {
        Geometry polygon = areaGroup.getGeometry();
        int added = 0;
        Set<IntersectionVertex> visibilityVertices = areaGroup.visibilityVertices();
        if (scope != Scope.PERMANENT) {
            int areaComplexity = polygon.getNumPoints();
            int totalCount = visibilityVertices.size();
            long appliedCount = (long)Math.max(6.0, Math.floor(2 * this.maxAreaNodes * this.maxAreaNodes / areaComplexity));
            if (appliedCount < (long)totalCount) {
                visibilityVertices = visibilityVertices.stream().sorted((v1, v2) -> Double.compare(this.distSquared((Vertex)v1, newVertex), this.distSquared((Vertex)v2, newVertex))).limit(appliedCount).collect(Collectors.toSet());
            }
        }
        for (IntersectionVertex v3 : visibilityVertices) {
            if (!this.addVisibilityEdges(newVertex, v3, areaGroup, scope, tempEdges, false)) continue;
            ++added;
        }
        if (added == 0) {
            if (force) {
                Optional<IntersectionVertex> nearest = areaGroup.visibilityVertices().stream().filter(v -> this.distSquared((Vertex)v, newVertex) >= DUPLICATE_NODE_EPSILON_DEGREES_SQUARED).sorted((v1, v2) -> Double.compare(this.distSquared((Vertex)v1, newVertex), this.distSquared((Vertex)v2, newVertex))).findFirst();
                if (!nearest.isPresent()) {
                    nearest = areaGroup.visibilityVertices().stream().findFirst();
                }
                if (nearest.isPresent()) {
                    return this.addVisibilityEdges(newVertex, nearest.get(), areaGroup, scope, tempEdges, true);
                }
            }
            return false;
        }
        if (scope == Scope.PERMANENT) {
            areaGroup.addVisibilityVertices(Set.of(newVertex));
        }
        return true;
    }

    public static Set<TraverseMode> getNoThruModes(Collection<Edge> edges) {
        HashSet<TraverseMode> modes = new HashSet<TraverseMode>(NO_THRU_MODES);
        for (Edge e : edges) {
            if (!(e instanceof StreetEdge)) continue;
            StreetEdge se = (StreetEdge)e;
            for (TraverseMode tm : NO_THRU_MODES) {
                if (se.isNoThruTraffic(tm)) continue;
                modes.remove((Object)tm);
            }
        }
        return modes;
    }

    private boolean addVisibilityEdges(IntersectionVertex from, IntersectionVertex to, AreaGroup ag, Scope scope, DisposableEdgeCollection tempEdges, boolean force) {
        if (from.isConnected(to)) {
            return true;
        }
        if (!force && this.distSquared(from, to) < DUPLICATE_NODE_EPSILON_DEGREES_SQUARED) {
            return false;
        }
        LineString line = GEOMETRY_FACTORY.createLineString(new Coordinate[]{from.getCoordinate(), to.getCoordinate()});
        if (!force && !ag.getGeometry().contains((Geometry)line)) {
            return false;
        }
        this.createEdges(line, from, to, ag, scope, tempEdges);
        return true;
    }

    private void createEdges(LineString line, IntersectionVertex from, IntersectionVertex to, AreaGroup ag, Scope scope, DisposableEdgeCollection tempEdges) {
        Area hit = null;
        List<Area> areas = ag.getAreas();
        if (areas.size() == 1) {
            hit = areas.getFirst();
        } else {
            for (Area area : areas) {
                Geometry polygon = area.getGeometry();
                Geometry intersection = polygon.intersection((Geometry)line);
                if (!(intersection.getLength() > 1.0E-6)) continue;
                hit = area;
                break;
            }
        }
        if (hit == null) {
            LOG.warn("No intersecting area found. This may indicate a bug.");
            hit = areas.getFirst();
        }
        double length = SphericalDistanceLibrary.distance(to.getCoordinate(), from.getCoordinate());
        Set<TraverseMode> incomingNoThruModes = VertexLinker.getNoThruModes(to.getIncoming());
        Set<TraverseMode> outgoingNoThruModes = VertexLinker.getNoThruModes(to.getOutgoing());
        AreaEdgeBuilder areaEdgeBuilder = ((AreaEdgeBuilder)((AreaEdgeBuilder)((AreaEdgeBuilder)((AreaEdgeBuilder)((AreaEdgeBuilder)((AreaEdgeBuilder)((AreaEdgeBuilder)((AreaEdgeBuilder)((AreaEdgeBuilder)new AreaEdgeBuilder().withFromVertex(from)).withToVertex(to)).withGeometry(line)).withName(hit.getName())).withMeterLength(length)).withPermission(hit.getPermission())).withBicycleSafetyFactor(hit.getBicycleSafety())).withWalkSafetyFactor(hit.getWalkSafety())).withBack(false)).withArea(ag);
        for (TraverseMode tm : outgoingNoThruModes) {
            areaEdgeBuilder.withNoThruTrafficTraverseMode(tm);
        }
        AreaEdge areaEdge = areaEdgeBuilder.buildAndConnect();
        if (scope != Scope.PERMANENT) {
            tempEdges.addEdge(areaEdge);
        }
        AreaEdgeBuilder reverseAreaEdgeBuilder = ((AreaEdgeBuilder)((AreaEdgeBuilder)((AreaEdgeBuilder)((AreaEdgeBuilder)((AreaEdgeBuilder)((AreaEdgeBuilder)((AreaEdgeBuilder)((AreaEdgeBuilder)((AreaEdgeBuilder)new AreaEdgeBuilder().withFromVertex(to)).withToVertex(from)).withGeometry(line.reverse())).withName(hit.getName())).withMeterLength(length)).withPermission(hit.getPermission())).withBicycleSafetyFactor(hit.getBicycleSafety())).withWalkSafetyFactor(hit.getWalkSafety())).withBack(true)).withArea(ag);
        for (TraverseMode tm : incomingNoThruModes) {
            reverseAreaEdgeBuilder.withNoThruTrafficTraverseMode(tm);
        }
        AreaEdge reverseAreaEdge = reverseAreaEdgeBuilder.buildAndConnect();
        if (scope != Scope.PERMANENT) {
            tempEdges.addEdge(reverseAreaEdge);
        }
    }

    private static class DistanceTo<T> {
        T item;
        double distanceDegreesLat;

        public DistanceTo(T item, double distanceDegreesLat) {
            this.item = item;
            this.distanceDegreesLat = distanceDegreesLat;
        }

        public int hashCode() {
            return Objects.hash(this.item);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            DistanceTo that = (DistanceTo)o;
            return Objects.equals(this.item, that.item);
        }
    }
}

