package org.opentripplanner.updater.trip.metrics;

import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.Tag;
import io.micrometer.core.instrument.Tags;
import java.util.Arrays;
import java.util.List;
import javax.annotation.Nullable;
import org.opentripplanner.updater.spi.UpdateError;
import org.opentripplanner.updater.spi.UpdateResult;
import org.opentripplanner.updater.spi.UpdateSuccess;
import org.opentripplanner.updater.trip.UrlUpdaterParameters;

/**
 * Records micrometer metrics for trip updaters that stream trip updates into the system, for
 * example GTFS-RT via MQTT.
 * <p>
 * It records the trip update as counters (continuously increasing numbers) since the concept of
 * "latest update" doesn't exist for them.
 * <p>
 * Use your metrics database to convert the counters to rates.
 */
public class StreamingTripUpdateMetrics extends TripUpdateMetrics {

  protected static final String METRICS_PREFIX = "streaming_trip_updates";
  private final boolean producerMetrics;

  public StreamingTripUpdateMetrics(UrlUpdaterParameters parameters) {
    super(parameters);
    this.producerMetrics = parameters.producerMetrics();
  }

  public void setCounters(UpdateResult result) {
    incrementWarningCounts(result);
    incrementFailureCounts(result);
    incrementSuccessCounts(result);
  }

  private void incrementWarningCounts(UpdateResult result) {
    for (var warningType : result.warnings()) {
      Tags tags = Tags.concat(baseTags, Tags.of("warningType", warningType.name()));
      Counter.builder(METRICS_PREFIX + "." + "warnings")
        .description("Total warnings by type generated by successful trip updates")
        .tags(tags)
        .register(Metrics.globalRegistry)
        .increment();
    }
  }

  private Tags andProducerMetrics(Tags tags, @Nullable String producer) {
    if (producer != null && !producer.isEmpty()) {
      return tags.and(Tag.of("producer", producer));
    }
    return tags.and(Tag.of("producer", "unknown_producer"));
  }

  private void incrementFailureCounts(UpdateResult result) {
    for (UpdateError error : result.errors()) {
      Tags tags = Tags.concat(baseTags, Tags.of("errorType", error.errorType().name()));
      if (producerMetrics) {
        tags = andProducerMetrics(tags, error.producer());
      }
      Counter.builder(METRICS_PREFIX + "." + "failed")
        .description("Total failed trip updates")
        .tags(tags)
        .register(Metrics.globalRegistry)
        .increment();
    }
  }

  private void incrementSuccessCounts(UpdateResult result) {
    for (UpdateSuccess success : result.successes()) {
      Tags tags = Tags.of(baseTags);
      if (producerMetrics) {
        tags = andProducerMetrics(tags, success.producer());
      }
      Counter.builder(METRICS_PREFIX + "." + "successful")
        .description("Total successfully applied trip updates")
        .tags(tags)
        .register(Metrics.globalRegistry)
        .increment();
    }
  }
}
