"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PonyfillRequest = void 0;
const Body_js_1 = require("./Body.js");
const Headers_js_1 = require("./Headers.js");
const utils_js_1 = require("./utils.js");
function isRequest(input) {
    return input[Symbol.toStringTag] === 'Request';
}
class PonyfillRequest extends Body_js_1.PonyfillBody {
    constructor(input, options) {
        let url;
        let bodyInit = null;
        let requestInit;
        if (typeof input === 'string') {
            url = input;
        }
        else if ('href' in input) {
            url = input.toString();
        }
        else if (isRequest(input)) {
            url = input.url;
            bodyInit = input.body;
            requestInit = input;
        }
        if (options != null) {
            bodyInit = options.body || null;
            requestInit = options;
        }
        super(bodyInit, options);
        this.cache = requestInit?.cache || 'default';
        this.credentials = requestInit?.credentials || 'same-origin';
        this.headers =
            requestInit?.headers && (0, Headers_js_1.isHeadersLike)(requestInit.headers)
                ? requestInit.headers
                : new Headers_js_1.PonyfillHeaders(requestInit?.headers);
        this.integrity = requestInit?.integrity || '';
        this.keepalive = requestInit?.keepalive != null ? requestInit?.keepalive : false;
        this.method = requestInit?.method?.toUpperCase() || 'GET';
        this.mode = requestInit?.mode || 'cors';
        this.redirect = requestInit?.redirect || 'follow';
        this.referrer = requestInit?.referrer || 'about:client';
        this.referrerPolicy = requestInit?.referrerPolicy || 'no-referrer';
        this._signal = requestInit?.signal;
        this.headersSerializer = requestInit?.headersSerializer || utils_js_1.getHeadersObj;
        this.agent = requestInit?.agent;
        this.url = url || '';
        this.destination = 'document';
        this.priority = 'auto';
        if (this.method !== 'GET' && this.method !== 'HEAD') {
            const contentTypeInHeaders = this.headers.get('content-type');
            if (!contentTypeInHeaders) {
                if (this.contentType) {
                    this.headers.set('content-type', this.contentType);
                }
            }
            else {
                this.contentType = contentTypeInHeaders;
            }
            const contentLengthInHeaders = this.headers.get('content-length');
            if (!contentLengthInHeaders) {
                if (this.contentLength) {
                    this.headers.set('content-length', this.contentLength.toString());
                }
            }
            else {
                this.contentLength = parseInt(contentLengthInHeaders, 10);
            }
        }
    }
    get signal() {
        // Create a new signal only if needed
        // Because the creation of signal is expensive
        if (!this._signal) {
            this._signal = new AbortController().signal;
        }
        return this._signal;
    }
    clone() {
        return new PonyfillRequest(this);
    }
}
exports.PonyfillRequest = PonyfillRequest;
