import { GraphQLObjectType, GraphQLSchema, Kind, visit } from 'graphql';
import { MapperKind, getDefinedRootType, getRootTypeMap, mapSchema, } from '@graphql-tools/utils';
const defaultRootTypeNames = {
    query: 'Query',
    mutation: 'Mutation',
    subscription: 'Subscription',
};
export class MoveRootField {
    constructor(from) {
        this.from = from;
        this.to = {
            query: {},
            mutation: {},
            subscription: {},
        };
        for (const operation in this.from) {
            const removedFields = this.from[operation];
            for (const fieldName in removedFields) {
                const newOperation = removedFields[fieldName];
                this.to[newOperation][fieldName] = operation;
            }
        }
    }
    transformSchema(schema, _subschemaConfig) {
        const rootTypeMap = getRootTypeMap(schema);
        const newRootFieldsMap = {
            query: rootTypeMap.get('query')?.toConfig()?.fields || {},
            mutation: rootTypeMap.get('mutation')?.toConfig()?.fields || {},
            subscription: rootTypeMap.get('subscription')?.toConfig()?.fields || {},
        };
        for (const operation in this.from) {
            const removedFields = this.from[operation];
            for (const fieldName in removedFields) {
                const fieldConfig = newRootFieldsMap[operation][fieldName];
                delete newRootFieldsMap[operation]?.[fieldName];
                const newOperation = removedFields[fieldName];
                newRootFieldsMap[newOperation][fieldName] = fieldConfig;
            }
        }
        const schemaConfig = schema.toConfig();
        for (const rootType in newRootFieldsMap) {
            const newRootFields = newRootFieldsMap[rootType];
            if (!schemaConfig[rootType] && Object.keys(newRootFields).length > 0) {
                schemaConfig[rootType] = new GraphQLObjectType({
                    name: defaultRootTypeNames[rootType],
                    fields: newRootFields,
                });
            }
        }
        return mapSchema(new GraphQLSchema(schemaConfig), {
            [MapperKind.QUERY]: type => {
                const queryConfig = type.toConfig();
                queryConfig.fields = newRootFieldsMap.query;
                return new GraphQLObjectType(queryConfig);
            },
            [MapperKind.MUTATION]: type => {
                const mutationConfig = type.toConfig();
                mutationConfig.fields = newRootFieldsMap.mutation;
                return new GraphQLObjectType(mutationConfig);
            },
            [MapperKind.SUBSCRIPTION]: type => {
                const subscriptionConfig = type.toConfig();
                subscriptionConfig.fields = newRootFieldsMap.subscription;
                return new GraphQLObjectType(subscriptionConfig);
            },
        });
    }
    transformRequest(originalRequest, delegationContext) {
        const newOperation = this.to[delegationContext.operation][delegationContext.fieldName];
        if (newOperation && newOperation !== delegationContext.operation) {
            return {
                ...originalRequest,
                document: visit(originalRequest.document, {
                    [Kind.OPERATION_DEFINITION]: node => {
                        return {
                            ...node,
                            operation: newOperation,
                        };
                    },
                }),
            };
        }
        return originalRequest;
    }
    transformResult(result, delegationContext) {
        if (result.data?.__typename) {
            const newOperation = this.to[delegationContext.operation][delegationContext.fieldName];
            if (newOperation && newOperation !== delegationContext.operation) {
                result.data.__typename = getDefinedRootType(delegationContext.targetSchema, newOperation)?.name;
            }
        }
        return result;
    }
}
