"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MoveRootField = void 0;
const graphql_1 = require("graphql");
const utils_1 = require("@graphql-tools/utils");
const defaultRootTypeNames = {
    query: 'Query',
    mutation: 'Mutation',
    subscription: 'Subscription',
};
class MoveRootField {
    constructor(from) {
        this.from = from;
        this.to = {
            query: {},
            mutation: {},
            subscription: {},
        };
        for (const operation in this.from) {
            const removedFields = this.from[operation];
            for (const fieldName in removedFields) {
                const newOperation = removedFields[fieldName];
                this.to[newOperation][fieldName] = operation;
            }
        }
    }
    transformSchema(schema, _subschemaConfig) {
        const rootTypeMap = (0, utils_1.getRootTypeMap)(schema);
        const newRootFieldsMap = {
            query: rootTypeMap.get('query')?.toConfig()?.fields || {},
            mutation: rootTypeMap.get('mutation')?.toConfig()?.fields || {},
            subscription: rootTypeMap.get('subscription')?.toConfig()?.fields || {},
        };
        for (const operation in this.from) {
            const removedFields = this.from[operation];
            for (const fieldName in removedFields) {
                const fieldConfig = newRootFieldsMap[operation][fieldName];
                delete newRootFieldsMap[operation]?.[fieldName];
                const newOperation = removedFields[fieldName];
                newRootFieldsMap[newOperation][fieldName] = fieldConfig;
            }
        }
        const schemaConfig = schema.toConfig();
        for (const rootType in newRootFieldsMap) {
            const newRootFields = newRootFieldsMap[rootType];
            if (!schemaConfig[rootType] && Object.keys(newRootFields).length > 0) {
                schemaConfig[rootType] = new graphql_1.GraphQLObjectType({
                    name: defaultRootTypeNames[rootType],
                    fields: newRootFields,
                });
            }
        }
        return (0, utils_1.mapSchema)(new graphql_1.GraphQLSchema(schemaConfig), {
            [utils_1.MapperKind.QUERY]: type => {
                const queryConfig = type.toConfig();
                queryConfig.fields = newRootFieldsMap.query;
                return new graphql_1.GraphQLObjectType(queryConfig);
            },
            [utils_1.MapperKind.MUTATION]: type => {
                const mutationConfig = type.toConfig();
                mutationConfig.fields = newRootFieldsMap.mutation;
                return new graphql_1.GraphQLObjectType(mutationConfig);
            },
            [utils_1.MapperKind.SUBSCRIPTION]: type => {
                const subscriptionConfig = type.toConfig();
                subscriptionConfig.fields = newRootFieldsMap.subscription;
                return new graphql_1.GraphQLObjectType(subscriptionConfig);
            },
        });
    }
    transformRequest(originalRequest, delegationContext) {
        const newOperation = this.to[delegationContext.operation][delegationContext.fieldName];
        if (newOperation && newOperation !== delegationContext.operation) {
            return {
                ...originalRequest,
                document: (0, graphql_1.visit)(originalRequest.document, {
                    [graphql_1.Kind.OPERATION_DEFINITION]: node => {
                        return {
                            ...node,
                            operation: newOperation,
                        };
                    },
                }),
            };
        }
        return originalRequest;
    }
    transformResult(result, delegationContext) {
        if (result.data?.__typename) {
            const newOperation = this.to[delegationContext.operation][delegationContext.fieldName];
            if (newOperation && newOperation !== delegationContext.operation) {
                result.data.__typename = (0, utils_1.getDefinedRootType)(delegationContext.targetSchema, newOperation)?.name;
            }
        }
        return result;
    }
}
exports.MoveRootField = MoveRootField;
