/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.atlas.generator.tools.spark.utilities;

import java.io.BufferedInputStream;
import java.io.BufferedWriter;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.io.Serializable;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;
import org.apache.commons.lang.ArrayUtils;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.openstreetmap.atlas.exception.CoreException;
import org.openstreetmap.atlas.generator.tools.filesystem.FileSystemCreator;
import org.openstreetmap.atlas.generator.tools.filesystem.FileSystemHelper;
import org.openstreetmap.atlas.generator.tools.spark.utilities.CountrySpecificAtlasFilePathFilter;
import org.openstreetmap.atlas.generator.tools.spark.utilities.SparkFileOutput;
import org.openstreetmap.atlas.generator.tools.spark.utilities.SparkFilePath;
import org.openstreetmap.atlas.streaming.resource.InputStreamResource;
import org.openstreetmap.atlas.streaming.resource.Resource;
import org.openstreetmap.atlas.streaming.resource.WritableResource;
import org.openstreetmap.atlas.streaming.resource.http.GetResource;
import org.openstreetmap.atlas.utilities.collections.Iterables;
import org.openstreetmap.atlas.utilities.collections.StringList;
import org.openstreetmap.atlas.utilities.runtime.Retry;
import org.openstreetmap.atlas.utilities.scalars.Duration;
import org.openstreetmap.atlas.utilities.threads.Pool;
import org.openstreetmap.atlas.utilities.time.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SparkFileHelper
implements Serializable {
    public static final String TEMPORARY_FOLDER_NAME = "_temp";
    public static final String EMPTY_STRING = "";
    public static final String DIRECTORY_SEPARATOR = "/";
    public static final String EXTENSION_SEPARATOR = ".";
    private static final long serialVersionUID = -5716285735225965942L;
    private static final Logger logger = LoggerFactory.getLogger(SparkFileHelper.class);
    private static final Duration MAX_DURATION_FOR_IO = Duration.hours((double)3.0);
    private static final int IO_RETRY_COUNT = 5;
    private static final Duration WAIT_DURATION_BEFORE_IO_RETRY = Duration.seconds((double)5.0);
    private static Retry IO_RETRY = new Retry(5, WAIT_DURATION_BEFORE_IO_RETRY);
    private final Map<String, String> sparkContext;

    public static String combine(String basePath, String ... paths) {
        StringBuilder builder = new StringBuilder(SparkFileHelper.pathNotEndingWithSeparator(basePath));
        for (String path : paths) {
            if (path.isEmpty()) continue;
            builder.append(DIRECTORY_SEPARATOR);
            builder.append(SparkFileHelper.pathNotStartingOrEndingWithSeparator(path));
        }
        return builder.toString();
    }

    public static String extensionStartingWithSeparator(String path) {
        return SparkFileHelper.pathStartingWithSeparator(path, EXTENSION_SEPARATOR);
    }

    public static String parentPath(String path) {
        if (path == null) {
            logger.warn("Null path. Returning empty path.");
            return EMPTY_STRING;
        }
        int lastSeparatorIndex = path.lastIndexOf(DIRECTORY_SEPARATOR);
        if (lastSeparatorIndex > 0) {
            return path.substring(0, lastSeparatorIndex);
        }
        logger.debug("Given path doesn't have a parent path. Returning the path as is.");
        return path;
    }

    public static String pathNotEndingWithSeparator(String path) {
        return SparkFileHelper.pathNotEndingWithSeparator(path, DIRECTORY_SEPARATOR);
    }

    public static String pathNotStartingOrEndingWithSeparator(String path) {
        return SparkFileHelper.pathNotStartingOrEndingWithSeparator(path, DIRECTORY_SEPARATOR);
    }

    public static String pathNotStartingWithSeparator(String path) {
        return SparkFileHelper.pathNotStartingWithSeparator(path, DIRECTORY_SEPARATOR);
    }

    private static String pathNotEndingWithSeparator(String path, String separator) {
        int lastSeparatorIndex;
        if (path == null) {
            logger.warn("Null path. Returning empty path.");
            return EMPTY_STRING;
        }
        if (separator == null) {
            logger.warn("Null separator. Returning empty path.");
            return EMPTY_STRING;
        }
        String scrubbedPath = path;
        while (scrubbedPath.lastIndexOf(separator) != -1 && (lastSeparatorIndex = scrubbedPath.lastIndexOf(separator)) == scrubbedPath.length() - 1) {
            scrubbedPath = scrubbedPath.substring(0, lastSeparatorIndex);
        }
        return scrubbedPath;
    }

    private static String pathNotStartingOrEndingWithSeparator(String path, String directorySeparator) {
        String newPath = SparkFileHelper.pathNotStartingWithSeparator(path, directorySeparator);
        newPath = SparkFileHelper.pathNotEndingWithSeparator(newPath, directorySeparator);
        return newPath;
    }

    private static String pathNotStartingWithSeparator(String path, String separator) {
        int firstSeparatorIndex;
        if (path == null) {
            logger.warn("Null path. Returning empty path.");
            return EMPTY_STRING;
        }
        if (separator == null) {
            logger.warn("Null separator. Returning empty path.");
            return EMPTY_STRING;
        }
        String scrubbedPath = path;
        while (scrubbedPath.indexOf(separator) == 0 && (firstSeparatorIndex = scrubbedPath.indexOf(separator)) == 0) {
            scrubbedPath = scrubbedPath.substring(1);
        }
        return scrubbedPath;
    }

    private static String pathStartingWithSeparator(String path, String separator) {
        if (path == null) {
            logger.warn("Null path. Returning empty path.");
            return EMPTY_STRING;
        }
        if (separator == null) {
            logger.warn("Null separator. Returning empty path.");
            return EMPTY_STRING;
        }
        int firstSeparatorIndex = path.indexOf(separator);
        return firstSeparatorIndex != 0 ? separator + path : path;
    }

    public SparkFileHelper(Map<String, String> sparkContext) {
        this.sparkContext = sparkContext;
    }

    public List<Resource> collectAtlasFiles(String directory, String country, boolean recursive) {
        CountrySpecificAtlasFilePathFilter filter = new CountrySpecificAtlasFilePathFilter(country);
        return recursive ? FileSystemHelper.listResourcesRecursively(directory, this.sparkContext, filter) : FileSystemHelper.resources(directory, this.sparkContext, filter);
    }

    public Optional<Resource> collectSourceFile(String uri, PathFilter ... filters) {
        Path path = new Path(uri);
        Resource resource = null;
        if (ArrayUtils.isEmpty((Object[])filters) || Stream.of(filters).anyMatch(filter -> filter.accept(path))) {
            String schema = URI.create(uri).getScheme();
            if ("http".equals(schema) || "https".equals(schema)) {
                logger.info("Downloading {}", (Object)uri);
                resource = new InputStreamResource((InputStream)new BufferedInputStream(new GetResource(uri).read())).withName(uri);
            }
            if (resource == null) {
                logger.info("Loading {}", (Object)uri);
                resource = FileSystemHelper.resource(uri, this.sparkContext);
            }
        }
        return Optional.ofNullable(resource);
    }

    public List<Resource> collectSourceFiles(String directory, boolean recursive, PathFilter ... filters) {
        return Iterables.stream((Iterable)Iterables.from((Object[])filters)).flatMap(filter -> recursive ? FileSystemHelper.listResourcesRecursively(directory, this.sparkContext, filter) : FileSystemHelper.resources(directory, this.sparkContext, filter)).collectToList();
    }

    public void commit(SparkFilePath path) {
        try {
            if (this.isDirectory(path.getTemporaryPath())) {
                logger.debug("Path {} is a directory. Renaming all the files under.", (Object)path);
                if (!this.exists(path.getTargetPath())) {
                    logger.debug("Creating {}.", (Object)path.getTargetPath());
                    this.mkdir(path.getTargetPath());
                }
                this.list(path.getTemporaryPath()).forEach(resource -> {
                    logger.debug("Renaming {} in {} into {}.", new Object[]{resource.getName(), path.getTemporaryPath(), path.getTargetPath()});
                    this.rename(SparkFileHelper.combine(path.getTemporaryPath(), resource.getName()), SparkFileHelper.combine(path.getTargetPath(), resource.getName()));
                });
            } else {
                logger.debug("Renaming {} to {}.", (Object)path.getTemporaryPath(), (Object)path.getTargetPath());
                this.rename(path.getTemporaryPath(), path.getTargetPath());
            }
        }
        catch (Exception e) {
            logger.warn("Renaming {} failed!", (Object)path, (Object)e);
        }
    }

    public void commitByCopy(SparkFilePath path) {
        try {
            if (this.isDirectory(path.getTemporaryPath())) {
                logger.debug("Path {} is a directory. Copying all the files under.", (Object)path);
                if (!this.exists(path.getTargetPath())) {
                    logger.debug("Creating {}.", (Object)path.getTargetPath());
                    this.mkdir(path.getTargetPath());
                }
                this.list(path.getTemporaryPath()).forEach(resource -> {
                    logger.debug("Copying {} in {} into {}.", new Object[]{resource.getName(), path.getTemporaryPath(), path.getTargetPath()});
                    this.copyFile((Resource)resource, SparkFileHelper.combine(path.getTargetPath(), resource.getName()));
                });
            } else {
                logger.debug("Copying {} to {}.", (Object)path.getTemporaryPath(), (Object)path.getTargetPath());
                this.copyFile(FileSystemHelper.resource(path.getTemporaryPath(), this.sparkContext), path.getTargetPath());
            }
        }
        catch (Exception e) {
            logger.warn("Copying {} failed!", (Object)path, (Object)e);
        }
    }

    public void deleteDirectory(String path) {
        if (!FileSystemHelper.delete(path, true, this.sparkContext)) {
            throw new CoreException("Delete directory for {} is failed.", new Object[]{path});
        }
    }

    public boolean directoryContainsExpectedCountryAtlases(String directory, StringList expectedCountries, boolean recursive) {
        for (String country : expectedCountries) {
            List<Resource> atlases = this.collectAtlasFiles(directory, country, recursive);
            if (!atlases.isEmpty()) continue;
            logger.error("Missing Atlas files for {}!", (Object)country);
            return false;
        }
        return true;
    }

    public boolean exists(String path) {
        try {
            FileSystem fileSystem = new FileSystemCreator().get(path, this.sparkContext);
            return fileSystem.exists(new Path(path));
        }
        catch (Exception e) {
            throw new CoreException("Unable to check if given path {} is a directory or not.", new Object[]{path, e});
        }
    }

    public boolean isDirectory(String path) {
        try {
            FileSystem fileSystem = new FileSystemCreator().get(path, this.sparkContext);
            return fileSystem.isDirectory(new Path(path));
        }
        catch (Exception e) {
            throw new CoreException("Unable to check if given path {} is a directory or not.", new Object[]{path, e});
        }
    }

    public List<Resource> list(String path) {
        return FileSystemHelper.resources(path, this.sparkContext);
    }

    public void mkdir(String path) {
        if (!FileSystemHelper.mkdir(path, this.sparkContext)) {
            throw new CoreException("Create directory for {} is failed.", new Object[]{path});
        }
    }

    public void rename(String sourcePath, String destinationPath) {
        if (!FileSystemHelper.rename(sourcePath, destinationPath, this.sparkContext)) {
            throw new CoreException("Rename from {} to {} is failed.", new Object[]{sourcePath, destinationPath});
        }
    }

    public void save(List<SparkFileOutput> outputs) {
        try (Pool writePool = new Pool(outputs.size(), "I/O pool", MAX_DURATION_FOR_IO);){
            for (SparkFileOutput output : outputs) {
                writePool.queue(() -> IO_RETRY.run(() -> {
                    logger.debug("Writing {}: {}.", (Object)output.getOperationName(), (Object)output.getPath().getTemporaryPath());
                    Time timer = Time.now();
                    output.getSaveFunction().accept(FileSystemHelper.writableResource(output.getPath().getTemporaryPath(), this.sparkContext));
                    logger.debug("{} write took {} ms.", (Object)output.getOperationName(), (Object)timer.elapsedSince().asMilliseconds());
                }));
            }
        }
        catch (Exception e) {
            logger.error("Failed save files.", (Throwable)e);
        }
    }

    public void save(SparkFileOutput ... outputs) {
        this.save(Arrays.asList(outputs));
    }

    public void write(String directory, String filename, byte[] content) {
        IO_RETRY.run(() -> {
            try {
                FileSystemHelper.writableResource(SparkFileHelper.combine(directory, filename), this.sparkContext).writeAndClose(content);
            }
            catch (Exception e) {
                throw new CoreException(String.format("Could not save into %s.", filename), (Throwable)e);
            }
        });
    }

    public void write(String directory, String filename, String content) {
        IO_RETRY.run(() -> {
            WritableResource resource = FileSystemHelper.writableResource(SparkFileHelper.combine(directory, filename), this.sparkContext);
            try (BufferedWriter out = new BufferedWriter(new OutputStreamWriter(resource.write(), StandardCharsets.UTF_8));){
                out.write(content);
            }
            catch (Exception e) {
                throw new CoreException(String.format("Could not save into %s.", resource.getName()), (Throwable)e);
            }
        });
    }

    private void copyFile(Resource resource, String targetPath) {
        WritableResource output = FileSystemHelper.writableResource(targetPath, this.sparkContext);
        resource.copyTo(output);
    }
}

