/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.atlas.checks.utility;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Stream;
import javax.annotation.Nonnull;
import org.apache.commons.lang3.tuple.Pair;
import org.openstreetmap.atlas.checks.base.ExternalDataFetcher;
import org.openstreetmap.atlas.checks.utility.CompressionUtilities;
import org.openstreetmap.atlas.geography.Location;
import org.openstreetmap.atlas.geography.Longitude;
import org.openstreetmap.atlas.streaming.resource.Resource;
import org.openstreetmap.atlas.utilities.configuration.Configuration;
import org.openstreetmap.atlas.utilities.scalars.Distance;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ElevationUtilities
implements Serializable {
    private static final long serialVersionUID = -5929570973909280629L;
    private static final String SRTM_EXTENSION = "hgt";
    private static final Logger logger = LoggerFactory.getLogger(ElevationUtilities.class);
    private static final double SRTM_EXTENT = 1.0;
    public static final short NO_ELEVATION = Short.MIN_VALUE;
    private static final short[][] EMPTY_MAP = new short[0][];
    private static final int DECIMAL_TO_PERCENTAGE = 100;
    private static final String[] POSSIBLE_COMPRESSED_EXTS = new String[]{"", ".zip", ".gz", ".xz", ".bz", ".bz2", ".tar"};
    private final Map<Pair<Integer, Integer>, short[][]> loadedSrtm = new HashMap<Pair<Integer, Integer>, short[][]>();
    private final double srtmExtent;
    private final String srtmExtension;
    private final String srtmPath;
    private ExternalDataFetcher fileFetcher;

    private static String configurationKey(String key) {
        return ElevationUtilities.formatKey(ElevationUtilities.class.getSimpleName(), key);
    }

    private static <U, V> V configurationValue(Configuration configuration, String key, U defaultValue, Function<U, V> transform) {
        return (V)configuration.get(ElevationUtilities.configurationKey(key), defaultValue, transform).value();
    }

    private static String formatKey(String name, String key) {
        return String.format("%s.%s", name, key);
    }

    public ElevationUtilities(Configuration configuration, ExternalDataFetcher fileFetcher) {
        this((Double)ElevationUtilities.configurationValue(configuration, "elevation.srtm_extent", 1.0, Function.identity()), (String)ElevationUtilities.configurationValue(configuration, "elevation.srtm_ext", SRTM_EXTENSION, Function.identity()), (String)ElevationUtilities.configurationValue(configuration, "elevation.path", "extra/elevation", Function.identity()), fileFetcher);
    }

    public ElevationUtilities(double srtmExtent, String srtmExtension, String srtmPath, ExternalDataFetcher fileFetcher) {
        this.srtmExtension = srtmExtension;
        this.srtmExtent = srtmExtent;
        this.srtmPath = srtmPath.endsWith(File.separator) ? srtmPath : srtmPath.concat(File.separator);
        this.fileFetcher = fileFetcher;
    }

    public short getElevation(Location location) {
        short[][] map = this.getMap(location);
        if (Arrays.equals((Object[])EMPTY_MAP, (Object[])map)) {
            return Short.MIN_VALUE;
        }
        int[] index = this.getIndex(location, map.length);
        return map[index[0]][index[1]];
    }

    public double getIncline(Location start, Location end) {
        short startElevation = this.getElevation(start);
        short endElevation = this.getElevation(end);
        if (startElevation == Short.MIN_VALUE || endElevation == Short.MIN_VALUE) {
            return Double.NaN;
        }
        double distance = end.distanceTo(start).asMeters();
        return (double)(endElevation - startElevation) / distance * 100.0;
    }

    public int[] getIndex(Location location, int mapSize) {
        double latDegrees = location.getLatitude().asDegrees();
        double lonDegrees = location.getLongitude().asDegrees();
        float fraction = (float)this.srtmExtent / (float)mapSize;
        int latitude = (int)Math.floor(Math.abs(latDegrees - (double)((int)latDegrees)) / (double)fraction);
        int longitude = (int)Math.floor(Math.abs(lonDegrees - (double)((int)lonDegrees)) / (double)fraction);
        if (latDegrees >= 0.0) {
            latitude = mapSize - 1 - latitude;
        }
        if (lonDegrees < 0.0) {
            longitude = mapSize - 1 - longitude;
        }
        return new int[]{latitude, longitude};
    }

    public short[][] getMap(Location location) {
        double latDegrees = location.getLatitude().asDegrees();
        double lonDegrees = location.getLongitude().asDegrees();
        int lat = (int)Math.floor(latDegrees);
        int lon = (int)Math.floor(lonDegrees);
        return this.loadedSrtm.computeIfAbsent((Pair<Integer, Integer>)Pair.of((Object)lat, (Object)lon), pair -> this.loadMap((Integer)pair.getLeft(), (Integer)pair.getRight()));
    }

    public Distance getResolution(Location location) {
        short[][] map = this.getMap(location);
        if (Arrays.equals((Object[])EMPTY_MAP, (Object[])map)) {
            return Distance.MAXIMUM;
        }
        float difference = (float)this.srtmExtent / (float)map.length;
        Location temp = new Location(location.getLatitude(), Longitude.degrees((double)(location.getLongitude().asDegrees() + (double)difference)));
        return temp.distanceTo(location);
    }

    public String getSrtmFileName(int latitude, int longitude) {
        int lat = latitude;
        int lon = longitude;
        String latPrefix = "N";
        if (lat < 0) {
            lat = Math.abs(lat);
            latPrefix = "S";
        }
        String lonPrefix = "E";
        if (lon < 0) {
            lon = Math.abs(lon);
            lonPrefix = "W";
        }
        return String.format("%s%02d%s%03d.%s", latPrefix, lat, lonPrefix, lon, this.srtmExtension);
    }

    public boolean inSameDataPoint(Location one, Location two) {
        int[] indexTwo;
        int[] indexOne;
        short[][] mapTwo;
        short[][] mapOne = this.getMap(one);
        return Arrays.equals((Object[])mapOne, (Object[])(mapTwo = this.getMap(two))) && !Arrays.equals((Object[])EMPTY_MAP, (Object[])mapOne) && Arrays.equals(indexOne = this.getIndex(one, mapOne.length), indexTwo = this.getIndex(two, mapTwo.length));
    }

    public void putMap(Location location, short[][] map) {
        double latDegrees = location.getLatitude().asDegrees();
        double lonDegrees = location.getLongitude().asDegrees();
        int lat = (int)Math.floor(latDegrees);
        int lon = (int)Math.floor(lonDegrees);
        this.loadedSrtm.put((Pair<Integer, Integer>)Pair.of((Object)lat, (Object)lon), map);
    }

    /*
     * Enabled aggressive exception aggregation
     */
    @Nonnull
    private synchronized short[][] loadMap(int lat, int lon) {
        if (this.fileFetcher == null) {
            logger.error("Cannot load maps -- fileFetcher is not initialized or is null");
            return EMPTY_MAP;
        }
        String filename = this.getSrtmFileName(lat, lon);
        Optional<Resource> path = Stream.of(POSSIBLE_COMPRESSED_EXTS).map(ext -> Paths.get(this.srtmPath, filename.concat((String)ext))).map(Object::toString).map(this.fileFetcher::apply).filter(Optional::isPresent).map(Optional::get).findFirst();
        if (path.isEmpty()) {
            return EMPTY_MAP;
        }
        try (InputStream temporaryInputStream = path.get().read();){
            short[][] sArray;
            block16: {
                InputStream inputStream = CompressionUtilities.getUncompressedInputStream(temporaryInputStream);
                try {
                    sArray = this.readStream(inputStream);
                    if (inputStream == null) break block16;
                }
                catch (Throwable throwable) {
                    if (inputStream != null) {
                        try {
                            inputStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                inputStream.close();
            }
            return sArray;
        }
        catch (IOException e) {
            return EMPTY_MAP;
        }
    }

    private short[][] readStream(InputStream inputStream) throws IOException {
        int squareSize = 1;
        short[] data = new short[(int)Math.pow(squareSize, 2.0)];
        int index = 0;
        ByteBuffer byteBuffer = ByteBuffer.wrap(inputStream.readAllBytes());
        byteBuffer.order(ByteOrder.BIG_ENDIAN);
        while (byteBuffer.hasRemaining()) {
            data[index] = byteBuffer.getShort();
            if (++index < data.length || !byteBuffer.hasRemaining()) continue;
            data = Arrays.copyOf(data, (int)Math.pow(++squareSize, 2.0));
        }
        short[][] realData = new short[squareSize][squareSize];
        for (int latitude = 0; latitude < squareSize; ++latitude) {
            for (int longitude = 0; longitude < squareSize; ++longitude) {
                realData[latitude][longitude] = data[latitude * squareSize + longitude];
            }
        }
        return realData;
    }
}

