package org.openstreetmap.atlas.checks.utility;

import java.io.Serializable;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.text.MessageFormat;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import org.openstreetmap.atlas.checks.base.ExternalDataFetcher;
import org.openstreetmap.atlas.generator.tools.filesystem.FileSystemHelper;
import org.openstreetmap.atlas.streaming.resource.File;
import org.openstreetmap.atlas.streaming.resource.Resource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sparkproject.guava.cache.CacheBuilder;
import org.sparkproject.guava.cache.CacheLoader;
import org.sparkproject.guava.cache.LoadingCache;

/**
 * This utility class is designed to take an SQLite database, and manage connections for you. It is
 * designed to be serializable, such that when it is deserialized/unserialized, everything just
 * works (tm).
 *
 * @author Taylor Smock
 */
public class SQLiteUtils implements Serializable
{
    /** Autogenerated UID */
    private static final long serialVersionUID = 3796921068577020645L;
    private final String database;
    private final String defaultTable;
    private final ExternalDataFetcher fileFetcher;
    private transient LoadingCache<String, Map<String, Object>> cachedMap;
    private transient Connection connection;
    /** The db "file" for in memory databases */
    public static final String MEMORY_DB = ":memory:";
    private static Logger logger = LoggerFactory.getLogger(SQLiteUtils.class);
    private static final int DEFAULT_CACHE_MAX = 1000;

    /**
     * Check that a database exists
     *
     * @param database
     *            The location of the database (may be {@code ":memory:"})
     * @return {@code true} if the database file exists
     */
    public static boolean isValidDatabase(final String database)
    {
        return database != null && (database.startsWith(MEMORY_DB)
                || (!database.isBlank() && (FileSystemHelper.exists(database))));
    }

    private static String sqlSanitize(final String string, final char quote)
    {
        final int baseLength = string.length();
        final var stringBuilder = new StringBuilder(baseLength);
        for (var index = 0; index < baseLength; index++)
        {
            final var current = string.charAt(index);
            if (quote == current)
            {
                stringBuilder.append(quote);
            }
            stringBuilder.append(current);
        }
        return stringBuilder.toString();
    }

    /**
     * Initialize a database
     *
     * @param fileFetcher
     *            The file fetcher to use to get files
     * @param database
     *            A database to use (should be checked first with {@link #isValidDatabase})
     * @param table
     *            The default table to use, if a passed table is null or blank
     */
    public SQLiteUtils(final ExternalDataFetcher fileFetcher, final String database,
            final String table)
    {
        this.fileFetcher = fileFetcher;
        this.database = database;
        this.defaultTable = table;
    }

    /**
     * Clear cached data
     */
    public void clear()
    {
        if (this.cachedMap != null)
        {
            this.cachedMap.cleanUp();
            this.cachedMap = null;
        }
    }

    /**
     * Get the file for the database
     *
     * @return The path for the database
     */
    public String getFile()
    {

        if (this.database != null && this.database.startsWith(SQLiteUtils.MEMORY_DB))
        {
            return this.database;
        }
        else
        {
            final Optional<Resource> wikiDataResource = this.fileFetcher.apply(this.database);
            if (wikiDataResource.isPresent() && wikiDataResource.get() instanceof File)
            {
                return ((File) wikiDataResource.get()).getPathString();
            }
        }
        return null;
    }

    /**
     * Get row(s) from the table
     *
     * @param values
     *            The values to filter by (column name, value )
     * @return The rows that match for the query ({@code SELECT * FROM {table} WHERE {key1}={value1}
     *         AND ...})
     */
    public Map<String, Object> getRows(final Map<String, String> values)
    {
        final StringBuilder statement = new StringBuilder("SELECT * FROM ")
                .append(this.defaultTable);
        if (!values.isEmpty())
        {
            statement.append(" WHERE ");
        }
        final var and = " AND ";
        for (final Map.Entry<String, String> entry : values.entrySet())
        {
            statement.append(entry.getKey()).append(" = '")
                    .append(sqlSanitize(entry.getValue(), '\'')).append('\'').append(and);
        }

        final var builtStatement = statement.substring(0,
                statement.length() > and.length() ? statement.length() - and.length()
                        : statement.length());
        if (this.cachedMap == null)
        {
            this.cachedMap = CacheBuilder.newBuilder().maximumSize(DEFAULT_CACHE_MAX)
                    .expireAfterAccess(1, TimeUnit.MINUTES)
                    .build(new CacheLoader<String, Map<String, Object>>()
                    {
                        @Override
                        public Map<String, Object> load(final String statement) throws Exception
                        {
                            return SQLiteUtils.this.getRowsReal(statement);
                        }
                    });
        }
        try
        {
            return this.cachedMap.get(builtStatement);
        }
        catch (final ExecutionException e)
        {
            logger.error("Failed to get row for " + statement, e);
            return Collections.emptyMap();
        }
    }

    private Connection getConnection() throws SQLException
    {
        if (this.connection == null)
        {
            final var properties = new Properties();
            // 1 == readonly, see {@link https://www.sqlite.org/c3ref/c_open_autoproxy.html}
            properties.setProperty("open_mode", "1");
            this.connection = DriverManager.getConnection(
                    MessageFormat.format("jdbc:sqlite:{0}", this.getFile()), properties);
            // Enable some read only optimizations
            this.connection.setReadOnly(true);
        }
        return this.connection;
    }

    /**
     * Get row(s) from the table
     *
     * @param statement
     *            The statement to get the rows
     * @return The rows that match for the query ({@code SELECT * FROM {table} WHERE {key1}={value1}
     *         AND ...})
     */
    private Map<String, Object> getRowsReal(final String statement)
    {
        // SQL JDBC
        try
        {
            final var currentConnection = this.getConnection();
            final Map<String, Object> queryResultMap = new HashMap<>();
            try (var databaseStatement = currentConnection.createStatement();
                    var results = databaseStatement.executeQuery(statement))
            {
                if (results.next())
                {
                    final ResultSetMetaData metaData = results.getMetaData();
                    for (var index = 1; index <= metaData.getColumnCount(); index++)
                    {
                        queryResultMap.put(metaData.getColumnLabel(index),
                                results.getObject(index));
                    }
                }
            }
            queryResultMap.entrySet().removeIf(e -> e.getValue() == null);
            return queryResultMap;
        }
        catch (final SQLException e)
        {
            // Continue on?
            logger.error("Bad connection", e);
        }
        return Collections.emptyMap();
    }
}
