/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.atlas.utilities.command.subcommands;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.io.ParseException;
import org.locationtech.jts.io.WKTReader;
import org.openstreetmap.atlas.geography.Location;
import org.openstreetmap.atlas.geography.PolyLine;
import org.openstreetmap.atlas.geography.Polygon;
import org.openstreetmap.atlas.geography.boundary.CountryBoundary;
import org.openstreetmap.atlas.geography.boundary.CountryBoundaryMap;
import org.openstreetmap.atlas.geography.converters.jts.JtsPointConverter;
import org.openstreetmap.atlas.geography.converters.jts.JtsPolyLineConverter;
import org.openstreetmap.atlas.geography.converters.jts.JtsPolygonConverter;
import org.openstreetmap.atlas.geography.sharding.Shard;
import org.openstreetmap.atlas.geography.sharding.Sharding;
import org.openstreetmap.atlas.geography.sharding.converters.StringToShardConverter;
import org.openstreetmap.atlas.streaming.resource.File;
import org.openstreetmap.atlas.streaming.resource.StringResource;
import org.openstreetmap.atlas.utilities.command.AtlasShellToolsException;
import org.openstreetmap.atlas.utilities.command.abstractcommand.AbstractAtlasShellToolsCommand;
import org.openstreetmap.atlas.utilities.command.abstractcommand.CommandOutputDelegate;
import org.openstreetmap.atlas.utilities.command.abstractcommand.OptionAndArgumentDelegate;
import org.openstreetmap.atlas.utilities.command.parsing.ArgumentArity;
import org.openstreetmap.atlas.utilities.command.parsing.ArgumentOptionality;
import org.openstreetmap.atlas.utilities.command.parsing.OptionOptionality;
import org.openstreetmap.atlas.utilities.command.terminal.TTYAttribute;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WKTShardCommand
extends AbstractAtlasShellToolsCommand {
    private static final Logger logger = LoggerFactory.getLogger(WKTShardCommand.class);
    private static final String TREE_OPTION_LONG = "tree";
    private static final String TREE_OPTION_DESCRIPTION = "The path to the dynamic sharding tree file. E.g. /Users/example/path/to/tree.txt";
    private static final String TREE_OPTION_HINT = "path";
    private static final String SLIPPY_OPTION_LONG = "slippy";
    private static final String SLIPPY_OPTION_DESCRIPTION = "The slippy tile zoom level for the sharding.";
    private static final String SLIPPY_OPTION_HINT = "zoom";
    private static final String GEOHASH_OPTION_LONG = "geohash";
    private static final String GEOHASH_OPTION_DESCRIPTION = "The geohash precision level for the sharding.";
    private static final String GEOHASH_OPTION_HINT = "precision";
    private static final String INPUT_FILE_OPTION_LONG = "input";
    private static final String INPUT_FILE_OPTION_DESCRIPTION = "An input file from which to source the WKT entities. See DESCRIPTION section for details.";
    private static final String INPUT_FILE_OPTION_HINT = "file";
    private static final String COUNTRY_BOUNDARY_OPTION_LONG = "country-boundary";
    private static final String COUNTRY_BOUNDARY_OPTION_DESCRIPTION = "A boundary file to use for intersection checks. See DESCRIPTION section for details.";
    private static final String COUNTRY_BOUNDARY_OPTION_HINT = "boundary-file";
    private static final Integer TREE_CONTEXT = 3;
    private static final Integer SLIPPY_CONTEXT = 4;
    private static final Integer GEOHASH_CONTEXT = 5;
    private static final Integer COUNTRY_BOUNDARY_CONTEXT = 6;
    private static final String INPUT_WKT_SHARD = "wkt|shard";
    private final OptionAndArgumentDelegate optionAndArgumentDelegate = this.getOptionAndArgumentDelegate();
    private final CommandOutputDelegate outputDelegate = this.getCommandOutputDelegate();

    public static void main(String[] args) {
        new WKTShardCommand().runSubcommandAndExit(args);
    }

    @Override
    public int execute() {
        ArrayList<String> inputWktOrShard = new ArrayList<String>();
        if (this.optionAndArgumentDelegate.hasOption(INPUT_FILE_OPTION_LONG)) {
            inputWktOrShard.addAll(this.readInputsFromFile(this.optionAndArgumentDelegate.getOptionArgument(INPUT_FILE_OPTION_LONG).orElse(null)));
        }
        inputWktOrShard.addAll(this.optionAndArgumentDelegate.getVariadicArgument(INPUT_WKT_SHARD));
        if (inputWktOrShard.isEmpty()) {
            this.outputDelegate.printlnWarnMessage("no input WKTs were found");
            return 0;
        }
        Sharding sharding = null;
        CountryBoundaryMap countryBoundaryMap = null;
        if (this.optionAndArgumentDelegate.getParserContext() == TREE_CONTEXT.intValue()) {
            sharding = Sharding.forString("dynamic@" + this.optionAndArgumentDelegate.getOptionArgument(TREE_OPTION_LONG).orElseThrow(AtlasShellToolsException::new));
        } else if (this.optionAndArgumentDelegate.getParserContext() == SLIPPY_CONTEXT.intValue()) {
            sharding = Sharding.forString("slippy@" + this.optionAndArgumentDelegate.getOptionArgument(SLIPPY_OPTION_LONG).orElseThrow(AtlasShellToolsException::new));
        } else if (this.optionAndArgumentDelegate.getParserContext() == GEOHASH_CONTEXT.intValue()) {
            sharding = Sharding.forString("geohash@" + this.optionAndArgumentDelegate.getOptionArgument(GEOHASH_OPTION_LONG).orElseThrow(AtlasShellToolsException::new));
        } else if (this.optionAndArgumentDelegate.getParserContext() == COUNTRY_BOUNDARY_CONTEXT.intValue()) {
            Optional<CountryBoundaryMap> mapOptional = this.loadCountryBoundaryMap();
            if (!mapOptional.isPresent()) {
                this.outputDelegate.printlnErrorMessage("failed to load country boundary");
                return 1;
            }
            countryBoundaryMap = mapOptional.get();
        } else {
            throw new AtlasShellToolsException();
        }
        for (int i = 0; i < inputWktOrShard.size(); ++i) {
            String wktOrShard = (String)inputWktOrShard.get(i);
            this.parseWktOrShardAndPrintOutput(wktOrShard, sharding, countryBoundaryMap);
            if (i >= inputWktOrShard.size() - 1) continue;
            this.outputDelegate.printlnStdout("", new TTYAttribute[0]);
        }
        return 0;
    }

    @Override
    public String getCommandName() {
        return "wkt-shard";
    }

    @Override
    public String getSimpleDescription() {
        return "perform various intersection lookups";
    }

    @Override
    public void registerManualPageSections() {
        this.addManualPageSection("DESCRIPTION", WKTShardCommand.class.getResourceAsStream("WKTShardCommandDescriptionSection.txt"));
        this.addManualPageSection("EXAMPLES", WKTShardCommand.class.getResourceAsStream("WKTShardCommandExamplesSection.txt"));
    }

    @Override
    public void registerOptionsAndArguments() {
        this.registerArgument(INPUT_WKT_SHARD, ArgumentArity.VARIADIC, ArgumentOptionality.OPTIONAL, TREE_CONTEXT, SLIPPY_CONTEXT, GEOHASH_CONTEXT, COUNTRY_BOUNDARY_CONTEXT);
        this.registerOptionWithRequiredArgument(INPUT_FILE_OPTION_LONG, INPUT_FILE_OPTION_DESCRIPTION, OptionOptionality.OPTIONAL, INPUT_FILE_OPTION_HINT, TREE_CONTEXT, SLIPPY_CONTEXT, GEOHASH_CONTEXT, COUNTRY_BOUNDARY_CONTEXT);
        this.registerOptionWithRequiredArgument(TREE_OPTION_LONG, TREE_OPTION_DESCRIPTION, OptionOptionality.REQUIRED, TREE_OPTION_HINT, TREE_CONTEXT);
        this.registerOptionWithRequiredArgument(SLIPPY_OPTION_LONG, SLIPPY_OPTION_DESCRIPTION, OptionOptionality.REQUIRED, SLIPPY_OPTION_HINT, SLIPPY_CONTEXT);
        this.registerOptionWithRequiredArgument(GEOHASH_OPTION_LONG, GEOHASH_OPTION_DESCRIPTION, OptionOptionality.REQUIRED, GEOHASH_OPTION_HINT, GEOHASH_CONTEXT);
        this.registerOptionWithRequiredArgument(COUNTRY_BOUNDARY_OPTION_LONG, COUNTRY_BOUNDARY_OPTION_DESCRIPTION, OptionOptionality.REQUIRED, COUNTRY_BOUNDARY_OPTION_HINT, COUNTRY_BOUNDARY_CONTEXT);
        super.registerOptionsAndArguments();
    }

    private Optional<CountryBoundaryMap> loadCountryBoundaryMap() {
        File boundaryMapFile = new File(this.optionAndArgumentDelegate.getOptionArgument(COUNTRY_BOUNDARY_OPTION_LONG).orElseThrow(AtlasShellToolsException::new));
        if (!boundaryMapFile.exists()) {
            this.outputDelegate.printlnErrorMessage("boundary file " + boundaryMapFile.getAbsolutePath() + " does not exist");
            return Optional.empty();
        }
        if (this.optionAndArgumentDelegate.hasVerboseOption()) {
            this.outputDelegate.printlnCommandMessage("loading country boundary map...");
        }
        Optional<CountryBoundaryMap> countryBoundaryMap = Optional.of(CountryBoundaryMap.fromPlainText(boundaryMapFile));
        if (this.optionAndArgumentDelegate.hasVerboseOption()) {
            this.outputDelegate.printlnCommandMessage("loaded boundary map");
        }
        return countryBoundaryMap;
    }

    private void parseWktOrShardAndPrintOutput(String wktOrShard, Sharding sharding, CountryBoundaryMap countryBoundaryMap) {
        Optional<Geometry> geometryOptional = this.parseWktOrShardString(wktOrShard);
        if (geometryOptional.isEmpty()) {
            this.outputDelegate.printlnErrorMessage("unable to parse " + wktOrShard + " as WKT or shard string");
            return;
        }
        Geometry geometry = geometryOptional.get();
        if (geometry instanceof Point) {
            this.printPointOutput(wktOrShard, geometry, sharding, countryBoundaryMap);
        } else if (geometry instanceof LineString) {
            this.printLineStringOutput(wktOrShard, geometry, sharding, countryBoundaryMap);
        } else if (geometry instanceof org.locationtech.jts.geom.Polygon) {
            this.printPolygonOutput(wktOrShard, geometry, sharding, countryBoundaryMap);
        } else {
            this.outputDelegate.printlnErrorMessage("unsupported geometry type " + wktOrShard);
        }
    }

    private Optional<Geometry> parseWktOrShardString(String wktOrShard) {
        WKTReader reader = new WKTReader();
        try {
            return Optional.of(reader.read(wktOrShard));
        }
        catch (ParseException exception) {
            logger.warn("unable to parse {} as wkt", (Object)wktOrShard, (Object)exception);
            try {
                StringToShardConverter converter = new StringToShardConverter();
                Shard shard = converter.convert(wktOrShard);
                return Optional.of(new WKTReader().read(shard.toWkt()));
            }
            catch (Exception exception2) {
                logger.warn("unable to parse {} as shard", (Object)wktOrShard, (Object)exception2);
                return Optional.empty();
            }
        }
    }

    private void printLineStringOutput(String wktOrShard, Geometry geometry, Sharding sharding, CountryBoundaryMap countryBoundaryMap) {
        this.outputDelegate.printlnStdout(wktOrShard + " intersects:", TTYAttribute.BOLD);
        PolyLine polyline = new JtsPolyLineConverter().backwardConvert((LineString)geometry);
        if (sharding != null) {
            Iterable<Shard> shards = sharding.shardsIntersecting(polyline);
            for (Shard shard : shards) {
                this.outputDelegate.printlnStdout(shard.toString(), TTYAttribute.GREEN);
            }
        }
        if (countryBoundaryMap != null) {
            List<CountryBoundary> boundaries = countryBoundaryMap.boundaries(polyline);
            for (CountryBoundary boundary : boundaries) {
                this.outputDelegate.printlnStdout(boundary.getCountryName(), TTYAttribute.GREEN);
            }
        }
    }

    private void printPointOutput(String wktOrShard, Geometry geometry, Sharding sharding, CountryBoundaryMap countryBoundaryMap) {
        this.outputDelegate.printlnStdout(wktOrShard + " covered by:", TTYAttribute.BOLD);
        Location location = new JtsPointConverter().backwardConvert((Point)geometry);
        if (sharding != null) {
            Iterable<Shard> shards = sharding.shardsCovering(location);
            for (Shard shard : shards) {
                this.outputDelegate.printlnStdout(shard.toString(), TTYAttribute.GREEN);
            }
        }
        if (countryBoundaryMap != null) {
            List<CountryBoundary> boundaries = countryBoundaryMap.boundaries(location);
            for (CountryBoundary boundary : boundaries) {
                this.outputDelegate.printlnStdout(boundary.getCountryName(), TTYAttribute.GREEN);
            }
        }
    }

    private void printPolygonOutput(String wktOrShard, Geometry geometry, Sharding sharding, CountryBoundaryMap countryBoundaryMap) {
        this.outputDelegate.printlnStdout(wktOrShard + " contains or intersects:", TTYAttribute.BOLD);
        Polygon polygon = new JtsPolygonConverter().backwardConvert((org.locationtech.jts.geom.Polygon)geometry);
        if (sharding != null) {
            Iterable<Shard> shards = sharding.shards(polygon);
            for (Shard shard : shards) {
                this.outputDelegate.printlnStdout(shard.toString(), TTYAttribute.GREEN);
            }
        }
        if (countryBoundaryMap != null) {
            List<org.locationtech.jts.geom.Polygon> polygons = countryBoundaryMap.query(geometry.getEnvelopeInternal()).stream().distinct().collect(Collectors.toList());
            HashSet countries = new HashSet();
            polygons.forEach(polygon2 -> countries.add(CountryBoundaryMap.getGeometryProperty(polygon2, "iso_country_code")));
            for (String country : countries) {
                this.outputDelegate.printlnStdout(country, TTYAttribute.GREEN);
            }
        }
    }

    private List<String> readInputsFromFile(String path) {
        String[] split;
        if (path == null) {
            throw new AtlasShellToolsException();
        }
        Path inputPath = Paths.get(path, new String[0]);
        if (inputPath.toString().startsWith("~")) {
            this.outputDelegate.printlnWarnMessage("the '~' was not expanded by your shell");
        }
        if (!inputPath.toAbsolutePath().toFile().canRead() || !inputPath.toAbsolutePath().toFile().isFile()) {
            this.outputDelegate.printlnErrorMessage(inputPath.toAbsolutePath().toString() + " is not a readable file");
            return new ArrayList<String>();
        }
        ArrayList<String> wktOrShardList = new ArrayList<String>();
        StringResource resource = new StringResource();
        resource.copyFrom(new File(inputPath.toAbsolutePath().toString()));
        String rawText = resource.all();
        for (String line : split = rawText.split(System.getProperty("line.separator"))) {
            if (line.isEmpty()) continue;
            wktOrShardList.add(line);
        }
        return wktOrShardList;
    }
}

