package org.openstreetmap.atlas.checks.validation.verifier;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;

import org.junit.Assert;
import org.junit.rules.Verifier;
import org.openstreetmap.atlas.checks.base.BaseCheck;
import org.openstreetmap.atlas.checks.base.Check;
import org.openstreetmap.atlas.checks.flag.CheckFlag;
import org.openstreetmap.atlas.geography.atlas.Atlas;

import com.google.common.collect.Iterables;

/**
 * JUnit {@link Verifier} to run verifier functions on the list of {@link CheckFlag}s generated by
 * the checks under test.
 *
 * @author mkalender
 */
public class ConsumerBasedExpectedCheckVerifier extends Verifier
{
    private final List<CheckFlag> generatedFlags;
    private final List<Consumer<List<CheckFlag>>> globalVerifiers;
    private final List<Consumer<CheckFlag>> flagVerifiers;

    /**
     * Default constructor
     */
    public ConsumerBasedExpectedCheckVerifier()
    {
        this.generatedFlags = new ArrayList<>();
        this.globalVerifiers = new ArrayList<>();
        this.flagVerifiers = new ArrayList<>();
    }

    /**
     * Adds a test {@link Atlas} and a {@link Check} to verify
     *
     * @param atlas
     *            a test {@link Atlas}
     * @param check
     *            the {@link Check} under test
     */
    public void actual(final Atlas atlas, final BaseCheck<?> check)
    {
        Iterables.addAll(this.generatedFlags, check.flags(atlas));
        check.clear();
    }

    /**
     * Registers a custom function to verify all of the resulting flags when the {@link Check} is
     * run over the test {@link Atlas}
     *
     * @param verifierFunction
     *            a function that verifies the complete results
     * @return the {@link ConsumerBasedExpectedCheckVerifier}
     */
    public ConsumerBasedExpectedCheckVerifier globallyVerify(
            final Consumer<List<CheckFlag>> verifierFunction)
    {
        this.globalVerifiers.add(verifierFunction);
        return this;
    }

    /**
     * Registers a custom function to verify each resulting {@link CheckFlag} individually
     *
     * @param verifierFunction
     *            a function that verifies each result individually
     * @return the {@link ConsumerBasedExpectedCheckVerifier}
     */
    public ConsumerBasedExpectedCheckVerifier verify(final Consumer<CheckFlag> verifierFunction)
    {
        this.flagVerifiers.add(verifierFunction);
        return this;
    }

    /**
     * Verifies that no {@link CheckFlag}s were returned when the {@link Check} is run over the test
     * {@link Atlas}
     *
     * @return the {@link ConsumerBasedExpectedCheckVerifier}
     */
    public ConsumerBasedExpectedCheckVerifier verifyEmpty()
    {
        this.globalVerifiers
                .add((final List<CheckFlag> flags) -> Assert.assertTrue(flags.size() == 0));
        return this;
    }

    /**
     * Verifies that a specific number of {@link CheckFlag}s were returned when the {@link Check} is
     * run over the test {@link Atlas}
     *
     * @param size
     *            The number of {@link CheckFlag}s expected
     * @return the {@link ConsumerBasedExpectedCheckVerifier}
     */
    public ConsumerBasedExpectedCheckVerifier verifyExpectedSize(final int size)
    {
        this.globalVerifiers
                .add((final List<CheckFlag> flags) -> Assert.assertEquals(size, flags.size()));
        return this;
    }

    /**
     * Verifies that at least one {@link CheckFlag} was returned when the {@link Check} is run over
     * the test {@link Atlas}
     * 
     * @return the {@link ConsumerBasedExpectedCheckVerifier}
     */
    public ConsumerBasedExpectedCheckVerifier verifyNotEmpty()
    {
        this.globalVerifiers
                .add((final List<CheckFlag> flags) -> Assert.assertTrue(flags.size() > 0));
        return this;
    }

    /**
     * Checks for null values returned by the {@link Check}
     *
     * @return the {@link ConsumerBasedExpectedCheckVerifier}
     */
    public ConsumerBasedExpectedCheckVerifier verifyNotNull()
    {
        this.globalVerifiers.add((final List<CheckFlag> flags) -> Assert.assertNotNull(flags));
        return this;
    }

    @Override
    protected void verify() throws Throwable
    {
        if (this.globalVerifiers.isEmpty() && this.flagVerifiers.isEmpty())
        {
            throw new Exception("Please create at least one function to verify test results.");
        }

        // Apply global verifiers
        this.globalVerifiers.forEach(consumer -> consumer.accept(this.generatedFlags));

        // Apply per flag verifiers
        this.generatedFlags.forEach((final CheckFlag flag) -> this.flagVerifiers
                .forEach(consumer -> consumer.accept(flag)));
    }
}
