/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.atlas.utilities.arrays;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import org.openstreetmap.atlas.exception.CoreException;
import org.openstreetmap.atlas.utilities.arrays.PrimitiveArray;
import org.openstreetmap.atlas.utilities.collections.Iterables;
import org.openstreetmap.atlas.utilities.scalars.Ratio;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class LargeArray<T>
implements Iterable<T>,
Serializable {
    private static final long serialVersionUID = -8827093729953143426L;
    private static final Logger logger = LoggerFactory.getLogger(LargeArray.class);
    private static final int DEFAULT_MEMORY_BLOCK_SIZE = 1024;
    private final List<PrimitiveArray<T>> arrays;
    private final long maximumSize;
    private long nextIndex = 0L;
    private final int memoryBlockSize;
    private final int subArraySize;
    private String name = null;

    public LargeArray(long maximumSize) {
        this(maximumSize, 1024, Integer.MAX_VALUE);
    }

    public LargeArray(long maximumSize, int memoryBlockSize, int subArraySize) {
        if (memoryBlockSize < 0) {
            throw new CoreException("memoryBlockSize ({}) cannot be negative", memoryBlockSize);
        }
        if (subArraySize < 0) {
            throw new CoreException("subArraySize ({}) cannot be negative", subArraySize);
        }
        this.maximumSize = maximumSize;
        this.arrays = new ArrayList<PrimitiveArray<T>>();
        this.memoryBlockSize = memoryBlockSize;
        this.subArraySize = subArraySize;
    }

    protected LargeArray() {
        this.arrays = null;
        this.maximumSize = 0L;
        this.memoryBlockSize = 0;
        this.subArraySize = 0;
    }

    public void add(T item) {
        if (this.nextIndex >= this.maximumSize) {
            throw new CoreException("The array is full. Cannot add " + item);
        }
        int arrayIndex = this.arrayIndex(this.nextIndex);
        int indexInside = this.indexInside(this.nextIndex);
        if (this.arrays.size() <= arrayIndex) {
            this.arrays.add(this.getNewArray(this.memoryBlockSize));
        }
        if (indexInside >= this.arrays.get(arrayIndex).size()) {
            PrimitiveArray<T> old = this.arrays.get(arrayIndex);
            int maximumSizeFromDoubling = Math.min(2 * old.size(), this.subArraySize);
            long filledArraysSize = this.filledArraysSize();
            int maximumSizeFromTotal = (int)Math.min(this.maximumSize - filledArraysSize, (long)this.subArraySize);
            int newSize = Math.min(maximumSizeFromDoubling, maximumSizeFromTotal);
            logger.warn("Resizing array {} of {} ({}), from {} to {}.", new Object[]{arrayIndex, this.getName() == null ? super.toString() : this.getName(), this.getClass().getSimpleName(), old.size(), newSize});
            this.arrays.set(arrayIndex, old.withNewSize(newSize));
        }
        this.arrays.get(this.arrayIndex(this.nextIndex)).set(this.indexInside(this.nextIndex), item);
        ++this.nextIndex;
    }

    public boolean equals(Object other) {
        if (other instanceof LargeArray) {
            if (this == other) {
                return true;
            }
            LargeArray that = (LargeArray)other;
            if (!Objects.equals(this.getName(), that.getName())) {
                return false;
            }
            if (this.size() != that.size()) {
                return false;
            }
            for (long index = 0L; index < this.size(); ++index) {
                if (this.get(index).equals(that.get(index))) continue;
                return false;
            }
            return true;
        }
        return false;
    }

    public T get(long index) {
        if (index >= this.nextIndex) {
            throw new CoreException(index + " is out of bounds (size = " + this.size() + ")");
        }
        return this.arrays.get(this.arrayIndex(index)).get(this.indexInside(index));
    }

    public String getName() {
        return this.name;
    }

    public int hashCode() {
        int initialPrime = 31;
        int hashSeed = 37;
        int nameHash = this.getName() == null ? 0 : this.getName().hashCode();
        int hash = 1147 + nameHash;
        hash = 37 * hash + Long.valueOf(this.size()).hashCode();
        for (long index = 0L; index < this.size(); ++index) {
            hash = 37 * hash + this.get(index).hashCode();
        }
        return hash;
    }

    public boolean isEmpty() {
        return this.size() == 0L;
    }

    @Override
    public Iterator<T> iterator() {
        return Iterables.indexBasedIterable(this.size(), this::get).iterator();
    }

    public void set(long index, T item) {
        if (index >= this.size()) {
            throw new CoreException("Cannot replace an element that is not there. Index: " + index + ", size: " + this.size());
        }
        this.arrays.get(this.arrayIndex(index)).set(this.indexInside(index), item);
    }

    public void setName(String name) {
        this.name = name;
    }

    public long size() {
        return this.nextIndex;
    }

    public String toString() {
        StringBuilder builder = new StringBuilder();
        builder.append("[");
        builder.append(this.getClass().getSimpleName());
        builder.append(" ");
        this.forEach(t -> {
            if (builder.length() > 0) {
                builder.append(", ");
            }
            builder.append(t);
        });
        builder.append("]");
        return builder.toString();
    }

    public void trim() {
        this.trimIfLessFilledThan(Ratio.MAXIMUM);
    }

    public void trimIfLessFilledThan(Ratio ratio) {
        logger.trace("Trimming {} with Ratio {}", (Object)this.getName(), (Object)ratio);
        if (this.arrays.isEmpty()) {
            return;
        }
        int arrayIndex = this.arrays.size() - 1;
        PrimitiveArray<T> rightmost = this.arrays.get(arrayIndex);
        if (this.rightmostSubarrayIsFull()) {
            return;
        }
        int indexInside = this.indexInside(this.nextIndex);
        if (Ratio.ratio((double)indexInside / (double)rightmost.size()).isLessThan(ratio)) {
            this.arrays.set(arrayIndex, rightmost.trimmed(indexInside));
        }
    }

    public LargeArray<T> withName(String name) {
        this.setName(name);
        return this;
    }

    protected List<PrimitiveArray<T>> getArrays() {
        return this.arrays;
    }

    protected abstract PrimitiveArray<T> getNewArray(int var1);

    private int arrayIndex(long index) {
        return (int)(index / (long)this.subArraySize);
    }

    private long filledArraysSize() {
        if (this.arrays.size() <= 1) {
            return 0L;
        }
        long result = 0L;
        for (int i = 0; i < this.arrays.size() - 2; ++i) {
            result += (long)this.arrays.get(i).size();
        }
        return result;
    }

    private int indexInside(long index) {
        return (int)(index % (long)this.subArraySize);
    }

    private boolean rightmostSubarrayIsFull() {
        return this.nextIndex > 0L && this.indexInside(this.nextIndex) == 0;
    }
}

