/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.atlas.geography.sharding;

import java.io.BufferedWriter;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.openstreetmap.atlas.exception.CoreException;
import org.openstreetmap.atlas.geography.GeometricSurface;
import org.openstreetmap.atlas.geography.Located;
import org.openstreetmap.atlas.geography.Location;
import org.openstreetmap.atlas.geography.PolyLine;
import org.openstreetmap.atlas.geography.Rectangle;
import org.openstreetmap.atlas.geography.geojson.GeoJsonBuilder;
import org.openstreetmap.atlas.geography.geojson.GeoJsonObject;
import org.openstreetmap.atlas.geography.sharding.Shard;
import org.openstreetmap.atlas.geography.sharding.Sharding;
import org.openstreetmap.atlas.geography.sharding.SlippyTile;
import org.openstreetmap.atlas.streaming.Streams;
import org.openstreetmap.atlas.streaming.resource.File;
import org.openstreetmap.atlas.streaming.resource.Resource;
import org.openstreetmap.atlas.streaming.resource.WritableResource;
import org.openstreetmap.atlas.streaming.writers.JsonWriter;
import org.openstreetmap.atlas.streaming.writers.SafeBufferedWriter;
import org.openstreetmap.atlas.utilities.collections.Iterables;
import org.openstreetmap.atlas.utilities.collections.StringList;
import org.openstreetmap.atlas.utilities.runtime.Command;
import org.openstreetmap.atlas.utilities.runtime.CommandMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DynamicTileSharding
extends Command
implements Sharding {
    public static final Command.Switch<Resource> DEFINITION = new Command.Switch("definition", "Resource containing the maxZoom - 1 tile to feature count mapping.", File::new, Command.Optionality.REQUIRED);
    public static final Command.Switch<WritableResource> GEOJSON = new Command.Switch("geoJson", "The resource where to save the geojson tree for debugging", File::new, Command.Optionality.OPTIONAL);
    public static final Command.Switch<Integer> MAXIMUM_COUNT = new Command.Switch("maxCount", "The maximum feature count. Any cell with a larger feature count will be split, up to maxZoom", Integer::valueOf, Command.Optionality.OPTIONAL, "200000");
    public static final Command.Switch<Integer> MAXIMUM_ZOOM = new Command.Switch("maxZoom", "The maximum zoom", Integer::valueOf, Command.Optionality.OPTIONAL, "10");
    public static final Command.Switch<Integer> MINIMUM_ZOOM = new Command.Switch("minZoom", "The minimum zoom", Integer::valueOf, Command.Optionality.OPTIONAL, "5");
    public static final Command.Switch<WritableResource> OUTPUT = new Command.Switch("output", "The resource where to save the serialized tree.", File::new, Command.Optionality.REQUIRED);
    private static final int MINIMUM_TO_SPLIT = 1000;
    private static final int READER_REPORT_FREQUENCY = 10000000;
    private static final Logger logger = LoggerFactory.getLogger(DynamicTileSharding.class);
    private static final long serialVersionUID = 229952569300405488L;
    private final Node root;

    public static void main(String[] args) {
        new DynamicTileSharding().run(args);
    }

    public DynamicTileSharding(Resource resource) {
        this.root = Node.read(resource);
    }

    private DynamicTileSharding() {
        this.root = new Node();
    }

    public boolean equals(Object other) {
        if (this == other) {
            return true;
        }
        if (other == null || this.getClass() != other.getClass()) {
            return false;
        }
        DynamicTileSharding that = (DynamicTileSharding)other;
        return this.root.deepEquals(that.root);
    }

    public int hashCode() {
        return Objects.hash(this.root);
    }

    @Override
    public Iterable<Shard> neighbors(Shard shard) {
        return this.root.neighbors(SlippyTile.forName(shard.getName())).stream().map(Node::getTile).collect(Collectors.toList());
    }

    public void save(WritableResource resource) {
        this.root.save(resource);
    }

    public void saveAsGeoJson(WritableResource resource) {
        JsonWriter writer = new JsonWriter(resource);
        GeoJsonObject geoJson = new GeoJsonBuilder().create(Iterables.translate(this.root.leafNodes(Rectangle.MAXIMUM), Node::toGeoJsonBuildingBlock));
        writer.write(geoJson.jsonObject());
        writer.close();
    }

    @Override
    public Shard shardForName(String name) {
        SlippyTile result = SlippyTile.forName(name);
        if (!this.root.leafNodesCovering(result.bounds().center()).contains(new Node(result))) {
            throw new CoreException("This tree does not contain tile {}", name);
        }
        return result;
    }

    @Override
    public Iterable<Shard> shards(GeometricSurface surface) {
        return Iterables.stream(this.root.leafNodes(surface)).map(Node::getTile);
    }

    @Override
    public Iterable<Shard> shardsCovering(Location location) {
        return Iterables.stream(this.root.leafNodesCovering(location)).map(Node::getTile);
    }

    @Override
    public Iterable<Shard> shardsIntersecting(PolyLine polyLine) {
        return Iterables.stream(this.root.leafNodesIntersecting(polyLine)).map(Node::getTile);
    }

    protected Map<SlippyTile, Long> calculateTileCountsForAllZoom(int firstZoomLayerToGenerate, Map<SlippyTile, Long> counts) {
        for (int currentZoom = firstZoomLayerToGenerate; currentZoom >= 0; --currentZoom) {
            long count = 0L;
            long tilesCalculated = 0L;
            int x = 0;
            while ((double)x < Math.pow(2.0, (double)currentZoom + 1.0)) {
                int y = 0;
                while ((double)y < Math.pow(2.0, (double)currentZoom + 1.0)) {
                    count = 0L;
                    count += counts.getOrDefault(new SlippyTile(x, y, currentZoom + 1), 0L).longValue();
                    count += counts.getOrDefault(new SlippyTile(x + 1, y, currentZoom + 1), 0L).longValue();
                    count += counts.getOrDefault(new SlippyTile(x, y + 1, currentZoom + 1), 0L).longValue();
                    if ((count += counts.getOrDefault(new SlippyTile(x + 1, y + 1, currentZoom + 1), 0L).longValue()) != 0L) {
                        counts.put(new SlippyTile(x / 2, y / 2, currentZoom), count);
                    }
                    if (++tilesCalculated % 10000000L == 0L) {
                        logger.info("Calculated {} zoom level {} tiles.", (Object)tilesCalculated, (Object)currentZoom);
                    }
                    y += 2;
                }
                x += 2;
            }
        }
        return counts;
    }

    @Override
    protected int onRun(CommandMap command) {
        Resource definition = (Resource)command.get(DEFINITION);
        int numberLines = (int)Iterables.size(definition.lines());
        logger.info("There are {} tiles.", (Object)numberLines);
        HashMap<SlippyTile, Long> counts = new HashMap<SlippyTile, Long>(numberLines);
        WritableResource output = (WritableResource)command.get(OUTPUT);
        int maximum = (Integer)command.get(MAXIMUM_COUNT);
        int minimumZoom = (Integer)command.get(MINIMUM_ZOOM);
        int maximumZoom = (Integer)command.get(MAXIMUM_ZOOM);
        WritableResource geoJson = (WritableResource)command.get(GEOJSON);
        int zoom = 0;
        int counter = 0;
        for (String line : definition.lines()) {
            StringList split = StringList.split(line, ",");
            SlippyTile tile2 = SlippyTile.forName(split.get(0));
            counts.put(tile2, Long.valueOf(split.get(1)));
            zoom = tile2.getZoom();
            if (++counter % 10000000 != 0) continue;
            logger.info("Read counts for {} zoom level {} tiles.", (Object)counter, (Object)zoom);
        }
        Map<SlippyTile, Long> allCounts = this.calculateTileCountsForAllZoom(maximumZoom - 2, counts);
        if (zoom == 0) {
            throw new CoreException("No tiles in definition");
        }
        int finalZoom = zoom;
        if (maximumZoom > finalZoom + 1) {
            throw new CoreException("Cannot go over the resolution of the counts definition. MaxZoom = {} has to be at most equal to definition zoom + 1 = {}", maximumZoom, finalZoom);
        }
        this.root.build(tile -> {
            long count = allCounts.getOrDefault(tile, 0L);
            if (count <= 1000L) {
                return false;
            }
            if (tile.getZoom() < minimumZoom) {
                return true;
            }
            if (tile.getZoom() >= maximumZoom) {
                return false;
            }
            return count > (long)maximum;
        });
        this.save(output);
        String outputLocation = this.lastRawCommand(OUTPUT);
        logger.info("Printed tree to {}. Loading for verification...", (Object)outputLocation);
        new DynamicTileSharding(new File(outputLocation));
        logger.info("Successfully loaded tree from {}", (Object)outputLocation);
        if (geoJson != null) {
            if (logger.isInfoEnabled()) {
                logger.info("Saving geojson to {}...", (Object)this.lastRawCommand(GEOJSON));
            }
            this.saveAsGeoJson(geoJson);
        }
        return 0;
    }

    @Override
    protected Command.SwitchList switches() {
        return new Command.SwitchList().with(DEFINITION, OUTPUT, MINIMUM_ZOOM, MAXIMUM_ZOOM, MAXIMUM_COUNT, GEOJSON);
    }

    private static class Node
    implements Located,
    Serializable {
        private static final int MAXIMUM_CHILDREN = 4;
        private static final long serialVersionUID = -7789058745501080439L;
        private final List<Node> children;
        private final SlippyTile tile;

        protected static Node read(Resource resource) {
            return Node.read(resource.lines().iterator());
        }

        private static Node read(Iterator<String> lineIterator) {
            String line = lineIterator.next();
            ArrayList<Node> children = new ArrayList<Node>();
            String tileName = line;
            if (line.endsWith("+")) {
                for (int i = 0; i < 4; ++i) {
                    children.add(Node.read(lineIterator));
                }
                tileName = line.substring(0, line.length() - 1);
            }
            return new Node(SlippyTile.forName(tileName), children);
        }

        protected Node() {
            this(SlippyTile.ROOT);
        }

        private Node(SlippyTile tile) {
            this.tile = tile;
            this.children = new ArrayList<Node>();
        }

        private Node(SlippyTile tile, List<Node> children) {
            this.tile = tile;
            this.children = children;
        }

        @Override
        public Rectangle bounds() {
            return this.tile.bounds();
        }

        public boolean equals(Object other) {
            if (other instanceof Node) {
                return ((Node)other).getTile().equals(this.tile);
            }
            return false;
        }

        public SlippyTile getTile() {
            return this.tile;
        }

        public int hashCode() {
            return this.tile.hashCode();
        }

        public Set<Node> leafNodes(GeometricSurface surface) {
            HashSet<Node> result = new HashSet<Node>();
            Rectangle polygonBounds = surface.bounds();
            if (polygonBounds.overlaps(this.bounds())) {
                if (this.isFinal() && surface.overlaps(this.bounds())) {
                    result.add(this);
                } else {
                    for (Node child : this.children) {
                        result.addAll(child.leafNodes(surface));
                    }
                }
            }
            return result;
        }

        public Set<Node> leafNodesCovering(Location location) {
            HashSet<Node> result = new HashSet<Node>();
            if (this.bounds().fullyGeometricallyEncloses(location)) {
                if (this.isFinal()) {
                    result.add(this);
                } else {
                    for (Node child : this.children) {
                        result.addAll(child.leafNodesCovering(location));
                    }
                }
            }
            return result;
        }

        public Set<Node> leafNodesIntersecting(PolyLine polyLine) {
            Rectangle polyLineBounds = polyLine.bounds();
            return this.leafNodesIntersecting(polyLine, polyLineBounds);
        }

        public Set<Node> leafNodesIntersecting(PolyLine polyLine, Rectangle polyLineBounds) {
            HashSet<Node> result = new HashSet<Node>();
            if (polyLineBounds.overlaps(this.bounds())) {
                if (this.isFinal() && (polyLine.intersects(this.bounds()) || this.bounds().fullyGeometricallyEncloses(polyLine))) {
                    result.add(this);
                } else {
                    for (Node child : this.children) {
                        result.addAll(child.leafNodesIntersecting(polyLine, polyLineBounds));
                    }
                }
            }
            return result;
        }

        public GeoJsonBuilder.LocationIterableProperties toGeoJsonBuildingBlock() {
            HashMap<String, String> tags = new HashMap<String, String>();
            tags.put("tile", this.name());
            return new GeoJsonBuilder.LocationIterableProperties(this.bounds(), tags);
        }

        protected void build(Predicate<SlippyTile> shouldSplit) {
            if (this.zoom() < 30 && shouldSplit.test(this.tile)) {
                this.split();
                for (Node child : this.children) {
                    child.build(shouldSplit);
                }
            }
        }

        protected boolean isFinal() {
            return this.children.isEmpty();
        }

        protected String name() {
            return this.tile.getName();
        }

        protected Set<Node> neighbors(SlippyTile targetTile) {
            HashSet<Node> neighboringNodes = new HashSet<Node>();
            for (Node leafNode : this.leafNodes(targetTile.bounds())) {
                Rectangle expandedBoundary = leafNode.bounds().expand(SlippyTile.calculateExpansionDistance(leafNode.bounds()));
                if (!targetTile.bounds().overlaps(expandedBoundary) || leafNode.bounds().equals(targetTile.bounds())) continue;
                neighboringNodes.add(leafNode);
            }
            return neighboringNodes;
        }

        protected void save(WritableResource resource) {
            SafeBufferedWriter writer = resource.writer();
            try {
                this.save(writer);
            }
            catch (Exception e) {
                Streams.close(writer);
                throw e;
            }
            Streams.close(writer);
        }

        protected void split() {
            this.children.addAll(this.tile.split(this.zoom() + 1).stream().map(Node::new).collect(Collectors.toList()));
        }

        protected int zoom() {
            return this.tile.getZoom();
        }

        private boolean deepEquals(Node other) {
            Comparator<Node> nodeCompare = Comparator.comparing(Node::getTile);
            LinkedList<Node> queue = new LinkedList<Node>();
            queue.offer(this);
            queue.offer(other);
            while (!queue.isEmpty()) {
                Node node2;
                Node node1 = (Node)queue.poll();
                if (node1.equals(node2 = (Node)queue.poll()) && node1.getChildren().size() == node2.getChildren().size()) {
                    List<Node> children1 = node1.getChildren();
                    List<Node> children2 = node2.getChildren();
                    children1.sort(nodeCompare);
                    children2.sort(nodeCompare);
                    for (int index = 0; index < children1.size(); ++index) {
                        queue.offer(children1.get(index));
                        queue.offer(children2.get(index));
                    }
                    continue;
                }
                return false;
            }
            return true;
        }

        private List<Node> getChildren() {
            return this.children;
        }

        private void save(BufferedWriter writer) {
            try {
                writer.write(this.tile.getName());
                if (!this.isFinal()) {
                    writer.write("+");
                }
                writer.write("\n");
            }
            catch (Exception e) {
                throw new CoreException("Unable to write slippy tile {}", this.tile, e);
            }
            for (Node child : this.children) {
                child.save(writer);
            }
        }
    }
}

