/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.atlas.geography.boundary;

import java.io.IOException;
import java.util.List;
import java.util.Set;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.MultiPolygon;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jtslab.SnapRoundOverlayFunctions;
import org.openstreetmap.atlas.exception.CoreException;
import org.openstreetmap.atlas.geography.Rectangle;
import org.openstreetmap.atlas.geography.atlas.Atlas;
import org.openstreetmap.atlas.geography.atlas.packed.PackedAtlas;
import org.openstreetmap.atlas.geography.boundary.CountryBoundary;
import org.openstreetmap.atlas.geography.boundary.CountryBoundaryMap;
import org.openstreetmap.atlas.geography.boundary.CountryBoundaryMapPrinter;
import org.openstreetmap.atlas.geography.converters.jts.JtsMultiPolygonConverter;
import org.openstreetmap.atlas.geography.converters.jts.JtsPolygonConverter;
import org.openstreetmap.atlas.geography.sharding.SlippyTile;
import org.openstreetmap.atlas.streaming.compression.Compressor;
import org.openstreetmap.atlas.streaming.resource.File;
import org.openstreetmap.atlas.streaming.resource.Resource;
import org.openstreetmap.atlas.streaming.resource.TemporaryFile;
import org.openstreetmap.atlas.utilities.runtime.Command;
import org.openstreetmap.atlas.utilities.runtime.CommandMap;
import org.openstreetmap.atlas.utilities.time.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CountryBoundaryMapArchiver
extends Command {
    private static final Logger logger = LoggerFactory.getLogger(CountryBoundaryMapArchiver.class);
    protected static final Command.Switch<File> SHAPE_FILE = new Command.Switch("shp", "path to the shape file", File::new, Command.Optionality.OPTIONAL);
    protected static final Command.Switch<Atlas> ATLAS = new Command.Switch("atlas", "path to the atlas file containing boundaries", path -> PackedAtlas.load(new File((String)path)), Command.Optionality.OPTIONAL);
    protected static final Command.Switch<File> BOUNDARY_FILE = new Command.Switch("boundaries", "path to the pre-existing boundary file", File::new, Command.Optionality.OPTIONAL);
    protected static final Command.Switch<File> OUTPUT = new Command.Switch("out", "The output file format", File::new, Command.Optionality.REQUIRED);
    protected static final Command.Switch<Rectangle> BOUNDS = new Command.Switch("bounds", "The bounds", Rectangle::forString, Command.Optionality.OPTIONAL, Rectangle.MAXIMUM.toCompactString());
    protected static final Command.Switch<Boolean> CREATE_SPATIAL_INDEX = new Command.Switch("createSpatialIndex", "Indicator whether to create a spatial grid index and include that in the output.", Boolean::parseBoolean, Command.Optionality.OPTIONAL, Boolean.FALSE.toString());
    protected static final Command.Switch<Integer> OCEAN_BOUNDARY_ZOOM_LEVEL = new Command.Switch("oceanBoundaryZoomLevel", "The zoom level at which to create ocean tiles to fill in potential voids. Recommended value: 3", Integer::parseInt, Command.Optionality.OPTIONAL);
    protected static final Command.Switch<Boolean> SAVE_GEOJSON_WKT = new Command.Switch("saveGeojsonWkt", "Save the country boundaries to Geojson and WKT", Boolean::parseBoolean, Command.Optionality.OPTIONAL, Boolean.FALSE.toString());
    private static final double JTS_SNAP_PRECISION = 1.0E-15;

    public static void main(String[] args) {
        new CountryBoundaryMapArchiver().run(args);
    }

    public CountryBoundaryMap read(Resource resource) {
        return CountryBoundaryMap.fromPlainText(resource);
    }

    protected CountryBoundaryMap generateOceanBoundaryMap(CountryBoundaryMap boundaryMap, Iterable<SlippyTile> allTiles) {
        CountryBoundaryMap finalBoundaryMap = new CountryBoundaryMap();
        int oceanCountryCount = 0;
        logger.info("Calculating ocean boundaries...");
        for (SlippyTile tile : allTiles) {
            Time start = Time.now();
            String countryCode = String.format("O%02d", oceanCountryCount);
            Geometry countryGeometry = this.geometryForShard(tile.bounds(), boundaryMap);
            if (!countryGeometry.isEmpty()) {
                if (countryGeometry instanceof Polygon) {
                    finalBoundaryMap.addCountry(countryCode, (Polygon)countryGeometry);
                }
                if (countryGeometry instanceof MultiPolygon) {
                    finalBoundaryMap.addCountry(countryCode, (MultiPolygon)countryGeometry);
                }
                logger.info("Added Ocean Country {} in {}", (Object)countryCode, (Object)start.elapsedSince());
                ++oceanCountryCount;
                continue;
            }
            logger.info("Skipped Ocean Country {} in {}. It is land covered.", (Object)tile.getName(), (Object)start.elapsedSince());
        }
        logger.info("Adding back country boundaries to the new ocean boundary map");
        JtsMultiPolygonConverter multiPolyConverter = new JtsMultiPolygonConverter();
        for (String country : boundaryMap.allCountryNames()) {
            for (CountryBoundary countryBoundary : boundaryMap.countryBoundary(country)) {
                GeometryFactory factory = new GeometryFactory();
                Set<Polygon> boundaryPolygons = multiPolyConverter.convert(countryBoundary.getBoundary());
                MultiPolygon countryGeometry = factory.createMultiPolygon(boundaryPolygons.toArray(new Polygon[boundaryPolygons.size()]));
                finalBoundaryMap.addCountry(country, countryGeometry);
            }
        }
        return finalBoundaryMap;
    }

    protected Geometry geometryForShard(Rectangle shardBounds, CountryBoundaryMap boundaryMap) {
        JtsMultiPolygonConverter multiPolyConverter = new JtsMultiPolygonConverter();
        JtsPolygonConverter polyConverter = new JtsPolygonConverter();
        GeometryFactory factory = new GeometryFactory();
        List<CountryBoundary> boundaries = boundaryMap.boundaries(shardBounds);
        Geometry shardPolyJts = polyConverter.convert(shardBounds);
        for (CountryBoundary boundary : boundaries) {
            Set<Polygon> boundaryPolygons = multiPolyConverter.convert(boundary.getBoundary());
            MultiPolygon countryGeometry = factory.createMultiPolygon(boundaryPolygons.toArray(new Polygon[boundaryPolygons.size()]));
            shardPolyJts = SnapRoundOverlayFunctions.difference(shardPolyJts, countryGeometry, 1.0E-15);
        }
        return shardPolyJts;
    }

    @Override
    protected int onRun(CommandMap command) {
        File shapeFile = (File)command.get(SHAPE_FILE);
        Atlas atlas = (Atlas)command.get(ATLAS);
        File boundaries = (File)command.get(BOUNDARY_FILE);
        File output = (File)command.get(OUTPUT);
        output.setCompressor(Compressor.GZIP);
        Rectangle bounds = (Rectangle)command.get(BOUNDS);
        boolean createIndex = (Boolean)command.get(CREATE_SPATIAL_INDEX);
        Integer oceanBoundaryZoomLevel = (Integer)command.get(OCEAN_BOUNDARY_ZOOM_LEVEL);
        boolean saveGeojsonWkt = (Boolean)command.get(SAVE_GEOJSON_WKT);
        Time timer = Time.now();
        CountryBoundaryMap map = new CountryBoundaryMap(bounds);
        if (atlas != null) {
            map.readFromAtlas(atlas);
        } else if (shapeFile != null) {
            map.readFromShapeFile(shapeFile.getFile());
        } else if (boundaries != null) {
            map.readFromPlainText(boundaries);
        } else {
            throw new CoreException("No input data was specified to build a Country Boundary Map");
        }
        if (oceanBoundaryZoomLevel != null) {
            Iterable<SlippyTile> allTiles = SlippyTile.allTiles(oceanBoundaryZoomLevel);
            map = this.generateOceanBoundaryMap(map, allTiles);
            try (TemporaryFile temporary = File.temporary();){
                map.writeToFile(temporary);
                map = new CountryBoundaryMap(bounds);
                map.readFromPlainText(temporary);
            }
            catch (IOException e) {
                throw new CoreException("Could not write CountryBoundaryMap.", e);
            }
        }
        if (createIndex) {
            logger.info("Building Grid Index...");
            Time startTime = Time.now();
            Set<String> loadedCountries = map.getLoadedCountries();
            map.initializeGridIndex(loadedCountries);
            logger.info("Finished building Grid Index in {}", (Object)startTime.elapsedSince());
        }
        try {
            logger.info("Saving CountryBoundaryMap to {}.", (Object)output);
            map.writeToFile(output);
        }
        catch (IOException e) {
            throw new CoreException("Could not write CountryBoundaryMap.", e);
        }
        if (saveGeojsonWkt) {
            new CountryBoundaryMapPrinter().print(output);
        }
        logger.info("CountryBoundaryMap creation took {}.", (Object)timer.elapsedSince());
        return 0;
    }

    @Override
    protected Command.SwitchList switches() {
        return new Command.SwitchList().with(SHAPE_FILE, ATLAS, BOUNDARY_FILE, OUTPUT, BOUNDS, CREATE_SPATIAL_INDEX, OCEAN_BOUNDARY_ZOOM_LEVEL, SAVE_GEOJSON_WKT);
    }
}

