/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.atlas.geography.atlas;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.openstreetmap.atlas.exception.CoreException;
import org.openstreetmap.atlas.geography.atlas.Atlas;
import org.openstreetmap.atlas.geography.atlas.builder.text.TextAtlasBuilder;
import org.openstreetmap.atlas.geography.atlas.items.AtlasEntity;
import org.openstreetmap.atlas.geography.atlas.multi.MultiAtlas;
import org.openstreetmap.atlas.geography.atlas.packed.PackedAtlas;
import org.openstreetmap.atlas.geography.atlas.sub.AtlasCutType;
import org.openstreetmap.atlas.streaming.compression.Decompressor;
import org.openstreetmap.atlas.streaming.resource.AbstractResource;
import org.openstreetmap.atlas.streaming.resource.File;
import org.openstreetmap.atlas.streaming.resource.FileSuffix;
import org.openstreetmap.atlas.streaming.resource.Resource;
import org.openstreetmap.atlas.utilities.collections.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AtlasResourceLoader {
    public static final Predicate<Resource> HAS_TEXT_ATLAS_EXTENSION = FileSuffix.resourceFilter(FileSuffix.ATLAS, FileSuffix.TEXT).or(FileSuffix.resourceFilter(FileSuffix.ATLAS, FileSuffix.TEXT, FileSuffix.GZIP));
    public static final Predicate<Resource> HAS_ATLAS_EXTENSION = FileSuffix.resourceFilter(FileSuffix.ATLAS).or(FileSuffix.resourceFilter(FileSuffix.ATLAS, FileSuffix.GZIP));
    private static final Logger logger = LoggerFactory.getLogger(AtlasResourceLoader.class);
    private static final Predicate<Resource> CONTENTS_LOOK_LIKE_TEXT_ATLAS = resource -> {
        AtlasResourceLoader.checkFileExistsAndIsNotDirectory(resource);
        AtlasResourceLoader.setDecompressorFor(resource);
        return resource.firstLine().equals(TextAtlasBuilder.getNodesHeader());
    };
    private Predicate<Resource> resourceFilter = resource -> true;
    private Predicate<AtlasEntity> atlasEntityFilter = null;
    private String multiAtlasName;

    private static void checkFileExistsAndIsNotDirectory(Resource resource) {
        if (resource instanceof File) {
            File fileResource = (File)resource;
            if (!fileResource.exists()) {
                throw new CoreException("Resource {} was of type File but it could not be found", resource.getName());
            }
            if (fileResource.isDirectory()) {
                throw new CoreException("Resource {} was of type File but it was a directory. Try loadRecursively instead.", resource.getName());
            }
        }
    }

    private static void setDecompressorFor(Resource resource) {
        if (FileSuffix.GZIP.matches(resource)) {
            if (resource instanceof AbstractResource) {
                ((AbstractResource)resource).setDecompressor(Decompressor.GZIP);
            } else {
                throw new CoreException("Provide resource was of type {} which does not support decompression.", resource.getClass().getName());
            }
        }
    }

    public Atlas load(Resource ... resources) {
        return this.load(Iterables.from(resources));
    }

    public Atlas load(Iterable<? extends Resource> resources) {
        List<Resource> atlasResources = Iterables.stream(resources).flatMap(this::upcastAndRemoveNullResources).filter(this.resourceFilter).collectToList();
        if (atlasResources.isEmpty()) {
            throw new CoreException("No loadable Resources were found.");
        }
        Optional<Atlas> resultAtlasOptional = atlasResources.size() == 1 ? this.loadAtlasResource(atlasResources.get(0)) : this.loadMultipleAtlasResources(atlasResources);
        if (!resultAtlasOptional.isPresent()) {
            throw new CoreException("Unable to load atlas from provided Resources. If you are seeing this you likely found a bug with AtlasResourceLoader. Please report it.");
        }
        return this.applyEntityFilter(resultAtlasOptional.get());
    }

    public Atlas loadRecursively(Resource ... resources) {
        return this.loadRecursively(Iterables.from(resources));
    }

    public Atlas loadRecursively(Iterable<Resource> resources) {
        List<Resource> atlasResources = Iterables.stream(resources).filter(Objects::nonNull).flatMap(this::expandFileOrDirectoryRecursively).filter(HAS_ATLAS_EXTENSION.or(HAS_TEXT_ATLAS_EXTENSION)).filter(this.resourceFilter).collectToList();
        Optional<Atlas> resultAtlasOptional = this.loadMultipleAtlasResources(atlasResources);
        if (!resultAtlasOptional.isPresent()) {
            throw new CoreException("Unable to load atlas from provided Resources. If you are seeing this you likely found a bug with AtlasResourceLoader. Please report it.");
        }
        return this.applyEntityFilter(resultAtlasOptional.get());
    }

    public Optional<Atlas> safeLoad(Resource ... resources) {
        return this.safeLoad(Iterables.from(resources));
    }

    public Optional<Atlas> safeLoad(Iterable<Resource> resources) {
        try {
            return Optional.of(this.load(resources));
        }
        catch (Exception exception) {
            logger.warn("Could not load atlas from supplied resources", (Throwable)exception);
            return Optional.empty();
        }
    }

    public Optional<Atlas> safeLoadRecursively(Resource ... resources) {
        return this.safeLoadRecursively(Iterables.from(resources));
    }

    public Optional<Atlas> safeLoadRecursively(Iterable<Resource> resources) {
        try {
            return Optional.of(this.loadRecursively(resources));
        }
        catch (Exception exception) {
            logger.warn("Could not load atlas from supplied resources", (Throwable)exception);
            return Optional.empty();
        }
    }

    public void setAtlasEntityFilter(Predicate<AtlasEntity> filter) {
        this.atlasEntityFilter = filter;
    }

    public void setResourceFilter(Predicate<Resource> filter) {
        this.resourceFilter = filter;
    }

    public AtlasResourceLoader withAtlasEntityFilter(Predicate<AtlasEntity> filter) {
        this.setAtlasEntityFilter(filter);
        return this;
    }

    public AtlasResourceLoader withMultiAtlasName(String multiAtlasName) {
        this.multiAtlasName = multiAtlasName;
        return this;
    }

    public AtlasResourceLoader withResourceFilter(Predicate<Resource> filter) {
        this.setResourceFilter(filter);
        return this;
    }

    private Atlas applyEntityFilter(Atlas atlasToFilter) {
        if (this.atlasEntityFilter != null) {
            Optional<Atlas> subAtlas = atlasToFilter.subAtlas(this.atlasEntityFilter, AtlasCutType.SOFT_CUT);
            return subAtlas.orElseThrow(() -> new CoreException("Entity filter resulted in an empty atlas"));
        }
        return atlasToFilter;
    }

    private List<Resource> expandFileOrDirectoryRecursively(Resource resource) {
        if (resource == null) {
            return new ArrayList<Resource>();
        }
        if (!(resource instanceof File)) {
            throw new CoreException("Resource {} was not a File, instead was {}", resource.getName(), resource.getClass().getName());
        }
        File file = (File)resource;
        ArrayList<Resource> result = new ArrayList<Resource>();
        if (file.isDirectory()) {
            file.listFilesRecursively().forEach(child -> {
                if (child.isGzipped()) {
                    child.setDecompressor(Decompressor.GZIP);
                }
                result.add((Resource)child);
            });
        } else {
            result.add(file);
        }
        return result;
    }

    private List<Resource> filterForBinaryAtlasResources(List<Resource> atlasResources) {
        return atlasResources.stream().filter(CONTENTS_LOOK_LIKE_TEXT_ATLAS.negate()).collect(Collectors.toList());
    }

    private List<Resource> filterForTextAtlasResources(List<Resource> atlasResources) {
        return atlasResources.stream().filter(CONTENTS_LOOK_LIKE_TEXT_ATLAS).collect(Collectors.toList());
    }

    private Optional<Atlas> loadAtlasResource(Resource resource) {
        PackedAtlas result;
        if (resource instanceof File) {
            AtlasResourceLoader.checkFileExistsAndIsNotDirectory(resource);
        }
        if (resource.length() == 0L) {
            throw new CoreException("{} {} had zero length!", resource.getClass().getName(), resource.getName());
        }
        if (CONTENTS_LOOK_LIKE_TEXT_ATLAS.test(resource)) {
            AtlasResourceLoader.setDecompressorFor(resource);
            result = new TextAtlasBuilder().read(resource);
        } else {
            try {
                result = PackedAtlas.load(resource);
            }
            catch (Exception exception) {
                throw new CoreException("Failed to load an atlas from {} with name {}", resource.getClass().getName(), resource.getName(), exception);
            }
        }
        return Optional.ofNullable(result);
    }

    private Optional<Atlas> loadMultipleAtlasResources(List<Resource> atlasResources) {
        List<Atlas> textAtlases;
        atlasResources.forEach(resource -> {
            if (resource instanceof File) {
                AtlasResourceLoader.checkFileExistsAndIsNotDirectory(resource);
            }
            if (resource.length() == 0L) {
                throw new CoreException("{} {} had zero length!", resource.getClass().getName(), resource.getName());
            }
        });
        List<Resource> binaryResources = this.filterForBinaryAtlasResources(atlasResources);
        List<Resource> textResources = this.filterForTextAtlasResources(atlasResources);
        if (binaryResources.isEmpty() && textResources.isEmpty()) {
            throw new CoreException("No loadable Resources were found.");
        }
        MultiAtlas resultAtlas = null;
        if (!binaryResources.isEmpty()) {
            resultAtlas = MultiAtlas.loadFromPackedAtlas(binaryResources);
        }
        if (!textResources.isEmpty() && !(textAtlases = this.loadTextAtlases(textResources)).isEmpty()) {
            MultiAtlas textMultiAtlas = new MultiAtlas(textAtlases);
            resultAtlas = resultAtlas != null ? new MultiAtlas(resultAtlas, textMultiAtlas) : textMultiAtlas;
        }
        if (this.multiAtlasName != null && resultAtlas != null) {
            resultAtlas.setName(this.multiAtlasName);
        }
        return Optional.ofNullable(resultAtlas);
    }

    private List<Atlas> loadTextAtlases(List<Resource> textAtlasResources) {
        ArrayList<Atlas> textAtlases = new ArrayList<Atlas>();
        for (Resource textResource : textAtlasResources) {
            AtlasResourceLoader.setDecompressorFor(textResource);
            PackedAtlas atlas = new TextAtlasBuilder().read(textResource);
            textAtlases.add(atlas);
        }
        return textAtlases;
    }

    private List<Resource> upcastAndRemoveNullResources(Resource resource) {
        ArrayList<Resource> result = new ArrayList<Resource>();
        if (resource != null) {
            result.add(resource);
        }
        return result;
    }
}

