/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.atlas.utilities.caching.strategies;

import java.net.URI;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Function;
import org.openstreetmap.atlas.streaming.resource.File;
import org.openstreetmap.atlas.streaming.resource.Resource;
import org.openstreetmap.atlas.utilities.caching.strategies.AbstractCachingStrategy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NamespaceCachingStrategy
extends AbstractCachingStrategy {
    private static final Logger logger = LoggerFactory.getLogger(NamespaceCachingStrategy.class);
    private static final String FILE_EXTENSION_DOT = ".";
    private static final String PROPERTY_LOCAL_TEMPORARY_DIRECTORY = "java.io.tmpdir";
    private static final String TEMPORARY_DIRECTORY_STRING = System.getProperty("java.io.tmpdir");
    private final String namespace;

    public NamespaceCachingStrategy(String namespace) {
        if (namespace.contains("/") || namespace.contains("\\")) {
            throw new IllegalArgumentException("The namespace cannot contain characters '\\' or '/'");
        }
        this.namespace = "NamespaceCachingStrategy_" + namespace + "_" + UUID.nameUUIDFromBytes(namespace.getBytes()).toString();
    }

    @Override
    public Optional<Resource> attemptFetch(URI resourceURI, Function<URI, Optional<Resource>> defaultFetcher) {
        if (TEMPORARY_DIRECTORY_STRING == null) {
            logger.error("StrategyID {}: failed to read property {}, skipping cache fetch...", (Object)this.getStrategyID(), (Object)PROPERTY_LOCAL_TEMPORARY_DIRECTORY);
            return Optional.empty();
        }
        if (resourceURI == null) {
            logger.warn("StrategyID {}: resourceURI was null, skipping cache fetch...", (Object)this.getStrategyID());
            return Optional.empty();
        }
        File cachedFile = this.getCachedFile(resourceURI);
        this.attemptToCacheFileLocally(cachedFile, defaultFetcher, resourceURI);
        if (cachedFile.exists()) {
            logger.trace("StrategyID {}: returning local copy of resource {}", (Object)this.getStrategyID(), (Object)resourceURI);
            return Optional.of(cachedFile);
        }
        logger.warn("StrategyID {}: could not find local copy of resource {}", (Object)this.getStrategyID(), (Object)resourceURI);
        return Optional.empty();
    }

    @Override
    public String getName() {
        return "NamespaceCachingStrategy";
    }

    @Override
    public void invalidate() {
        Path storageDirectory = this.getStorageDirectory();
        try {
            new File(storageDirectory.toString()).deleteRecursively();
        }
        catch (Exception exception) {
            logger.warn("StrategyID {}: invalidate failed due to {}", (Object)this.getStrategyID(), (Object)exception);
        }
    }

    @Override
    public void invalidate(URI resourceURI) {
        try {
            this.getCachedFile(resourceURI).delete();
        }
        catch (Exception exception) {
            logger.warn("StrategyID {}: invalidate of resource {} failed due to {}", new Object[]{this.getStrategyID(), resourceURI, exception});
        }
    }

    private void attemptToCacheFileLocally(File cachedFile, Function<URI, Optional<Resource>> defaultFetcher, URI resourceURI) {
        if (!cachedFile.exists()) {
            logger.trace("StrategyID {}: attempting to cache resource {} in temporary file {}", new Object[]{this.getStrategyID(), resourceURI, cachedFile.toString()});
            Optional<Resource> resourceFromDefaultFetcher = defaultFetcher.apply(resourceURI);
            if (!resourceFromDefaultFetcher.isPresent()) {
                logger.warn("StrategyID {}: application of default fetcher for {} returned empty Optional!", (Object)this.getStrategyID(), (Object)resourceURI);
                return;
            }
            File temporaryLocalFile = File.temporary();
            try {
                resourceFromDefaultFetcher.get().copyTo(temporaryLocalFile);
            }
            catch (Exception exception) {
                logger.error("StrategyID {}: something went wrong copying {} to temporary local file {}", new Object[]{this.getStrategyID(), resourceFromDefaultFetcher.toString(), temporaryLocalFile.toString(), exception});
                return;
            }
            if (!cachedFile.exists()) {
                try {
                    Path temporaryLocalFilePath = Paths.get(temporaryLocalFile.getPath(), new String[0]);
                    Path cachedFilePath = Paths.get(cachedFile.getPath(), new String[0]);
                    Files.move(temporaryLocalFilePath, cachedFilePath, StandardCopyOption.ATOMIC_MOVE);
                }
                catch (FileAlreadyExistsException exception) {
                    logger.trace("StrategyID {}: file {} is already cached", (Object)this.getStrategyID(), (Object)cachedFile.toString());
                    return;
                }
                catch (Exception exception) {
                    logger.error("StrategyID {}: something went wrong moving {} to {}", new Object[]{this.getStrategyID(), temporaryLocalFile.toString(), cachedFile.toString(), exception});
                }
            }
        }
    }

    private File getCachedFile(URI resourceURI) {
        Path storageDirectory = this.getStorageDirectory();
        Optional<String> resourceExtension = this.getFileExtensionFromURI(resourceURI);
        String cachedFileName = resourceExtension.isPresent() ? this.getUUIDForResourceURI(resourceURI).toString() + FILE_EXTENSION_DOT + resourceExtension.get() : this.getUUIDForResourceURI(resourceURI).toString();
        Path cachedFilePath = Paths.get(storageDirectory.toString(), cachedFileName);
        return new File(cachedFilePath.toString());
    }

    private Optional<String> getFileExtensionFromURI(URI resourceURI) {
        String asciiString = resourceURI.toASCIIString();
        int lastIndexOfDot = asciiString.lastIndexOf(FILE_EXTENSION_DOT);
        if (lastIndexOfDot < 0) {
            return Optional.empty();
        }
        String extension = asciiString.substring(lastIndexOfDot + 1);
        if (extension.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(extension);
    }

    private Path getStorageDirectory() {
        return Paths.get(TEMPORARY_DIRECTORY_STRING, this.namespace);
    }
}

