/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.atlas.geography.atlas.raw.creation;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import org.openstreetmap.atlas.exception.CoreException;
import org.openstreetmap.atlas.geography.GeometricSurface;
import org.openstreetmap.atlas.geography.Location;
import org.openstreetmap.atlas.geography.MultiPolygon;
import org.openstreetmap.atlas.geography.atlas.Atlas;
import org.openstreetmap.atlas.geography.atlas.AtlasMetaData;
import org.openstreetmap.atlas.geography.atlas.builder.AtlasSize;
import org.openstreetmap.atlas.geography.atlas.builder.RelationBean;
import org.openstreetmap.atlas.geography.atlas.items.AtlasEntity;
import org.openstreetmap.atlas.geography.atlas.items.AtlasObject;
import org.openstreetmap.atlas.geography.atlas.items.ItemType;
import org.openstreetmap.atlas.geography.atlas.items.Point;
import org.openstreetmap.atlas.geography.atlas.packed.PackedAtlasBuilder;
import org.openstreetmap.atlas.geography.atlas.pbf.AtlasLoadingOption;
import org.openstreetmap.atlas.geography.atlas.pbf.CloseableOsmosisReader;
import org.openstreetmap.atlas.geography.atlas.raw.creation.OsmPbfCounter;
import org.openstreetmap.atlas.geography.atlas.raw.creation.OsmPbfReader;
import org.openstreetmap.atlas.streaming.resource.Resource;
import org.openstreetmap.atlas.streaming.resource.WritableResource;
import org.openstreetmap.atlas.tags.AtlasTag;
import org.openstreetmap.atlas.tags.LayerTag;
import org.openstreetmap.atlas.tags.SyntheticDuplicateOsmNodeTag;
import org.openstreetmap.atlas.utilities.collections.Iterables;
import org.openstreetmap.atlas.utilities.time.Time;
import org.openstreetmap.osmosis.core.task.v0_6.Sink;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RawAtlasGenerator {
    private static final Logger logger = LoggerFactory.getLogger(RawAtlasGenerator.class);
    private final OsmPbfCounter pbfCounter;
    private final OsmPbfReader pbfReader;
    private final GeometricSurface boundingBox;
    private final PackedAtlasBuilder builder;
    private final AtlasLoadingOption atlasLoadingOption;
    private final Supplier<CloseableOsmosisReader> osmosisReaderSupplier;
    private AtlasMetaData metaData = new AtlasMetaData();

    public RawAtlasGenerator(Resource resource) {
        this(resource, AtlasLoadingOption.createOptionWithOnlySectioning(), MultiPolygon.MAXIMUM);
    }

    public RawAtlasGenerator(Resource resource, AtlasLoadingOption loadingOption, MultiPolygon boundingBox) {
        this(() -> new CloseableOsmosisReader(resource.read()), loadingOption, (GeometricSurface)boundingBox);
    }

    public RawAtlasGenerator(Resource resource, MultiPolygon boundingBox) {
        this(resource, AtlasLoadingOption.createOptionWithNoSlicing(), boundingBox);
    }

    public RawAtlasGenerator(Supplier<CloseableOsmosisReader> osmosisReaderSupplier, AtlasLoadingOption atlasLoadingOption, GeometricSurface boundingBox) {
        this.osmosisReaderSupplier = osmosisReaderSupplier;
        this.atlasLoadingOption = atlasLoadingOption;
        this.boundingBox = boundingBox;
        this.builder = new PackedAtlasBuilder();
        this.pbfReader = new OsmPbfReader(atlasLoadingOption, this.builder);
        this.pbfCounter = new OsmPbfCounter(atlasLoadingOption, this.boundingBox);
    }

    public Atlas build() {
        this.countOsmPbfEntities();
        this.populateAtlasMetadata();
        this.setAtlasSizeEstimate();
        this.pbfReader.setIncludedNodes(this.pbfCounter.getIncludedNodeIdentifiers());
        this.pbfReader.setIncludedWays(this.pbfCounter.getIncludedWayIdentifiers());
        return this.buildRawAtlas();
    }

    public void saveAsGeojson(WritableResource resource) {
        logger.info("Saving Raw Atlas as geojson");
        this.build().saveAsGeoJson(resource);
    }

    public void saveAsText(WritableResource resource) {
        logger.info("Saving Raw Atlas as text");
        this.build().saveAsText(resource);
    }

    public void saveAtlas(WritableResource resource) {
        logger.info("Saving Raw Atlas file");
        this.build().save(resource);
    }

    public RawAtlasGenerator withMetaData(AtlasMetaData metaData) {
        this.metaData = metaData;
        return this;
    }

    private Atlas buildRawAtlas() {
        String shardName = this.metaData.getShardName().orElse("unknown");
        Time parseTime = Time.now();
        try (CloseableOsmosisReader reader = this.connectOsmPbfToPbfConsumer(this.pbfReader);){
            reader.run();
        }
        catch (Exception e) {
            throw new CoreException("Atlas creation error for PBF shard {}", shardName, e);
        }
        logger.info("Read PBF for {} in {}", (Object)shardName, (Object)parseTime.elapsedSince());
        Time buildTime = Time.now();
        Atlas atlas = this.builder.get();
        logger.info("Built Raw Atlas for {} in {}", (Object)shardName, (Object)buildTime.elapsedSince());
        if (atlas == null) {
            logger.info("Generated empty raw Atlas for PBF Shard {}", (Object)shardName);
            return atlas;
        }
        Time trimTime = Time.now();
        Atlas trimmedAtlas = this.removeDuplicateAndExtraneousPointsFromAtlas(atlas);
        logger.info("Trimmed Raw Atlas for {} in {}", (Object)shardName, (Object)trimTime.elapsedSince());
        if (trimmedAtlas == null) {
            logger.info("Empty raw Atlas after filtering for PBF Shard {}", (Object)shardName);
        }
        return trimmedAtlas;
    }

    private CloseableOsmosisReader connectOsmPbfToPbfConsumer(Sink consumer) {
        CloseableOsmosisReader reader = this.osmosisReaderSupplier.get();
        reader.setSink(consumer);
        return reader;
    }

    private void countOsmPbfEntities() {
        Time countTime = Time.now();
        try (CloseableOsmosisReader counter = this.connectOsmPbfToPbfConsumer(this.pbfCounter);){
            counter.run();
        }
        catch (Exception e) {
            throw new CoreException("Error counting PBF entities", e);
        }
        logger.info("Counted PBF Entities in {}", (Object)countTime.elapsedSince());
    }

    private long getLayerTagValueForPoint(Atlas atlas, long identifier) {
        return LayerTag.getTaggedOrImpliedValue(atlas.point(identifier), 0L);
    }

    private boolean isRelationMember(Atlas atlas, long pointIdentifier) {
        return !atlas.point(pointIdentifier).relations().isEmpty();
    }

    private boolean isShapePoint(Atlas atlas, long pointIdentifier) {
        return Iterables.size(atlas.linesContaining(atlas.point(pointIdentifier).getLocation())) > 0L;
    }

    private boolean isSimplePoint(Atlas atlas, long pointIdentifier) {
        return atlas.point(pointIdentifier).getTags().size() == AtlasTag.TAGS_FROM_OSM.size();
    }

    private boolean locationPartOfMultipleWaysWithDifferentLayerTags(Atlas atlas, Location location) {
        long distinctLayerTagValues = StreamSupport.stream(atlas.linesContaining(location).spliterator(), false).map(line -> LayerTag.getTaggedOrImpliedValue(atlas.line(line.getIdentifier()), 0L)).distinct().count();
        return distinctLayerTagValues > 1L;
    }

    private void populateAtlasMetadata() {
        this.metaData.getTags().put("osmPbfNodeConfiguration", this.atlasLoadingOption.getOsmPbfNodeFilter().toString());
        this.metaData.getTags().put("osmPbfWayConfiguration", this.atlasLoadingOption.getOsmPbfWayFilter().toString());
        this.metaData.getTags().put("osmPbfRelationConfiguration", this.atlasLoadingOption.getOsmPbfRelationFilter().toString());
        this.builder.setMetaData(this.metaData);
    }

    private Set<Long> preFilterPointsToRemove(Atlas atlas) {
        return this.pbfReader.getPointIdentifiersFromFilteredLines().stream().filter(identifier -> atlas.point((long)identifier) != null).filter(identifier -> this.isSimplePoint(atlas, (long)identifier)).filter(identifier -> !this.isRelationMember(atlas, (long)identifier)).filter(identifier -> !this.isShapePoint(atlas, (long)identifier)).collect(Collectors.toSet());
    }

    private Atlas rebuildAtlas(Atlas atlas, Set<Long> pointsToRemove, Set<Long> pointsNeedingSyntheticTag) {
        PackedAtlasBuilder rebuilder = new PackedAtlasBuilder();
        rebuilder.setMetaData(this.metaData);
        AtlasSize size = new AtlasSize(0L, 0L, 0L, atlas.numberOfLines(), atlas.numberOfPoints(), atlas.numberOfRelations());
        rebuilder.setSizeEstimates(size);
        atlas.points().forEach(point -> {
            long identifier = point.getIdentifier();
            if (!pointsToRemove.contains(identifier)) {
                Map<String, String> tags = point.getTags();
                if (pointsNeedingSyntheticTag.contains(identifier)) {
                    tags.put("synthetic_duplicate_osm_node", SyntheticDuplicateOsmNodeTag.YES.toString());
                }
                rebuilder.addPoint(identifier, point.getLocation(), tags);
            }
        });
        atlas.lines().forEach(line -> rebuilder.addLine(line.getIdentifier(), line.asPolyLine(), line.getTags()));
        HashSet relationsToCheckForRemoval = new HashSet();
        atlas.relationsLowerOrderFirst().forEach(relation -> {
            RelationBean bean = new RelationBean();
            relation.members().forEach(member -> {
                AtlasEntity entity = member.getEntity();
                long memberIdentifier = entity.getIdentifier();
                if (entity.getType() == ItemType.POINT && pointsToRemove.contains(memberIdentifier)) {
                    logger.debug("Excluding point {} from relation {} since point was removed from Atlas", (Object)memberIdentifier, (Object)relation.getIdentifier());
                } else if (entity.getType() == ItemType.RELATION && relationsToCheckForRemoval.contains(memberIdentifier)) {
                    logger.debug("Excluding relation member {} from parent relation {} since that relation member became empty", (Object)memberIdentifier, (Object)relation.getIdentifier());
                } else {
                    bean.addItem(memberIdentifier, member.getRole(), entity.getType());
                }
            });
            if (!bean.isEmpty()) {
                rebuilder.addRelation(relation.getIdentifier(), relation.getOsmIdentifier(), bean, relation.getTags());
            } else {
                long relationIdentifier = relation.getIdentifier();
                logger.debug("Relation {} bean is empty, dropping from Atlas", (Object)relationIdentifier);
                relationsToCheckForRemoval.add(relationIdentifier);
            }
        });
        return rebuilder.get();
    }

    private Atlas removeDuplicateAndExtraneousPointsFromAtlas(Atlas atlas) {
        HashSet<Long> pointsToRemove = new HashSet<Long>();
        HashSet<Long> duplicatePointsToKeep = new HashSet<Long>();
        for (Point point : atlas.points()) {
            Set<Long> duplicatePoints;
            if (pointsToRemove.contains(point.getIdentifier()) || duplicatePointsToKeep.contains(point.getIdentifier()) || (duplicatePoints = Iterables.stream(atlas.pointsAt(point.getLocation())).map(AtlasObject::getIdentifier).collectToSet()).isEmpty() || duplicatePoints.size() <= 1) continue;
            if (this.locationPartOfMultipleWaysWithDifferentLayerTags(atlas, point.getLocation())) {
                duplicatePointsToKeep.addAll(duplicatePoints);
                continue;
            }
            SortedSet<Long> sortedDuplicates = Iterables.asSortedSet(duplicatePoints);
            HashSet<Long> uniqueLayerValues = new HashSet<Long>();
            Iterator duplicateIterator = sortedDuplicates.iterator();
            long duplicatePointToKeep = (Long)duplicateIterator.next();
            long layerValue = this.getLayerTagValueForPoint(atlas, duplicatePointToKeep);
            duplicatePointsToKeep.add(duplicatePointToKeep);
            duplicateIterator.remove();
            uniqueLayerValues.add(layerValue);
            while (duplicateIterator.hasNext()) {
                long candidateToKeep = (Long)duplicateIterator.next();
                long candidateLayerValue = this.getLayerTagValueForPoint(atlas, candidateToKeep);
                if (uniqueLayerValues.contains(candidateLayerValue)) continue;
                duplicatePointsToKeep.add(candidateToKeep);
                duplicateIterator.remove();
            }
            pointsToRemove.addAll(sortedDuplicates);
        }
        if (!this.pbfReader.getPointIdentifiersFromFilteredLines().isEmpty()) {
            pointsToRemove.addAll(this.preFilterPointsToRemove(atlas));
        }
        if (pointsToRemove.isEmpty()) {
            return atlas;
        }
        return this.rebuildAtlas(atlas, pointsToRemove, duplicatePointsToKeep);
    }

    private void setAtlasSizeEstimate() {
        AtlasSize size = new AtlasSize(0L, 0L, 0L, this.pbfCounter.lineCount(), this.pbfCounter.pointCount(), this.pbfCounter.relationCount());
        this.builder.setSizeEstimates(size);
    }
}

