/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.atlas.utilities.timezone;

import java.io.IOException;
import java.net.URL;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.TimeZone;
import java.util.regex.Pattern;
import org.geotools.data.FileDataStore;
import org.geotools.data.FileDataStoreFinder;
import org.geotools.data.simple.SimpleFeatureIterator;
import org.opengis.feature.Feature;
import org.opengis.feature.Property;
import org.opengis.geometry.BoundingBox;
import org.openstreetmap.atlas.geography.Latitude;
import org.openstreetmap.atlas.geography.Located;
import org.openstreetmap.atlas.geography.Location;
import org.openstreetmap.atlas.geography.Longitude;
import org.openstreetmap.atlas.geography.Polygon;
import org.openstreetmap.atlas.geography.Rectangle;
import org.openstreetmap.atlas.geography.index.JtsSpatialIndex;
import org.openstreetmap.atlas.geography.index.RTree;
import org.openstreetmap.atlas.utilities.conversion.Converter;
import org.openstreetmap.atlas.utilities.scalars.Distance;
import org.openstreetmap.atlas.utilities.scalars.Surface;
import org.openstreetmap.atlas.utilities.statistic.storeless.CounterWithStatistic;
import org.openstreetmap.atlas.utilities.timezone.TimeZoneBoundary;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TimeZoneMap
implements Located {
    private static final Distance SEA_TERRITORY_ZONE_WITH_BUFFER = Distance.SEA_TERRITORY_ZONE.scaleBy(2.0);
    private static final Logger logger = LoggerFactory.getLogger(TimeZoneMap.class);
    private static final int DEGREES_PER_TIME_ZONE = 15;
    private static final int LOG_PRINT_FREQUENCY = 1000;
    private final JtsSpatialIndex<TimeZoneBoundary> index = new RTree<TimeZoneBoundary>();
    private final Rectangle bound;

    public TimeZoneMap() {
        this(Rectangle.MAXIMUM);
    }

    public TimeZoneMap(Rectangle bound) {
        this.bound = bound;
        try {
            this.loadTimeZoneBoundaries();
        }
        catch (IOException e) {
            logger.error("Errors happened when loading timezone", (Throwable)e);
        }
    }

    public List<TimeZoneBoundary> allTimeZoneBoundaries() {
        return this.index.get(Rectangle.MAXIMUM);
    }

    @Override
    public Rectangle bounds() {
        return this.bound;
    }

    public synchronized TimeZone timeZone(Location location) {
        List<TimeZoneBoundary> boundaries = this.index.get(location.bounds(), boundary -> boundary.getPolygon().fullyGeometricallyEncloses(location));
        if (boundaries.size() >= 1) {
            return this.smallest(boundaries).getTimeZone();
        }
        Rectangle larger = location.bounds().expand(SEA_TERRITORY_ZONE_WITH_BUFFER);
        boundaries = this.index.get(larger);
        if (boundaries.size() >= 1) {
            return boundaries.get(0).getTimeZone();
        }
        int offset = (int)location.getLongitude().asDegrees() / 15;
        return TimeZone.getTimeZone(ZoneOffset.of(offset > 0 ? "+" + offset : "" + offset));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void loadTimeZoneBoundaries() throws IOException {
        CounterWithStatistic counter = new CounterWithStatistic(logger);
        counter.setLogPrintFrequency(1000L);
        URL url = TimeZoneMap.class.getResource("tz_world.shp");
        FileDataStore store = FileDataStoreFinder.getDataStore(url);
        TimeZoneBoundaryConverter converter = new TimeZoneBoundaryConverter();
        SimpleFeatureIterator iterator = store.getFeatureSource().getFeatures().features();
        try {
            while (iterator.hasNext()) {
                Object feature = iterator.next();
                BoundingBox boundingBox = feature.getBounds();
                Rectangle featureBound = Rectangle.forLocations(new Location(Latitude.degrees(boundingBox.getMinY()), Longitude.degrees(boundingBox.getMinX())), new Location(Latitude.degrees(boundingBox.getMaxY()), Longitude.degrees(boundingBox.getMaxX())));
                if (!this.bound.overlaps(featureBound)) continue;
                TimeZoneBoundary boundary = converter.convert((Feature)feature);
                this.index.add(boundary.bounds(), boundary);
                counter.increment();
            }
        }
        finally {
            iterator.close();
            store.dispose();
        }
        counter.summary();
        logger.info("index size is {}", (Object)this.index.size());
    }

    private TimeZoneBoundary smallest(List<TimeZoneBoundary> boundaries) {
        if (boundaries.size() == 1) {
            return boundaries.get(0);
        }
        TimeZoneBoundary target = null;
        Surface minimum = Surface.MAXIMUM;
        for (TimeZoneBoundary boundary : boundaries) {
            Surface area = boundary.getPolygon().bounds().surface();
            if (!area.isLessThan(minimum)) continue;
            minimum = area;
            target = boundary;
        }
        return target;
    }

    private static class TimeZoneBoundaryConverter
    implements Converter<Feature, TimeZoneBoundary> {
        private static final int PREFIX_LENGTH = "MULTIPOLYGON (((".length();
        private static final int SUFFIX_LENGTH = ")))".length();
        private static final Pattern MULTI_POLYGON_SEPARATOR = Pattern.compile("\\), \\(");
        private static final Pattern POINT_SEPARATOR = Pattern.compile(", ");
        private static final Pattern LATITUDE_LONGITUDE_SEPARATOR = Pattern.compile(" ");

        private TimeZoneBoundaryConverter() {
        }

        @Override
        public TimeZoneBoundary convert(Feature feature) {
            Collection<Property> timeZoneIdentifiers = feature.getProperties("TZID");
            if (timeZoneIdentifiers.isEmpty() || timeZoneIdentifiers.size() != 1) {
                logger.error("feature {} has more than two time zone identifiers", (Object)feature);
            }
            TimeZone timeZone = TimeZone.getTimeZone(timeZoneIdentifiers.iterator().next().getValue().toString());
            Collection<Property> geometries = feature.getProperties("the_geom");
            if (geometries.isEmpty() || geometries.size() != 1) {
                logger.error("feature {} has more than two geometries", (Object)feature);
            }
            String geometry = geometries.iterator().next().getValue().toString();
            String[] multiPolygons = MULTI_POLYGON_SEPARATOR.split(geometry.substring(PREFIX_LENGTH, geometry.length() - SUFFIX_LENGTH));
            String[] points = POINT_SEPARATOR.split(multiPolygons[0]);
            ArrayList<Location> locations = new ArrayList<Location>(points.length);
            for (int j = 0; j < points.length; ++j) {
                String[] coordinate = LATITUDE_LONGITUDE_SEPARATOR.split(points[j]);
                locations.add(new Location(Latitude.degrees(Double.parseDouble(coordinate[1])), Longitude.degrees(Double.parseDouble(coordinate[0]))));
            }
            return new TimeZoneBoundary(timeZone, new Polygon((List<Location>)locations));
        }
    }
}

