/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.atlas.geography.sharding;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Queue;
import java.util.Set;
import org.openstreetmap.atlas.exception.CoreException;
import org.openstreetmap.atlas.geography.Latitude;
import org.openstreetmap.atlas.geography.Location;
import org.openstreetmap.atlas.geography.Longitude;
import org.openstreetmap.atlas.geography.Rectangle;
import org.openstreetmap.atlas.geography.sharding.Shard;
import org.openstreetmap.atlas.geography.sharding.SlippyTileSharding;
import org.openstreetmap.atlas.geography.sharding.converters.SlippyTileConverter;
import org.openstreetmap.atlas.utilities.collections.Iterables;
import org.openstreetmap.atlas.utilities.scalars.Distance;

public class SlippyTile
implements Shard,
Comparable<SlippyTile> {
    private static final long serialVersionUID = -3752920878013084039L;
    public static final SlippyTile ROOT = new SlippyTile(0, 0, 0);
    public static final int MAX_ZOOM = 30;
    private static final SlippyTileConverter CONVERTER = new SlippyTileConverter();
    private static final double CIRCULAR_MULTIPLIER = 2.0;
    private static final double ZOOM_LEVEL_POWER = 2.0;
    private static final int BIT_SHIFT = 2;
    private static final double FULL_ROTATION_DEGREES = 360.0;
    private static final double HALF_ROTATION_DEGREES = 180.0;
    private static final double LONGITUDE_BOUNDARY = 180.0;
    private static final double NEIGHBOR_EXPANSION_SCALE = 0.1;
    private Rectangle bounds;
    private final int xAxis;
    private final int yAxis;
    private final int zoom;

    public static Iterable<SlippyTile> allTiles(int zoom) {
        return SlippyTile.allTiles(zoom, Rectangle.MAXIMUM);
    }

    public static Iterable<SlippyTile> allTiles(int zoom, Rectangle bounds) {
        if (zoom > 30) {
            throw new CoreException("Zoom too large.");
        }
        Iterable result = () -> SlippyTile.allTilesIterator(zoom, bounds);
        List<SlippyTile> list = Iterables.asList(result);
        if (list.isEmpty()) {
            throw new CoreException("List cannot be empty");
        }
        return list;
    }

    public static Iterator<SlippyTile> allTilesIterator(final int zoom, Rectangle bounds) {
        if (zoom > 30) {
            throw new CoreException("Zoom too large.");
        }
        SlippyTile lowerLeft = new SlippyTile(bounds.lowerLeft(), zoom);
        SlippyTile upperRight = new SlippyTile(bounds.upperRight(), zoom);
        final int minX = lowerLeft.getX();
        final int maxX = upperRight.getX();
        final int minY = upperRight.getY();
        final int maxY = lowerLeft.getY();
        Iterator<SlippyTile> result = new Iterator<SlippyTile>(){
            private int xAxis;
            private int yAxis;
            {
                this.xAxis = minX;
                this.yAxis = minY;
            }

            @Override
            public boolean hasNext() {
                return this.yAxis <= maxY && this.xAxis <= maxX;
            }

            @Override
            public SlippyTile next() {
                if (this.yAxis > maxY) {
                    return null;
                }
                SlippyTile result = new SlippyTile(this.xAxis, this.yAxis, zoom);
                ++this.xAxis;
                if (this.xAxis > maxX) {
                    ++this.yAxis;
                    this.xAxis = minX;
                }
                return result;
            }
        };
        return result;
    }

    public static Distance calculateExpansionDistance(Rectangle bounds) {
        Distance shorterSide = bounds.width().onEarth().isLessThanOrEqualTo(bounds.height().onEarth()) ? bounds.width().onEarth() : bounds.height().onEarth();
        return shorterSide.scaleBy(0.1);
    }

    public static SlippyTile forName(String name) {
        return CONVERTER.backwardConvert(name);
    }

    public SlippyTile(int xAxis, int yAxis, int zoom) {
        if (zoom > 30) {
            throw new CoreException("Zoom {} is too large.", zoom);
        }
        this.zoom = zoom;
        this.xAxis = xAxis;
        this.yAxis = yAxis;
    }

    public SlippyTile(Location location, int zoom) {
        if (zoom > 30) {
            throw new CoreException("Zoom {} is too large.", zoom);
        }
        List<Integer> tileNumbers = this.getTileNumbers(location, zoom);
        this.zoom = zoom;
        this.xAxis = tileNumbers.get(0);
        this.yAxis = tileNumbers.get(1);
    }

    @Override
    public Rectangle bounds() {
        if (this.bounds == null) {
            this.bounds = this.tile2boundingBox(this.xAxis, this.yAxis, this.zoom);
        }
        return this.bounds;
    }

    @Override
    public int compareTo(SlippyTile other) {
        int zoomLevelDelta = this.getZoom() - other.getZoom();
        if (zoomLevelDelta > 0) {
            return 1;
        }
        if (zoomLevelDelta < 0) {
            return -1;
        }
        int xDelta = this.getX() - other.getX();
        if (xDelta > 0) {
            return 1;
        }
        if (xDelta < 0) {
            return -1;
        }
        int yDelta = this.getY() - other.getY();
        if (yDelta > 0) {
            return 1;
        }
        if (yDelta < 0) {
            return -1;
        }
        return 0;
    }

    public boolean equals(Object other) {
        if (other instanceof SlippyTile) {
            SlippyTile that = (SlippyTile)other;
            return this.getZoom() == that.getZoom() && this.getX() == that.getX() && this.getY() == that.getY();
        }
        return false;
    }

    @Override
    public String getName() {
        return CONVERTER.convert(this);
    }

    public int getX() {
        return this.xAxis;
    }

    public int getY() {
        return this.yAxis;
    }

    public int getZoom() {
        return this.zoom;
    }

    public int hashCode() {
        long result = (long)Math.pow(this.getMaxXorY(this.zoom - 1), 2.0) + (long)Math.pow(this.xAxis, 2.0) + (long)this.yAxis;
        return (int)result % Integer.MAX_VALUE;
    }

    public Set<SlippyTile> neighbors() {
        return Iterables.stream(new SlippyTileSharding(this.zoom).shards(this.bounds().expand(SlippyTile.calculateExpansionDistance(this.bounds())))).map(shard -> (SlippyTile)shard).filter(tile -> !this.equals(tile)).collectToSet();
    }

    public SlippyTile parent() {
        if (this.zoom == 0) {
            return this;
        }
        int parentZoom = this.zoom - 1;
        int parentX = this.xAxis / 2;
        int parentY = this.yAxis / 2;
        return new SlippyTile(parentX, parentY, parentZoom);
    }

    public List<SlippyTile> split() {
        if (this.zoom == 30) {
            throw new CoreException("Cannot split further than zoom {}", 30);
        }
        ArrayList<SlippyTile> result = new ArrayList<SlippyTile>();
        for (int i = 2 * this.xAxis; i <= 2 * this.xAxis + 1; ++i) {
            for (int j = 2 * this.yAxis; j <= 2 * this.yAxis + 1; ++j) {
                result.add(new SlippyTile(i, j, this.zoom + 1));
            }
        }
        return result;
    }

    public List<SlippyTile> split(int newZoom) {
        int temporaryZoom;
        Serializable temporary;
        if (newZoom < 0 || newZoom > 30) {
            throw new CoreException("Cannot split to a zoom {} which is not between 0 and {} included.", newZoom, 30);
        }
        Serializable result = new ArrayList<SlippyTile>();
        if (newZoom == this.zoom) {
            result.add(this);
        }
        if (newZoom > this.zoom) {
            temporary = new ArrayList<SlippyTile>();
            temporary.add(this);
            for (temporaryZoom = this.zoom + 1; temporaryZoom <= newZoom; ++temporaryZoom) {
                ArrayList<SlippyTile> newTemporary = new ArrayList<SlippyTile>();
                Iterator iterator = temporary.iterator();
                while (iterator.hasNext()) {
                    SlippyTile temporaryTile = (SlippyTile)iterator.next();
                    newTemporary.addAll(temporaryTile.split());
                }
                temporary = newTemporary;
            }
            result = temporary;
        }
        if (newZoom < this.zoom) {
            temporary = this;
            for (temporaryZoom = ((SlippyTile)temporary).getZoom() - 1; temporaryZoom >= newZoom; --temporaryZoom) {
                temporary = ((SlippyTile)temporary).parent();
            }
            result.add((SlippyTile)temporary);
        }
        return result;
    }

    public String toString() {
        return "[SlippyTile: zoom = " + this.zoom + ", x = " + this.xAxis + ", y = " + this.yAxis + "]";
    }

    protected void getNeighborsForAllZoomLevels(Queue<SlippyTile> candidates, Set<String> visitedTiles, SlippyTile targetTile) {
        SlippyTile parent = targetTile.parent();
        for (SlippyTile child : parent.split()) {
            if (visitedTiles.contains(child.getName()) || child.equals(targetTile)) continue;
            candidates.add(child);
        }
        candidates.add(parent);
    }

    private int getMaxXorY(int zoom) {
        if (zoom <= 0) {
            return 1;
        }
        int result = this.getTileNumbers(new Location(Latitude.MINIMUM, Longitude.ZERO), zoom).get(0);
        return result;
    }

    private List<Integer> getTileNumbers(Location location, int zoom) {
        double latitude = location.getLatitude().asDegrees();
        double longitude = location.getLongitude().asDegrees();
        int xAxis = (int)Math.floor((longitude + 180.0) / 360.0 * (double)(1 << zoom));
        int yAxis = (int)Math.floor((1.0 - Math.log(Math.tan(Math.toRadians(latitude)) + 1.0 / Math.cos(Math.toRadians(latitude))) / Math.PI) / 2.0 * (double)(1 << zoom));
        if (xAxis < 0) {
            xAxis = 0;
        }
        if (xAxis >= 1 << zoom) {
            xAxis = (1 << zoom) - 1;
        }
        if (yAxis < 0) {
            yAxis = 0;
        }
        if (yAxis >= 1 << zoom) {
            yAxis = (1 << zoom) - 1;
        }
        ArrayList<Integer> result = new ArrayList<Integer>();
        result.add(xAxis);
        result.add(yAxis);
        return result;
    }

    private Rectangle tile2boundingBox(int xAxis, int yAxis, int zoom) {
        Latitude minLat = this.tile2lat(yAxis + 1, zoom);
        Latitude maxLat = this.tile2lat(yAxis, zoom);
        Longitude minLon = this.tile2lon(xAxis, zoom);
        Longitude maxLon = this.tile2lon(xAxis + 1, zoom);
        return Rectangle.forCorners(new Location(minLat, minLon), new Location(maxLat, maxLon));
    }

    private Latitude tile2lat(int yAxis, int zoom) {
        double pivot = Math.PI - Math.PI * 2 * (double)yAxis / Math.pow(2.0, zoom);
        return Latitude.degrees(Math.toDegrees(Math.atan(Math.sinh(pivot))));
    }

    private Longitude tile2lon(int xAxis, int zoom) {
        double longitude = (double)xAxis / Math.pow(2.0, zoom) * 360.0 - 180.0;
        return longitude >= 180.0 ? Longitude.MAXIMUM : (longitude < -180.0 ? Longitude.MINIMUM : Longitude.degrees(longitude));
    }
}

