/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.atlas.checks.validation.intersections;

import com.vividsolutions.jts.geom.TopologyException;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import org.openstreetmap.atlas.checks.base.BaseCheck;
import org.openstreetmap.atlas.checks.flag.CheckFlag;
import org.openstreetmap.atlas.geography.PolyLine;
import org.openstreetmap.atlas.geography.Polygon;
import org.openstreetmap.atlas.geography.atlas.items.Area;
import org.openstreetmap.atlas.geography.atlas.items.AtlasObject;
import org.openstreetmap.atlas.geography.clipping.Clip;
import org.openstreetmap.atlas.tags.BuildingTag;
import org.openstreetmap.atlas.tags.Taggable;
import org.openstreetmap.atlas.utilities.configuration.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IntersectingBuildingsCheck
extends BaseCheck<String> {
    private static final Logger logger = LoggerFactory.getLogger(IntersectingBuildingsCheck.class);
    private static final long serialVersionUID = 5796448445672515517L;
    private static final String INTERSECT_INSTRUCTION = "Building (id={0,number,#}) intersects with another building (id={1,number,#}).";
    private static final String OVERLAP_INSTRUCTION = "Building (id={0,number,#}) is overlapped by another building (id={1,number,#}).";
    private static final List<String> FALLBACK_INSTRUCTIONS = Arrays.asList("Building (id={0,number,#}) is overlapped by another building (id={1,number,#}).", "Building (id={0,number,#}) intersects with another building (id={1,number,#}).");
    private static final double INTERSECTION_LOWER_LIMIT_DEFAULT = 0.01;
    private static final double OVERLAP_LOWER_LIMIT_DEFAULT = 0.9;
    private static final int MINIMUM_POINT_COUNT_FOR_POLYGON = 3;
    private static final String UNIQUE_IDENTIFIER_FORMAT = "%s,%s";
    private final double intersectionLowerLimit;
    private final double overlapLowerLimit;

    private static String getIdentifierTuple(Area area, Area otherArea) {
        return area.getIdentifier() < otherArea.getIdentifier() ? String.format(UNIQUE_IDENTIFIER_FORMAT, area.getIdentifier(), otherArea.getIdentifier()) : String.format(UNIQUE_IDENTIFIER_FORMAT, otherArea.getIdentifier(), area.getIdentifier());
    }

    public IntersectingBuildingsCheck(Configuration configuration) {
        super(configuration);
        this.intersectionLowerLimit = this.configurationValue(configuration, "intersection.lower.limit", 0.01, Double::valueOf);
        this.overlapLowerLimit = this.configurationValue(configuration, "overlap.lower.limit", 0.9, Double::valueOf);
    }

    @Override
    public boolean validCheckForObject(AtlasObject object) {
        return object instanceof Area && BuildingTag.isBuilding((Taggable)object);
    }

    @Override
    protected Optional<CheckFlag> flag(AtlasObject object) {
        Area building = (Area)object;
        Polygon buildingPolygon = building.asPolygon();
        if (buildingPolygon.size() < 3) {
            return Optional.empty();
        }
        Iterable possiblyIntersectingBuildings = object.getAtlas().areasIntersecting((Polygon)building.bounds(), area -> BuildingTag.isBuilding((Taggable)area) && building.getIdentifier() != area.getIdentifier() && area.intersects(buildingPolygon));
        CheckFlag flag = new CheckFlag(this.getTaskIdentifier(object));
        flag.addObject(object);
        boolean hadIntersection = false;
        for (Area otherBuilding : possiblyIntersectingBuildings) {
            Polygon otherBuildingsPolygon = otherBuilding.asPolygon();
            if (otherBuildingsPolygon.size() < 3) continue;
            String uniqueIdentifier = IntersectingBuildingsCheck.getIdentifierTuple(building, otherBuilding);
            if (this.isFlagged(IntersectingBuildingsCheck.getIdentifierTuple(building, otherBuilding))) continue;
            IntersectionType resultType = this.findIntersectionType(buildingPolygon, otherBuildingsPolygon);
            if (resultType == IntersectionType.OVERLAP) {
                flag.addObject((AtlasObject)otherBuilding, this.getLocalizedInstruction(0, object.getOsmIdentifier(), otherBuilding.getOsmIdentifier()));
                this.markAsFlagged(uniqueIdentifier);
                hadIntersection = true;
                continue;
            }
            if (resultType != IntersectionType.INTERSECT) continue;
            flag.addObject((AtlasObject)otherBuilding, this.getLocalizedInstruction(1, object.getOsmIdentifier(), otherBuilding.getOsmIdentifier()));
            this.markAsFlagged(uniqueIdentifier);
            hadIntersection = true;
        }
        if (hadIntersection) {
            return Optional.of(flag);
        }
        return Optional.empty();
    }

    @Override
    protected List<String> getFallbackInstructions() {
        return FALLBACK_INSTRUCTIONS;
    }

    private IntersectionType findIntersectionType(Polygon polygon, Polygon otherPolygon) {
        Clip clip = null;
        try {
            clip = polygon.clip(otherPolygon, Clip.ClipType.AND);
        }
        catch (TopologyException e) {
            logger.warn(String.format("Skipping intersection check. Error clipping [%s] and [%s].", polygon, otherPolygon), (Throwable)e);
        }
        if (clip == null) {
            return IntersectionType.NONE;
        }
        long intersectionArea = 0L;
        for (PolyLine polyline : clip.getClip()) {
            if (polyline == null || !(polyline instanceof Polygon)) continue;
            Polygon clippedPolygon = (Polygon)polyline;
            intersectionArea += clippedPolygon.surface().asDm7Squared();
        }
        if (intersectionArea == 0L) {
            return IntersectionType.NONE;
        }
        long baselineArea = Math.min(polygon.surface().asDm7Squared(), otherPolygon.surface().asDm7Squared());
        double proportion = (double)intersectionArea / (double)baselineArea;
        if (proportion >= this.overlapLowerLimit) {
            return IntersectionType.OVERLAP;
        }
        if (proportion >= this.intersectionLowerLimit) {
            return IntersectionType.INTERSECT;
        }
        return IntersectionType.NONE;
    }

    public static enum IntersectionType {
        NONE,
        INTERSECT,
        OVERLAP;

    }
}

