/*
 * Decompiled with CFR 0.152.
 */
package com.amazon.dlic.auth.http.saml;

import com.amazon.dlic.auth.http.saml.Saml2SettingsProvider;
import com.amazon.dlic.auth.http.saml.SamlConfigException;
import com.amazon.dlic.auth.http.saml.SamlNameIdFormat;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.base.Strings;
import com.onelogin.saml2.authn.SamlResponse;
import com.onelogin.saml2.exception.SettingsException;
import com.onelogin.saml2.exception.ValidationError;
import com.onelogin.saml2.settings.Saml2Settings;
import com.onelogin.saml2.util.Util;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.AccessController;
import java.security.Permission;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPathExpressionException;
import org.apache.commons.lang3.StringUtils;
import org.apache.cxf.jaxrs.json.basic.JsonMapObject;
import org.apache.cxf.jaxrs.json.basic.JsonMapObjectReaderWriter;
import org.apache.cxf.rs.security.jose.jwk.JsonWebKey;
import org.apache.cxf.rs.security.jose.jwk.KeyType;
import org.apache.cxf.rs.security.jose.jwk.PublicKeyUse;
import org.apache.cxf.rs.security.jose.jws.JwsUtils;
import org.apache.cxf.rs.security.jose.jwt.JoseJwtProducer;
import org.apache.cxf.rs.security.jose.jwt.JwtClaims;
import org.apache.cxf.rs.security.jose.jwt.JwtToken;
import org.apache.cxf.rs.security.jose.jwt.JwtUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.joda.time.DateTime;
import org.opensearch.OpenSearchSecurityException;
import org.opensearch.SpecialPermission;
import org.opensearch.common.bytes.BytesReference;
import org.opensearch.common.settings.Settings;
import org.opensearch.common.xcontent.XContentType;
import org.opensearch.rest.BytesRestResponse;
import org.opensearch.rest.RestChannel;
import org.opensearch.rest.RestRequest;
import org.opensearch.rest.RestResponse;
import org.opensearch.rest.RestStatus;
import org.opensearch.security.DefaultObjectMapper;
import org.opensearch.security.dlic.rest.api.AuthTokenProcessorAction;
import org.xml.sax.SAXException;

class AuthTokenProcessorHandler {
    private static final Logger log = LogManager.getLogger(AuthTokenProcessorHandler.class);
    private static final Logger token_log = LogManager.getLogger((String)"com.amazon.dlic.auth.http.saml.Token");
    private static final Pattern EXPIRY_SETTINGS_PATTERN = Pattern.compile("\\s*(\\w+)\\s*(?:\\+\\s*(\\w+))?\\s*");
    private Saml2SettingsProvider saml2SettingsProvider;
    private JoseJwtProducer jwtProducer;
    private String jwtSubjectKey;
    private String jwtRolesKey;
    private String samlSubjectKey;
    private String samlRolesKey;
    private String kibanaRootUrl;
    private long expiryOffset = 0L;
    private ExpiryBaseValue expiryBaseValue = ExpiryBaseValue.AUTO;
    private JsonWebKey signingKey;
    private JsonMapObjectReaderWriter jsonMapReaderWriter = new JsonMapObjectReaderWriter();
    private Pattern samlRolesSeparatorPattern;

    AuthTokenProcessorHandler(Settings settings, Settings jwtSettings, Saml2SettingsProvider saml2SettingsProvider) throws Exception {
        this.saml2SettingsProvider = saml2SettingsProvider;
        this.jwtRolesKey = jwtSettings.get("roles_key", "roles");
        this.jwtSubjectKey = jwtSettings.get("subject_key", "sub");
        this.samlRolesKey = settings.get("roles_key");
        this.samlSubjectKey = settings.get("subject_key");
        String samlRolesSeparator = settings.get("roles_separator", settings.get("roles_seperator"));
        this.kibanaRootUrl = settings.get("kibana_url");
        if (samlRolesSeparator != null) {
            this.samlRolesSeparatorPattern = Pattern.compile(samlRolesSeparator);
        }
        if (this.samlRolesKey == null || this.samlRolesKey.length() == 0) {
            log.warn("roles_key is not configured, will only extract subject from SAML");
            this.samlRolesKey = null;
        }
        if (this.samlSubjectKey == null || this.samlSubjectKey.length() == 0) {
            this.samlSubjectKey = null;
        }
        if (samlRolesSeparator == null || samlRolesSeparator.length() == 0) {
            samlRolesSeparator = null;
        }
        this.initJwtExpirySettings(settings);
        this.signingKey = this.createJwkFromSettings(settings, jwtSettings);
        this.jwtProducer = new JoseJwtProducer();
        this.jwtProducer.setSignatureProvider(JwsUtils.getSignatureProvider((JsonWebKey)this.signingKey));
    }

    boolean handle(final RestRequest restRequest, final RestChannel restChannel) throws Exception {
        try {
            SecurityManager sm = System.getSecurityManager();
            if (sm != null) {
                sm.checkPermission((Permission)new SpecialPermission());
            }
            return AccessController.doPrivileged(new PrivilegedExceptionAction<Boolean>(){

                @Override
                public Boolean run() throws XPathExpressionException, SamlConfigException, IOException, ParserConfigurationException, SAXException, SettingsException {
                    return AuthTokenProcessorHandler.this.handleLowLevel(restRequest, restChannel);
                }
            });
        }
        catch (PrivilegedActionException e) {
            if (e.getCause() instanceof Exception) {
                throw (Exception)e.getCause();
            }
            throw new RuntimeException(e);
        }
    }

    private AuthTokenProcessorAction.Response handleImpl(RestRequest restRequest, RestChannel restChannel, String samlResponseBase64, String samlRequestId, String acsEndpoint, Saml2Settings saml2Settings) throws XPathExpressionException, ParserConfigurationException, SAXException, IOException, SettingsException {
        if (token_log.isDebugEnabled()) {
            try {
                token_log.debug("SAMLResponse for {}\n{}", (Object)samlRequestId, (Object)new String(Util.base64decoder((String)samlResponseBase64), "UTF-8"));
            }
            catch (Exception e) {
                token_log.warn("SAMLResponse for {} cannot be decoded from base64\n{}", (Object)samlRequestId, (Object)samlResponseBase64, (Object)e);
            }
        }
        try {
            SamlResponse samlResponse = new SamlResponse(saml2Settings, null);
            samlResponse.setDestinationUrl(acsEndpoint);
            samlResponse.loadXmlFromBase64(samlResponseBase64);
            if (!samlResponse.isValid(samlRequestId)) {
                log.warn("Error while validating SAML response in /_opendistro/_security/api/authtoken");
                return null;
            }
            AuthTokenProcessorAction.Response responseBody = new AuthTokenProcessorAction.Response();
            responseBody.setAuthorization("bearer " + this.createJwt(samlResponse));
            return responseBody;
        }
        catch (ValidationError e) {
            log.warn("Error while validating SAML response", (Throwable)e);
            return null;
        }
        catch (Exception e) {
            log.error("Error while converting SAML to JWT", (Throwable)e);
            return null;
        }
    }

    private boolean handleLowLevel(RestRequest restRequest, RestChannel restChannel) throws SamlConfigException, IOException, XPathExpressionException, ParserConfigurationException, SAXException, SettingsException {
        try {
            AuthTokenProcessorAction.Response responseBody;
            if (restRequest.getXContentType() != XContentType.JSON) {
                throw new OpenSearchSecurityException("/_opendistro/_security/api/authtoken expects content with type application/json", RestStatus.UNSUPPORTED_MEDIA_TYPE, new Object[0]);
            }
            if (restRequest.method() != RestRequest.Method.POST) {
                throw new OpenSearchSecurityException("/_opendistro/_security/api/authtoken expects POST requests", RestStatus.METHOD_NOT_ALLOWED, new Object[0]);
            }
            Saml2Settings saml2Settings = this.saml2SettingsProvider.getCached();
            BytesReference bytesReference = restRequest.requiredContent();
            JsonNode jsonRoot = DefaultObjectMapper.objectMapper.readTree(BytesReference.toBytes((BytesReference)bytesReference));
            if (!(jsonRoot instanceof ObjectNode)) {
                throw new JsonParseException(null, "Unexpected json format: " + jsonRoot);
            }
            if (((ObjectNode)jsonRoot).get("SAMLResponse") == null) {
                log.warn("SAMLResponse is missing from request ");
                throw new OpenSearchSecurityException("SAMLResponse is missing from request", RestStatus.BAD_REQUEST, new Object[0]);
            }
            String samlResponseBase64 = ((ObjectNode)jsonRoot).get("SAMLResponse").asText();
            String samlRequestId = ((ObjectNode)jsonRoot).get("RequestId") != null ? ((ObjectNode)jsonRoot).get("RequestId").textValue() : null;
            String acsEndpoint = saml2Settings.getSpAssertionConsumerServiceUrl().toString();
            if (((ObjectNode)jsonRoot).get("acsEndpoint") != null && ((ObjectNode)jsonRoot).get("acsEndpoint").textValue() != null) {
                acsEndpoint = this.getAbsoluteAcsEndpoint(((ObjectNode)jsonRoot).get("acsEndpoint").textValue());
            }
            if ((responseBody = this.handleImpl(restRequest, restChannel, samlResponseBase64, samlRequestId, acsEndpoint, saml2Settings)) == null) {
                return false;
            }
            String responseBodyString = DefaultObjectMapper.objectMapper.writeValueAsString((Object)responseBody);
            BytesRestResponse authenticateResponse = new BytesRestResponse(RestStatus.OK, "application/json", responseBodyString);
            restChannel.sendResponse((RestResponse)authenticateResponse);
            return true;
        }
        catch (JsonProcessingException e) {
            log.warn("Error while parsing JSON for /_opendistro/_security/api/authtoken", (Throwable)e);
            BytesRestResponse authenticateResponse = new BytesRestResponse(RestStatus.BAD_REQUEST, "JSON could not be parsed");
            restChannel.sendResponse((RestResponse)authenticateResponse);
            return true;
        }
    }

    JsonWebKey createJwkFromSettings(Settings settings, Settings jwtSettings) throws Exception {
        String exchangeKey = settings.get("exchange_key");
        if (!Strings.isNullOrEmpty((String)exchangeKey)) {
            JsonWebKey jwk = new JsonWebKey();
            jwk.setKeyType(KeyType.OCTET);
            jwk.setAlgorithm("HS512");
            jwk.setPublicKeyUse(PublicKeyUse.SIGN);
            jwk.setProperty("k", (Object)exchangeKey);
            return jwk;
        }
        Settings jwkSettings = jwtSettings.getAsSettings("key");
        if (jwkSettings.isEmpty()) {
            throw new Exception("Settings for key exchange missing. Please specify at least the option exchange_key with a shared secret.");
        }
        JsonWebKey jwk = new JsonWebKey();
        for (String key : jwkSettings.keySet()) {
            jwk.setProperty(key, (Object)jwkSettings.get(key));
        }
        return jwk;
    }

    private String createJwt(SamlResponse samlResponse) throws Exception {
        String sessionIndex;
        JwtClaims jwtClaims = new JwtClaims();
        JwtToken jwt = new JwtToken(jwtClaims);
        jwtClaims.setNotBefore(Long.valueOf(System.currentTimeMillis() / 1000L));
        jwtClaims.setExpiryTime(Long.valueOf(this.getJwtExpiration(samlResponse)));
        jwtClaims.setProperty(this.jwtSubjectKey, (Object)this.extractSubject(samlResponse));
        if (this.samlSubjectKey != null) {
            jwtClaims.setProperty("saml_ni", (Object)samlResponse.getNameId());
        }
        if (samlResponse.getNameIdFormat() != null) {
            jwtClaims.setProperty("saml_nif", (Object)SamlNameIdFormat.getByUri(samlResponse.getNameIdFormat()).getShortName());
        }
        if ((sessionIndex = samlResponse.getSessionIndex()) != null) {
            jwtClaims.setProperty("saml_si", (Object)sessionIndex);
        }
        if (this.samlRolesKey != null && this.jwtRolesKey != null) {
            String[] roles = this.extractRoles(samlResponse);
            jwtClaims.setProperty(this.jwtRolesKey, (Object)roles);
        }
        String encodedJwt = this.jwtProducer.processJwt(jwt);
        if (token_log.isDebugEnabled()) {
            token_log.debug("Created JWT: " + encodedJwt + "\n" + this.jsonMapReaderWriter.toJson((JsonMapObject)jwt.getJwsHeaders()) + "\n" + JwtUtils.claimsToJson((JwtClaims)jwt.getClaims()));
        }
        return encodedJwt;
    }

    private long getJwtExpiration(SamlResponse samlResponse) throws Exception {
        DateTime sessionNotOnOrAfter = samlResponse.getSessionNotOnOrAfter();
        if (this.expiryBaseValue == ExpiryBaseValue.NOW) {
            return System.currentTimeMillis() / 1000L + this.expiryOffset;
        }
        if (this.expiryBaseValue == ExpiryBaseValue.SESSION) {
            if (sessionNotOnOrAfter != null) {
                return sessionNotOnOrAfter.getMillis() / 1000L + this.expiryOffset;
            }
            throw new Exception("Error while determining JWT expiration time: SamlResponse did not contain sessionNotOnOrAfter value");
        }
        if (sessionNotOnOrAfter != null) {
            return sessionNotOnOrAfter.getMillis() / 1000L;
        }
        return System.currentTimeMillis() / 1000L + (this.expiryOffset > 0L ? this.expiryOffset : 3600L);
    }

    private void initJwtExpirySettings(Settings settings) {
        String expiry = settings.get("jwt.expiry");
        if (Strings.isNullOrEmpty((String)expiry)) {
            return;
        }
        Matcher matcher = EXPIRY_SETTINGS_PATTERN.matcher(expiry);
        if (!matcher.matches()) {
            log.error("Invalid value for jwt.expiry: {}; using defaults.", (Object)expiry);
            return;
        }
        String baseValue = matcher.group(1);
        String offset = matcher.group(2);
        if (offset != null && !StringUtils.isNumeric((CharSequence)offset)) {
            log.error("Invalid offset value for jwt.expiry: {}; using defaults.", (Object)expiry);
            return;
        }
        if (!Strings.isNullOrEmpty((String)baseValue)) {
            try {
                this.expiryBaseValue = ExpiryBaseValue.valueOf(baseValue.toUpperCase());
            }
            catch (IllegalArgumentException e) {
                log.error("Invalid base value for jwt.expiry: {}; using defaults", (Object)expiry);
                return;
            }
        }
        if (offset != null) {
            this.expiryOffset = Integer.parseInt(offset) * 60;
        }
    }

    private String extractSubject(SamlResponse samlResponse) throws Exception {
        if (this.samlSubjectKey == null) {
            return samlResponse.getNameId();
        }
        List values = (List)samlResponse.getAttributes().get(this.samlSubjectKey);
        if (values == null || values.size() == 0) {
            return null;
        }
        return (String)values.get(0);
    }

    private String[] extractRoles(SamlResponse samlResponse) throws XPathExpressionException, ValidationError {
        if (this.samlRolesKey == null) {
            return new String[0];
        }
        List<String> values = (List<String>)samlResponse.getAttributes().get(this.samlRolesKey);
        if (values == null || values.size() == 0) {
            return null;
        }
        if (this.samlRolesSeparatorPattern != null) {
            values = this.splitRoles(values);
        }
        return values.toArray(new String[values.size()]);
    }

    private List<String> splitRoles(List<String> values) {
        return values.stream().flatMap(v -> this.samlRolesSeparatorPattern.splitAsStream((CharSequence)v)).filter(r -> !Strings.isNullOrEmpty((String)r)).collect(Collectors.toList());
    }

    private String getAbsoluteAcsEndpoint(String acsEndpoint) {
        try {
            URI acsEndpointUri = new URI(acsEndpoint);
            if (acsEndpointUri.isAbsolute()) {
                return acsEndpoint;
            }
            return new URI(this.kibanaRootUrl).resolve(acsEndpointUri).toString();
        }
        catch (URISyntaxException e) {
            log.error("Could not parse URI for acsEndpoint: {}", (Object)acsEndpoint);
            return acsEndpoint;
        }
    }

    public JsonWebKey getSigningKey() {
        return this.signingKey;
    }

    private static enum ExpiryBaseValue {
        AUTO,
        NOW,
        SESSION;

    }
}

