package org.openingo.boot.es.ik.redict.dictionary;

import io.lettuce.core.RedisClient;
import io.lettuce.core.RedisURI;
import io.lettuce.core.TransactionResult;
import io.lettuce.core.api.StatefulRedisConnection;
import io.lettuce.core.api.sync.RedisCommands;
import lombok.extern.slf4j.Slf4j;
import org.openingo.boot.es.ik.redict.configuration.RemoteDictionaryConfigurationProperties;

import java.util.Objects;

/**
 * RedisRemoteDictionary
 *
 * @author Qicz
 * @since 2021/7/14 18:49
 */
@Slf4j
public class RedisRemoteDictionary extends AbstractRemoteDictionary {

	private final StatefulRedisConnection<String, String> redisConnection;

	private final static String KEY_PREFIX = "es-ik-words";

	public RedisRemoteDictionary(RemoteDictionaryConfigurationProperties properties) {
		super(properties);
		this.redisConnection = this.getRedisConnection();
	}

	@Override
	protected boolean addWord(String domain, DictionaryType dictionaryType, String word) {
		log.info("'{}' remote dictionary add new word '{}' for dictionary '{}'", this.etymology(), word, dictionaryType);
		RedisCommands<String, String> sync = this.redisConnection.sync();
		sync.multi();
		String key = this.getKey(dictionaryType, domain);
		sync.lpush(key, word);
		String state = String.format("%s:state", key);
		sync.set(state, "true");
		TransactionResult transactionResult = sync.exec();
		for (Object txRet : transactionResult) {
			log.info("txRet '{}'", txRet);
		}
		log.info("'{} add new word '{}' success.", this.etymology(), word);
		return true;
	}

	@Override
	protected void closeResource() {
		String etymology = this.etymology();
		log.info("'{}' remote dictionary is closing...", etymology);
		this.redisConnection.close();
		log.info("'{}' remote dictionary is closed", etymology);
	}

	@Override
	protected String etymology() {
		return "redis";
	}

	private String getKey(DictionaryType dictionaryType, String domain) {
		// # main-words key: es-ik-words:{domain}:main-words
		// # stop-words key: es-ik-words:{domain}:stop-words
		return String.format("%s:%s:%s", KEY_PREFIX, domain, dictionaryType.getDictName());
	}

	private StatefulRedisConnection<String, String> getRedisConnection() {
		RemoteDictionaryConfigurationProperties.Redis redis = this.properties.getRedis();
		RedisURI.Builder builder = RedisURI.builder()
				.withHost(redis.getHost())
				.withPort(redis.getPort())
				.withDatabase(redis.getDatabase());
		String username = redis.getUsername();
		String password = redis.getPassword();
		if (Objects.nonNull(username) && Objects.nonNull(password)) {
			builder.withAuthentication(redis.getUsername(), password.toCharArray());
		} else if (Objects.nonNull(password)) {
			builder.withPassword(password.toCharArray());
		}
		return RedisClient.create(builder.build()).connect();
	}
}
