package org.opencrx.application.uses.ezvcard.io.scribe;

import static org.opencrx.application.uses.ezvcard.util.StringUtils.NEWLINE;
import static org.opencrx.application.uses.ezvcard.util.StringUtils.join;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import javax.xml.namespace.QName;

import org.opencrx.application.uses.ezvcard.VCard;
import org.opencrx.application.uses.ezvcard.VCardDataType;
import org.opencrx.application.uses.ezvcard.VCardVersion;
import org.opencrx.application.uses.ezvcard.io.CannotParseException;
import org.opencrx.application.uses.ezvcard.io.EmbeddedVCardException;
import org.opencrx.application.uses.ezvcard.io.SkipMeException;
import org.opencrx.application.uses.ezvcard.io.text.VCardRawWriter;
import org.opencrx.application.uses.ezvcard.io.xml.XCardElement;
import org.opencrx.application.uses.ezvcard.parameter.VCardParameters;
import org.opencrx.application.uses.ezvcard.property.Categories;
import org.opencrx.application.uses.ezvcard.property.Organization;
import org.opencrx.application.uses.ezvcard.property.StructuredName;
import org.opencrx.application.uses.ezvcard.property.VCardProperty;
import org.opencrx.application.uses.ezvcard.util.VCardDateFormat;
import org.opencrx.application.uses.ezvcard.util.XmlUtils;
import org.opencrx.application.uses.ezvcard.util.StringUtils.JoinCallback;
import org.w3c.dom.Element;

/*
 Copyright (c) 2013, Michael Angstadt
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions are met: 

 1. Redistributions of source code must retain the above copyright notice, this
 list of conditions and the following disclaimer. 
 2. Redistributions in binary form must reproduce the above copyright notice,
 this list of conditions and the following disclaimer in the documentation
 and/or other materials provided with the distribution. 

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
 ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
 * Base class for vCard property scribes (marshallers).
 * @param <T> the property class
 * @author Michael Angstadt
 */
public abstract class VCardPropertyScribe<T extends VCardProperty> {
	protected final Class<T> clazz;
	protected final String propertyName;
	protected final QName qname;

	/**
	 * Creates a new marshaller.
	 * @param clazz the property class
	 * @param propertyName the property name (e.g. "FN")
	 */
	public VCardPropertyScribe(Class<T> clazz, String propertyName) {
		this(clazz, propertyName, new QName(VCardVersion.V4_0.getXmlNamespace(), propertyName.toLowerCase()));
	}

	/**
	 * Creates a new marshaller.
	 * @param clazz the property class
	 * @param propertyName the property name (e.g. "FN")
	 * @param qname the XML element name and namespace to use for xCard
	 * documents (by default, the XML element name is set to the lower-cased
	 * property name, and the element namespace is set to the xCard namespace)
	 */
	public VCardPropertyScribe(Class<T> clazz, String propertyName, QName qname) {
		this.clazz = clazz;
		this.propertyName = propertyName;
		this.qname = qname;
	}

	/**
	 * Gets the property class.
	 * @return the property class
	 */
	public Class<T> getPropertyClass() {
		return clazz;
	}

	/**
	 * Gets the property name.
	 * @return the property name (e.g. "FN")
	 */
	public String getPropertyName() {
		return propertyName;
	}

	/**
	 * Gets this property's local name and namespace for xCard documents.
	 * @return the XML local name and namespace
	 */
	public QName getQName() {
		return qname;
	}

	/**
	 * Sanitizes a property's parameters (called before the property is
	 * written). Note that a copy of the parameters is returned so that the
	 * property object does not get modified.
	 * @param property the property
	 * @param version the version of the vCard that is being generated
	 * @param vcard the vCard that the property belongs to
	 * @return the sanitized parameters
	 */
	public final VCardParameters prepareParameters(T property, VCardVersion version, VCard vcard) {
		//make a copy because the property should not get modified when it is marshalled
		VCardParameters copy = new VCardParameters(property.getParameters());
		_prepareParameters(property, copy, version, vcard);
		return copy;
	}

	/**
	 * <p>
	 * Determines the property's default data type.
	 * </p>
	 * <p>
	 * When writing a plain-text vCard, if the data type of a property instance
	 * (as determined by the {@link #dataType} method) matches the default data
	 * type, then a VALUE parameter will *not* be written.
	 * </p>
	 * <p>
	 * When parsing a plain-text vCard, if a property has no VALUE parameter,
	 * then the property's default data type will be passed into the
	 * {@link #parseText} method.
	 * </p>
	 * @param version the vCard version
	 * @return the default data type or null if unknown
	 */
	public final VCardDataType defaultDataType(VCardVersion version) {
		return _defaultDataType(version);
	}

	/**
	 * Determines the data type of a property instance.
	 * @param property the property
	 * @param version the version of the vCard that is being generated
	 * @return the data type or null if unknown
	 */
	public final VCardDataType dataType(T property, VCardVersion version) {
		return _dataType(property, version);
	}

	/**
	 * Marshals a property's value to a string.
	 * @param property the property
	 * @param version the version of the vCard that is being generated
	 * @return the marshalled value
	 * @throws SkipMeException if the property should not be written to the data
	 * stream
	 */
	public final String writeText(T property, VCardVersion version) {
		return _writeText(property, version);
	}

	/**
	 * Marshals a property's value to an XML element (xCard).
	 * @param property the property
	 * @param element the property's XML element.
	 * @throws SkipMeException if the property should not be written to the data
	 * stream
	 */
	public final void writeXml(T property, Element element) {
		XCardElement xCardElement = new XCardElement(element);
		_writeXml(property, xCardElement);
	}

	/**
	 * Unmarshals a property from a plain-text vCard.
	 * @param value the value as read off the wire
	 * @param dataType the data type of the property value. The property's VALUE
	 * parameter is used to determine the data type. If the property has no
	 * VALUE parameter, then this parameter will be set to the property's
	 * default datatype, as determined by the {@link #defaultDataType} method.
	 * Note that the VALUE parameter is removed from the property's parameter
	 * list after it has been read.
	 * @param version the version of the vCard that is being parsed
	 * @param parameters the parsed parameters
	 * @return the unmarshalled property and its warnings
	 * @throws CannotParseException if the marshaller could not parse the
	 * property's value
	 * @throws SkipMeException if the property should not be added to the final
	 * {@link VCard} object
	 * @throws EmbeddedVCardException if the property value is an embedded vCard
	 * (i.e. the AGENT property)
	 */
	public final Result<T> parseText(String value, VCardDataType dataType, VCardVersion version, VCardParameters parameters) {
		List<String> warnings = new ArrayList<String>(0);
		T property = _parseText(value, dataType, version, parameters, warnings);
		property.setParameters(parameters);
		return new Result<T>(property, warnings);
	}

	/**
	 * Unmarshals a property's value from an XML document (xCard).
	 * @param element the property's XML element
	 * @param parameters the parsed parameters
	 * @return the unmarshalled property and its warnings
	 * @throws CannotParseException if the marshaller could not parse the
	 * property's value
	 * @throws SkipMeException if the property should not be added to the final
	 * {@link VCard} object
	 */
	public final Result<T> parseXml(Element element, VCardParameters parameters) {
		List<String> warnings = new ArrayList<String>(0);
		T property = _parseXml(new XCardElement(element), parameters, warnings);
		property.setParameters(parameters);
		return new Result<T>(property, warnings);
	}

	/**
	 * <p>
	 * Sanitizes a property's parameters before the property is written.
	 * </p>
	 * <p>
	 * This method should be overridden by child classes that wish to tweak the
	 * property's parameters before the property is written. The default
	 * implementation of this method does nothing.
	 * </p>
	 * @param property the property
	 * @param copy the list of parameters to make modifications to (it is a copy
	 * of the property's parameters)
	 * @param version the version of the vCard that is being generated
	 * @param vcard the vCard that the property belongs to
	 */
	protected void _prepareParameters(T property, VCardParameters copy, VCardVersion version, VCard vcard) {
		//do nothing
	}

	/**
	 * <p>
	 * Determines the property's default data type.
	 * </p>
	 * <p>
	 * When writing a plain-text vCard, if the data type of a property instance
	 * (as determined by the {@link #dataType} method) matches the default data
	 * type, then a VALUE parameter will *not* be written.
	 * </p>
	 * <p>
	 * When parsing a plain-text vCard, if a property has no VALUE parameter,
	 * then the property's default data type will be passed into the
	 * {@link #parseText} method.
	 * </p>
	 * @param version the vCard version
	 * @return the default data type or null if unknown
	 */
	protected abstract VCardDataType _defaultDataType(VCardVersion version);

	/**
	 * <p>
	 * Determines the data type of a property instance.
	 * </p>
	 * <p>
	 * This method should be overridden by child classes if a property's data
	 * type changes depending on its value. The default implementation of this
	 * method calls {@link #_defaultDataType}.
	 * </p>
	 * @param property the property
	 * @param version the version of the vCard that is being generated
	 * @return the data type or null if unknown
	 */
	protected VCardDataType _dataType(T property, VCardVersion version) {
		return _defaultDataType(version);
	}

	/**
	 * Marshals a property's value to a string.
	 * @param property the property
	 * @param version the version of the vCard that is being generated
	 * @return the marshalled value
	 * @throws SkipMeException if the property should not be written to the data
	 * stream
	 */
	protected abstract String _writeText(T property, VCardVersion version);

	/**
	 * <p>
	 * Marshals a property's value to an XML element (xCard).
	 * <p>
	 * <p>
	 * This method should be overridden by child classes that wish to support
	 * xCard. The default implementation of this method will append one child
	 * element to the property's XML element. The child element's name will be
	 * that of the property's data type (retrieved using the {@link #dataType}
	 * method), and the child element's text content will be set to the
	 * property's marshalled plain-text value (retrieved using the
	 * {@link #writeText} method).
	 * </p>
	 * @param property the property
	 * @param element the property's XML element
	 * @throws SkipMeException if the property should not be written to the data
	 * stream
	 */
	protected void _writeXml(T property, XCardElement element) {
		String value = writeText(property, VCardVersion.V4_0);
		VCardDataType dataType = dataType(property, VCardVersion.V4_0);
		element.append(dataType, value);
	}

	/**
	 * Unmarshals a property from a plain-text vCard.
	 * @param value the value as read off the wire
	 * @param dataType the data type of the property value. The property's VALUE
	 * parameter is used to determine the data type. If the property has no
	 * VALUE parameter, then this parameter will be set to the property's
	 * default datatype, as determined by the {@link #defaultDataType} method.
	 * Note that the VALUE parameter is removed from the property's parameter
	 * list after it has been read.
	 * @param version the version of the vCard that is being parsed
	 * @param parameters the parsed parameters. These parameters will be
	 * assigned to the property object once this method returns. Therefore, do
	 * not assign any parameters to the property object itself whilst inside of
	 * this method, or else they will be overwritten.
	 * @param warnings allows the programmer to alert the user to any
	 * note-worthy (but non-critical) issues that occurred during the
	 * unmarshalling process
	 * @return the unmarshalled property object
	 * @throws CannotParseException if the marshaller could not parse the
	 * property's value
	 * @throws SkipMeException if the property should not be added to the final
	 * {@link VCard} object
	 */
	protected abstract T _parseText(String value, VCardDataType dataType, VCardVersion version, VCardParameters parameters, List<String> warnings);

	/**
	 * <p>
	 * Unmarshals a property from an XML document (xCard).
	 * </p>
	 * <p>
	 * This method should be overridden by child classes that wish to support
	 * xCard. The default implementation of this method will find the first
	 * child element with the xCard namespace. The element's name will be used
	 * as the property's data type and its text content will be passed into the
	 * {@link #_parseText} method. If no such child element is found, then the
	 * parent element's text content will be passed into {@link #_parseText} and
	 * the data type will be null.
	 * </p>
	 * @param element the property's XML element
	 * @param parameters the parsed parameters. These parameters will be
	 * assigned to the property object once this method returns. Therefore, do
	 * not assign any parameters to the property object itself whilst inside of
	 * this method, or else they will be overwritten.
	 * @param warnings allows the programmer to alert the user to any
	 * note-worthy (but non-critical) issues that occurred during the
	 * unmarshalling process
	 * @return the unmarshalled property object
	 * @throws CannotParseException if the marshaller could not parse the
	 * property's value
	 * @throws SkipMeException if the property should not be added to the final
	 * {@link VCard} object
	 */
	protected T _parseXml(XCardElement element, VCardParameters parameters, List<String> warnings) {
		String value = null;
		VCardDataType dataType = null;
		Element rawElement = element.element();
		VCardVersion version = element.version();

		//get the text content of the first child element with the xCard namespace
		List<Element> children = XmlUtils.toElementList(rawElement.getChildNodes());
		for (Element child : children) {
			String elementNamespace = version.getXmlNamespace();
			String childNamespace = child.getNamespaceURI();
			if (elementNamespace.equals(childNamespace)) {
				String dataTypeStr = child.getLocalName();
				dataType = "unknown".equals(dataTypeStr) ? null : VCardDataType.get(dataTypeStr);
				value = child.getTextContent();
				break;
			}
		}

		if (dataType == null) {
			//get the text content of the property element
			value = rawElement.getTextContent();
		}

		value = escape(value);
		return _parseText(value, dataType, version, parameters, warnings);
	}

	/**
	 * Unescapes all special characters that are escaped with a backslash, as
	 * well as escaped newlines.
	 * @param text the text to unescape
	 * @return the unescaped text
	 */
	public static String unescape(String text) {
		if (text == null) {
			return null;
		}

		StringBuilder sb = null; //only instantiate the StringBuilder if the string needs to be modified
		boolean escaped = false;
		for (int i = 0; i < text.length(); i++) {
			char ch = text.charAt(i);

			if (escaped) {
				if (sb == null) {
					sb = new StringBuilder(text.length());
					sb.append(text.substring(0, i - 1));
				}

				escaped = false;

				if (ch == 'n' || ch == 'N') {
					//newlines appear as "\n" or "\N" (see RFC 5545 p.46)
					sb.append(NEWLINE);
					continue;
				}

				sb.append(ch);
				continue;
			}

			if (ch == '\\') {
				escaped = true;
				continue;
			}

			if (sb != null) {
				sb.append(ch);
			}
		}
		return (sb == null) ? text : sb.toString();
	}

	/**
	 * <p>
	 * Escapes all special characters within a vCard value. These characters
	 * are:
	 * </p>
	 * <ul>
	 * <li>backslashes ({@code \})</li>
	 * <li>commas ({@code ,})</li>
	 * <li>semi-colons ({@code ;})</li>
	 * </ul>
	 * <p>
	 * Newlines are not escaped by this method. They are escaped when the vCard
	 * is serialized (in the {@link VCardRawWriter} class).
	 * </p>
	 * @param text the text to escape
	 * @return the escaped text
	 */
	public static String escape(String text) {
		if (text == null) {
			return null;
		}

		String chars = "\\,;";
		StringBuilder sb = null; //only instantiate the StringBuilder if the string needs to be modified
		for (int i = 0; i < text.length(); i++) {
			char ch = text.charAt(i);
			if (chars.indexOf(ch) >= 0) {
				if (sb == null) {
					sb = new StringBuilder(text.length());
					sb.append(text.substring(0, i));
				}
				sb.append('\\');
			}

			if (sb != null) {
				sb.append(ch);
			}
		}
		return (sb == null) ? text : sb.toString();
	}

	/**
	 * Creates a string splitter (takes escaped characters into account).
	 * @param delimiter the delimiter character (e.g. ',')
	 * @return the splitter object
	 */
	protected static Splitter splitter(char delimiter) {
		return new Splitter(delimiter);
	}

	/**
	 * A helper class for splitting strings.
	 */
	protected static class Splitter {
		private char delimiter;
		private boolean unescape = false;
		private boolean nullEmpties = false;
		private int limit = -1;

		/**
		 * Creates a new splitter object.
		 * @param delimiter the delimiter character (e.g. ',')
		 */
		public Splitter(char delimiter) {
			this.delimiter = delimiter;
		}

		/**
		 * Sets whether to unescape each split string.
		 * @param unescape true to unescape, false not to (default is false)
		 * @return this
		 */
		public Splitter unescape(boolean unescape) {
			this.unescape = unescape;
			return this;
		}

		/**
		 * Sets whether to treat empty elements as null elements.
		 * @param nullEmpties true to treat them as null elements, false to
		 * treat them as empty strings (default is false)
		 * @return this
		 */
		public Splitter nullEmpties(boolean nullEmpties) {
			this.nullEmpties = nullEmpties;
			return this;
		}

		/**
		 * Sets the max number of split strings it should parse.
		 * @param limit the max number of split strings
		 * @return this
		 */
		public Splitter limit(int limit) {
			this.limit = limit;
			return this;
		}

		/**
		 * Performs the split operation.
		 * @param string the string to split (e.g. "one,two,three")
		 * @return the split string
		 */
		public List<String> split(String string) {
			//doing it this way is 10x faster than a regex

			List<String> list = new ArrayList<String>();
			boolean escaped = false;
			int start = 0;
			for (int i = 0; i < string.length(); i++) {
				char ch = string.charAt(i);

				if (escaped) {
					escaped = false;
					continue;
				}

				if (ch == delimiter) {
					add(string.substring(start, i), list);
					start = i + 1;
					if (limit > 0 && list.size() == limit - 1) {
						break;
					}

					continue;
				}

				if (ch == '\\') {
					escaped = true;
					continue;
				}
			}

			add(string.substring(start), list);

			return list;
		}

		private void add(String str, List<String> list) {
			str = str.trim();

			if (nullEmpties && str.length() == 0) {
				str = null;
			} else if (unescape) {
				str = VCardPropertyScribe.unescape(str);
			}

			list.add(str);
		}
	}

	/**
	 * Parses a "list" property value. This is used in plain-text vCards to
	 * parse properties such as {@link Categories}.
	 * @param value the string to parse (e.g. "one,two,three\,four")
	 * @return the parsed list (e.g. ["one", "two", "three,four"])
	 */
	protected static List<String> list(String value) {
		if (value.length() == 0) {
			return new ArrayList<String>(0);
		}
		return splitter(',').unescape(true).split(value);
	}

	/**
	 * Generates a "list" property value. This is used in plain-text vCards to
	 * parse properties such as {@link Categories}.
	 * @param values the values to write (the {@code toString()} method is
	 * invoked on each object, null objects are ignored, e.g. ["one", "two",
	 * "three,four"])
	 * @return the property value (e.g. "one,two,three\,four")
	 */
	protected static String list(Object... values) {
		return list(Arrays.asList(values));
	}

	/**
	 * Generates a "list" property value. This is used in plain-text vCards to
	 * parse properties such as {@link Categories}).
	 * @param values the values to write (the {@code toString()} method is
	 * invoked on each object, null objects are ignored, e.g. ["one", "two",
	 * "three,four"])
	 * @return the property value (e.g. "one,two,three\,four")
	 */
	protected static <T> String list(Collection<T> values) {
		return join(values, ",", new JoinCallback<T>() {
			public void handle(StringBuilder sb, T value) {
				if (value == null) {
					return;
				}
				sb.append(escape(value.toString()));
			}
		});
	}

	/**
	 * Parses a "semi-structured" property value (a "structured" property value
	 * whose items cannot be multi-valued). This is used in plain-text vCards to
	 * parse properties such as {@link Organization}.
	 * @param value the string to parse (e.g. "one;two;three\;four,five")
	 * @return the parsed values (e.g. ["one", "two", "three;four,five"]
	 */
	protected static SemiStructuredIterator semistructured(String value) {
		return semistructured(value, -1);
	}

	/**
	 * Parses a "semi-structured" property value (a "structured" property value
	 * whose items cannot be multi-valued). This is used in plain-text vCards to
	 * parse properties such as {@link Organization}.
	 * @param value the string to parse (e.g. "one;two;three\;four,five")
	 * @param limit the max number of items to parse (see
	 * {@link String#split(String, int)})
	 * @return the parsed values (e.g. ["one", "two", "three;four,five"]
	 */
	protected static SemiStructuredIterator semistructured(String value, int limit) {
		List<String> split = splitter(';').unescape(true).limit(limit).split(value);
		return new SemiStructuredIterator(split.iterator());
	}

	/**
	 * Parses a "structured" property value. This is used in plain-text vCards
	 * to parse properties such as {@link StructuredName}.
	 * @param value the string to parse (e.g. "one;two,three;four\,five\;six")
	 * @return an iterator for accessing the parsed values (e.g. ["one", ["two",
	 * "three"], "four,five;six"])
	 */
	protected static StructuredIterator structured(String value) {
		List<String> split = splitter(';').split(value);
		List<List<String>> components = new ArrayList<List<String>>(split.size());
		for (String s : split) {
			components.add(list(s));
		}
		return new StructuredIterator(components.iterator());
	}

	/**
	 * <p>
	 * Writes a "structured" property value. This is used in plain-text vCards
	 * to marshal properties such as {@link StructuredName}.
	 * </p>
	 * <p>
	 * This method accepts a list of {@link Object} instances.
	 * {@link Collection} objects will be treated as multi-valued components.
	 * Null objects will be treated as empty components. All other objects will
	 * have their {@code toString()} method invoked to generate the string
	 * value.
	 * </p>
	 * @param values the values to write
	 * @return the structured value string
	 */
	protected static String structured(Object... values) {
		return join(Arrays.asList(values), ";", new JoinCallback<Object>() {
			public void handle(StringBuilder sb, Object value) {
				if (value == null) {
					return;
				}

				if (value instanceof Collection) {
					Collection<?> list = (Collection<?>) value;
					sb.append(list(list));
					return;
				}

				sb.append(escape(value.toString()));
			}
		});
	}

	/**
	 * Iterates over the items in a "structured" property value.
	 */
	protected static class StructuredIterator {
		private final Iterator<List<String>> it;

		/**
		 * Constructs a new structured iterator.
		 * @param it the iterator to wrap
		 */
		public StructuredIterator(Iterator<List<String>> it) {
			this.it = it;
		}

		/**
		 * Gets the first value of the next component.
		 * @return the first value, null if the value is an empty string, or
		 * null if there are no more components
		 */
		public String nextString() {
			if (!hasNext()) {
				return null;
			}

			List<String> list = it.next();
			if (list.isEmpty()) {
				return null;
			}

			String value = list.get(0);
			return (value.length() == 0) ? null : value;
		}

		/**
		 * Gets the next component.
		 * @return the next component, an empty list if the component is empty,
		 * or an empty list of there are no more components
		 */
		public List<String> nextComponent() {
			if (!hasNext()) {
				return new ArrayList<String>(0); //the lists should be mutable so they can be directly assigned to the property object's fields
			}

			List<String> list = it.next();
			if (list.size() == 1 && list.get(0).length() == 0) {
				return new ArrayList<String>(0);
			}

			return list;
		}

		public boolean hasNext() {
			return it.hasNext();
		}
	}

	/**
	 * Iterates over the items in a "semi-structured" property value.
	 */
	protected static class SemiStructuredIterator {
		private final Iterator<String> it;

		/**
		 * Constructs a new structured iterator.
		 * @param it the iterator to wrap
		 */
		public SemiStructuredIterator(Iterator<String> it) {
			this.it = it;
		}

		/**
		 * Gets the next value.
		 * @return the next value or null if there are no more values
		 */
		public String next() {
			return hasNext() ? it.next() : null;
		}

		public boolean hasNext() {
			return it.hasNext();
		}
	}

	/**
	 * Parses a date string.
	 * @param value the date string
	 * @return the parsed date
	 * @throws IllegalArgumentException if the date cannot be parsed
	 */
	protected static Date date(String value) {
		return VCardDateFormat.parse(value);
	}

	/**
	 * Formats a {@link Date} object as a string.
	 * @param date the date
	 * @return a helper object for customizing the write operation
	 */
	protected static DateWriter date(Date date) {
		return new DateWriter(date);
	}

	/**
	 * A helper class for writing date values.
	 */
	protected static class DateWriter {
		private Date date;
		private boolean hasTime = true;
		private boolean extended = false;
		private boolean utc = true;

		/**
		 * Creates a new date writer object.
		 * @param date the date to format
		 */
		public DateWriter(Date date) {
			this.date = date;
		}

		/**
		 * Sets whether to output the date's time component.
		 * @param hasTime true include the time, false if it's strictly a date
		 * (defaults to "true")
		 * @return this
		 */
		public DateWriter time(boolean hasTime) {
			this.hasTime = hasTime;
			return this;
		}

		/**
		 * Sets whether to use extended format or basic.
		 * @param extended true to use extended format, false to use basic
		 * (defaults to "false")
		 * @return this
		 */
		public DateWriter extended(boolean extended) {
			this.extended = extended;
			return this;
		}

		/**
		 * Sets whether to format the date in UTC time, or to include a UTC
		 * offset.
		 * @param utc true to format in UTC time, false to include the local
		 * timezone's UTC offset (defaults to "true")
		 * @return this
		 */
		public DateWriter utc(boolean utc) {
			this.utc = utc;
			return this;
		}

		/**
		 * Creates the date string.
		 * @return the date string
		 */
		public String write() {
			VCardDateFormat format;
			if (hasTime) {
				if (utc) {
					format = extended ? VCardDateFormat.UTC_DATE_TIME_EXTENDED : VCardDateFormat.UTC_DATE_TIME_BASIC;
				} else {
					format = extended ? VCardDateFormat.DATE_TIME_EXTENDED : VCardDateFormat.DATE_TIME_BASIC;
				}
			} else {
				format = extended ? VCardDateFormat.DATE_EXTENDED : VCardDateFormat.DATE_BASIC;
			}

			return format.format(date);
		}
	}

	/**
	 * Creates a {@link CannotParseException} to indicate that a scribe could
	 * not find the necessary XML elements required in order to successfully
	 * parse a property (xCards only).
	 * @param dataTypes the expected data types (null for "unknown")
	 * @return the exception object (note that the exception is NOT thrown!)
	 */
	protected static CannotParseException missingXmlElements(VCardDataType... dataTypes) {
		String[] elements = new String[dataTypes.length];
		for (int i = 0; i < dataTypes.length; i++) {
			VCardDataType dataType = dataTypes[i];
			elements[i] = (dataType == null) ? "unknown" : dataType.getName().toLowerCase();
		}
		return missingXmlElements(elements);
	}

	/**
	 * Creates a {@link CannotParseException} to indicate that a scribe could
	 * not find the necessary XML elements required in order to successfully
	 * parse a property (xCards only).
	 * @param elements the names of the expected XML elements.
	 * @return the exception object (note that the exception is NOT thrown!)
	 */
	protected static CannotParseException missingXmlElements(String... elements) {
		return new CannotParseException(0, Arrays.toString(elements));
	}

	/**
	 * A utility method for switching between the "PREF" and "TYPE=PREF"
	 * parameters when marshalling a property (version 4.0 vCards use "PREF=1",
	 * while version 3.0 vCards use "TYPE=PREF"). This method is meant to be
	 * called from a scribe's {@link #_prepareParameters} method.
	 * @param property the property that is being marshalled
	 * @param copy the parameters that are being marshalled
	 * @param version the vCard version
	 * @param vcard the vCard that's being marshalled
	 */
	protected static void handlePrefParam(VCardProperty property, VCardParameters copy, VCardVersion version, VCard vcard) {
		switch (version) {
		case V2_1:
		case V3_0:
			copy.setPref(null);

			//find the property with the lowest PREF value in the vCard
			VCardProperty mostPreferred = null;
			for (VCardProperty p : vcard.getProperties(property.getClass())) {
				Integer pref = p.getParameters().getPref();
				if (pref == null) {
					continue;
				}

				if (mostPreferred == null || pref < mostPreferred.getParameters().getPref()) {
					mostPreferred = p;
				}
			}

			if (property == mostPreferred) {
				copy.addType("pref");
			}

			break;
		case V4_0:
			for (String type : property.getParameters().getTypes()) {
				if ("pref".equalsIgnoreCase(type)) {
					copy.removeType(type);
					copy.setPref(1);
					break;
				}
			}
			break;
		}
	}

	/**
	 * Represents the result of an unmarshal operation.
	 * @author Michael Angstadt
	 * @param <T> the unmarshalled property class
	 */
	public static class Result<T extends VCardProperty> {
		private final T property;
		private final List<String> warnings;

		/**
		 * Creates a new result.
		 * @param property the property object
		 * @param warnings the warnings
		 */
		public Result(T property, List<String> warnings) {
			this.property = property;
			this.warnings = warnings;
		}

		/**
		 * Gets the warnings.
		 * @return the warnings
		 */
		public List<String> getWarnings() {
			return warnings;
		}

		/**
		 * Gets the property object.
		 * @return the property object
		 */
		public T getProperty() {
			return property;
		}
	}
}
