/**
 * NoiseModelling is a library capable of producing noise maps. It can be freely used either for research and education, as well as by experts in a professional use.
 * <p>
 * NoiseModelling is distributed under GPL 3 license. You can read a copy of this License in the file LICENCE provided with this software.
 * <p>
 * Official webpage : http://noise-planet.org/noisemodelling.html
 * Contact: contact@noise-planet.org
 */

package org.noise_planet.noisemodelling.pathfinder.path;

import org.h2gis.api.ProgressVisitor;
import org.h2gis.utilities.SpatialResultSet;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Geometry;
import org.noise_planet.noisemodelling.pathfinder.profilebuilder.ProfileBuilder;
import org.noise_planet.noisemodelling.pathfinder.utils.geometry.QueryGeometryStructure;
import org.noise_planet.noisemodelling.pathfinder.utils.geometry.QueryRTree;
//import org.noise_planet.noisemodelling.pathfinder.aeffacer.GeoWithSoilType;
import org.noise_planet.noisemodelling.pathfinder.utils.geometry.Orientation;

import java.util.*;


/**
 * Data input for a propagation process (Sub domain generated by ReceiverGrideNoiseMapLoader).
 *
 * @author Nicolas Fortin
 * @author Pierre Aumond
 * @author Adrien Le Bellec
 */
public class Scene {
    public static final double DEFAULT_MAX_PROPAGATION_DISTANCE = 1200;
    public static final double DEFAULT_MAXIMUM_REF_DIST = 700;
    public static final double DEFAULT_RECEIVER_DIST = 1.0;
    public static final double DEFAULT_G = 0.0;
    public static final double DEFAULT_G_BUILDING = 0.0;
    public static final String YAW_DATABASE_FIELD = "YAW";
    public static final String PITCH_DATABASE_FIELD = "PITCH";
    public static final String ROLL_DATABASE_FIELD = "ROLL";
    public static final String DIRECTIVITY_DATABASE_FIELD = "DIR_ID";
    public static final String GS_DATABASE_FIELD = "GS";

    public List<Long> receiversPk = new ArrayList<>();
    public List<Long> sourcesPk = new ArrayList<>();
    /** coordinate of receivers */
    public List<Coordinate> receivers = new ArrayList<>();
    /** Profile builder */
    public ProfileBuilder profileBuilder;
    /** Source Index */
    public QueryGeometryStructure sourcesIndex = new QueryRTree();
    /** Sources geometries. Can be LINESTRING or POINT */
    public List<Geometry> sourceGeometries = new ArrayList<>();

    /** Source orientation for emission computation */
    public Map<Long, Orientation> sourceOrientation = new HashMap<>();

    /** Maximum reflexion order */
    public int reflexionOrder = 1;

    public double defaultGroundAttenuation = 0;

    public Scene() {
        this.profileBuilder = new ProfileBuilder();
    }

    public Scene(ProfileBuilder profileBuilder) {
        this.profileBuilder = profileBuilder;
    }

    public boolean isBodyBarrier() {
        return bodyBarrier;
    }

    public void setBodyBarrier(boolean bodyBarrier) {
        this.bodyBarrier = bodyBarrier;
    }

    /** bodyBarrier effet */
    boolean bodyBarrier = false;

    /** Compute horizontal diffraction rays over vertical edges */
    public boolean computeHorizontalDiffraction = true;

    /** True will compute vertical diffraction over horizontal edges */
    public  boolean computeVerticalDiffraction;

    /** Maximum source distance */
    public double maxSrcDist = DEFAULT_MAX_PROPAGATION_DISTANCE;
    /** Maximum reflection wall distance from receiver to source line */
    public double maxRefDist = DEFAULT_MAXIMUM_REF_DIST;


    /**
     * Add the geometry of the source
     * @param geom
     */
    public void addSource(Geometry geom) {
        sourceGeometries.add(geom);
        sourcesIndex.appendGeometry(geom, sourceGeometries.size() - 1);
    }

    /**
     * Add geometry with additional attributes
     * @param pk Unique source identifier
     * @param geom Source geometry
     */
    public void addSource(Long pk, Geometry geom) {
        addSource(geom);
        sourcesPk.add(pk);
    }

    /**
     * Add geometry with additional attributes
     * @param pk Unique source identifier
     * @param geom Source geometry
     * @param orientation Additional attributes
     */
    public void addSource(Long pk, Geometry geom, Orientation orientation) {
        addSource(pk, geom);
        sourceOrientation.put(pk, orientation);
    }

    /**
     * Replace the sources by the given list
     * @param sourceGeometries
     */
    public void setSources(List<Geometry> sourceGeometries) {
        sourcesIndex = new QueryRTree();
        int i = 0;
        for(Geometry source : sourceGeometries) {
            sourcesIndex.appendGeometry(source, i++);
        }
        this.sourceGeometries = sourceGeometries;
    }

    /**
     *
     * @param receiver
     */
    public void addReceiver(Coordinate... receiver) {
        receivers.addAll(Arrays.asList(receiver));
    }

    public void addReceiver(long pk, Coordinate position) {
        receivers.add(position);
        receiversPk.add(pk);
    }

    public void addReceiver(long pk, Coordinate position, SpatialResultSet rs) {
        addReceiver(pk, position);
    }

    public int getReflexionOrder() {
        return reflexionOrder;
    }

    public void setReflexionOrder(int reflexionOrder) {
        this.reflexionOrder = reflexionOrder;
    }

    public void setComputeHorizontalDiffraction(boolean computeHorizontalDiffraction) {
        this.computeHorizontalDiffraction = computeHorizontalDiffraction;
    }

    public void setComputeVerticalDiffraction(boolean computeVerticalDiffraction) {
        this.computeVerticalDiffraction = computeVerticalDiffraction;
    }

    public void setDefaultGroundAttenuation(double gS) {
        this.defaultGroundAttenuation = gS;
    }

    public void clearSources() {
        sourceGeometries.clear();
        sourceOrientation.clear();
        sourcesPk.clear();
        sourcesIndex = new QueryRTree();
    }
}


