/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.commandline.admin.security;

import java.io.IOException;
import java.io.PrintStream;
import java.nio.file.CopyOption;
import java.nio.file.Path;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;
import org.neo4j.cli.ExecutionContext;
import org.neo4j.commandline.admin.security.SetInitialPasswordCommand;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.kernel.impl.security.User;
import org.neo4j.logging.LogProvider;
import org.neo4j.logging.NullLogProvider;
import org.neo4j.server.security.auth.FileUserRepository;
import org.neo4j.string.UTF8;
import org.neo4j.test.extension.EphemeralFileSystemExtension;
import org.neo4j.test.extension.Inject;
import picocli.CommandLine;

@ExtendWith(value={EphemeralFileSystemExtension.class})
class SetInitialPasswordCommandIT {
    @Inject
    private FileSystemAbstraction fileSystem;
    private Path confDir;
    private Path homeDir;
    private PrintStream out;
    private PrintStream err;
    private static final String successMessage = "Changed password for user 'neo4j'. IMPORTANT: this change will only take effect if performed before the database is started for the first time.";

    SetInitialPasswordCommandIT() {
    }

    @BeforeEach
    void setup() {
        Path graphDir = Path.of("neo4j", new String[0]);
        this.confDir = graphDir.resolve("conf");
        this.homeDir = graphDir.resolve("home");
        this.out = (PrintStream)Mockito.mock(PrintStream.class);
        this.err = (PrintStream)Mockito.mock(PrintStream.class);
    }

    @AfterEach
    void tearDown() throws Exception {
        this.fileSystem.close();
    }

    @Test
    void shouldSetPassword() throws Throwable {
        this.executeCommand("abc");
        this.assertAuthIniFile("abc", false);
        ((PrintStream)Mockito.verify((Object)this.out)).println(successMessage);
    }

    @Test
    void shouldSetPasswordWithRequirePasswordChange() throws Throwable {
        this.executeCommand("abc", "--require-password-change");
        this.assertAuthIniFile("abc", true);
        ((PrintStream)Mockito.verify((Object)this.out)).println(successMessage);
    }

    @Test
    void shouldSetPasswordWithRequirePasswordChangeOtherOrder() throws Throwable {
        this.executeCommand("--require-password-change", "abc");
        this.assertAuthIniFile("abc", true);
        ((PrintStream)Mockito.verify((Object)this.out)).println(successMessage);
    }

    @Test
    void shouldOverwriteIfSetPasswordAgain() throws Throwable {
        this.executeCommand("abc");
        this.assertAuthIniFile("abc", false);
        this.executeCommand("muchBetter");
        this.assertAuthIniFile("muchBetter", false);
        ((PrintStream)Mockito.verify((Object)this.out, (VerificationMode)Mockito.times((int)2))).println(successMessage);
    }

    @Test
    void shouldWorkWithSamePassword() throws Throwable {
        this.executeCommand("neo4j");
        this.assertAuthIniFile("neo4j", false);
        this.executeCommand("neo4j");
        this.assertAuthIniFile("neo4j", false);
        ((PrintStream)Mockito.verify((Object)this.out, (VerificationMode)Mockito.times((int)2))).println(successMessage);
    }

    @Test
    void shouldErrorIfRealUsersAlreadyExistCommunity() throws Throwable {
        Path authFile = this.getAuthFile("auth");
        this.fileSystem.mkdirs(authFile.getParent());
        this.fileSystem.write(authFile);
        Exception e = (Exception)org.junit.jupiter.api.Assertions.assertThrows(Exception.class, () -> this.executeCommand("will-be-ignored"));
        Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"the provided initial password was not set because existing Neo4j users were detected"});
        this.assertNoAuthIniFile();
    }

    @Test
    void shouldErrorIfRealUsersAlreadyExistEnterprise() throws Throwable {
        Path authFile = this.getAuthFile("auth");
        Path rolesFile = this.getAuthFile("roles");
        this.fileSystem.mkdirs(authFile.getParent());
        this.fileSystem.write(authFile);
        this.fileSystem.write(rolesFile);
        Exception e = (Exception)org.junit.jupiter.api.Assertions.assertThrows(Exception.class, () -> this.executeCommand("will-be-ignored"));
        Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"the provided initial password was not set because existing Neo4j users were detected"});
        this.assertNoAuthIniFile();
    }

    @Test
    void shouldErrorIfRealUsersAlreadyExistV2() throws Throwable {
        this.executeCommand("not-the-default-password");
        Path authFile = this.getAuthFile("auth");
        this.fileSystem.mkdirs(authFile.getParent());
        this.fileSystem.renameFile(this.getAuthFile("auth.ini"), authFile, new CopyOption[0]);
        Exception e = (Exception)org.junit.jupiter.api.Assertions.assertThrows(Exception.class, () -> this.executeCommand("will-be-ignored"));
        Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"the provided initial password was not set because existing Neo4j users were detected"});
        this.assertNoAuthIniFile();
        ((PrintStream)Mockito.verify((Object)this.out)).println(successMessage);
    }

    @Test
    void shouldNotErrorIfOnlyTheUnmodifiedDefaultNeo4jUserAlreadyExists() throws Throwable {
        this.executeCommand("neo4j");
        Path authFile = this.getAuthFile("auth");
        this.fileSystem.mkdirs(authFile.getParent());
        this.fileSystem.renameFile(this.getAuthFile("auth.ini"), authFile, new CopyOption[0]);
        this.executeCommand("should-not-be-ignored");
        this.assertAuthIniFile("should-not-be-ignored", false);
        ((PrintStream)Mockito.verify((Object)this.out, (VerificationMode)Mockito.times((int)2))).println(successMessage);
    }

    private void assertAuthIniFile(String password, boolean passwordChangeRequired) throws Throwable {
        Path authIniFile = this.getAuthFile("auth.ini");
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.fileSystem.fileExists(authIniFile));
        FileUserRepository userRepository = new FileUserRepository(this.fileSystem, authIniFile, (LogProvider)NullLogProvider.getInstance());
        userRepository.start();
        User neo4j = userRepository.getUserByName("neo4j");
        org.junit.jupiter.api.Assertions.assertNotNull((Object)neo4j);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)neo4j.credentials().matchesPassword(UTF8.encode((String)password)));
        Assertions.assertThat((boolean)neo4j.hasFlag("password_change_required")).isEqualTo(passwordChangeRequired);
    }

    private void assertNoAuthIniFile() {
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.fileSystem.fileExists(this.getAuthFile("auth.ini")));
    }

    private Path getAuthFile(String name) {
        return this.homeDir.resolve("data").resolve("dbms").resolve(name);
    }

    private void executeCommand(String ... args) throws IOException {
        ExecutionContext ctx = new ExecutionContext(this.homeDir, this.confDir, this.out, this.err, this.fileSystem);
        SetInitialPasswordCommand command = new SetInitialPasswordCommand(ctx);
        CommandLine.populateCommand((Object)command, (String[])args);
        command.execute();
    }
}

