/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.server.security.auth;

import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.Path;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.neo4j.io.fs.DelegatingFileSystemAbstraction;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.kernel.api.exceptions.InvalidArgumentsException;
import org.neo4j.kernel.impl.security.Credential;
import org.neo4j.kernel.impl.security.User;
import org.neo4j.logging.AssertableLogProvider;
import org.neo4j.logging.LogAssertions;
import org.neo4j.logging.LogProvider;
import org.neo4j.logging.NullLogProvider;
import org.neo4j.server.security.auth.FileRepositorySerializer;
import org.neo4j.server.security.auth.FileUserRepository;
import org.neo4j.server.security.auth.LegacyCredential;
import org.neo4j.server.security.auth.ListSnapshot;
import org.neo4j.string.UTF8;
import org.neo4j.test.DoubleLatch;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.testdirectory.EphemeralTestDirectoryExtension;
import org.neo4j.test.rule.TestDirectory;

@EphemeralTestDirectoryExtension
class FileUserRepositoryTest {
    @Inject
    private FileSystemAbstraction fs;
    @Inject
    private TestDirectory testDirectory;
    private final LogProvider logProvider = NullLogProvider.getInstance();
    private Path authFile;

    FileUserRepositoryTest() {
    }

    @BeforeEach
    void setUp() {
        this.authFile = this.testDirectory.directory("dbms").resolve("auth");
    }

    @Test
    void shouldStoreAndRetrieveUsersByName() throws Exception {
        FileUserRepository users = new FileUserRepository(this.fs, this.authFile, this.logProvider);
        User user = new User.Builder("jake", (Credential)LegacyCredential.INACCESSIBLE).withRequiredPasswordChange(true).build();
        users.create(user);
        User result = users.getUserByName(user.name());
        Assertions.assertThat((Object)result).isEqualTo((Object)user);
    }

    @Test
    void shouldPersistUsers() throws Throwable {
        FileUserRepository users = new FileUserRepository(this.fs, this.authFile, this.logProvider);
        User user = new User.Builder("jake", (Credential)LegacyCredential.INACCESSIBLE).withRequiredPasswordChange(true).build();
        users.create(user);
        users = new FileUserRepository(this.fs, this.authFile, this.logProvider);
        users.start();
        User resultByName = users.getUserByName(user.name());
        Assertions.assertThat((Object)resultByName).isEqualTo((Object)user);
    }

    @Test
    void shouldNotAllowComplexNames() throws Exception {
        FileUserRepository users = new FileUserRepository(this.fs, this.authFile, this.logProvider);
        users.assertValidUsername("neo4j");
        users.assertValidUsername("johnosbourne");
        users.assertValidUsername("john_osbourne");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> users.assertValidUsername(null)).isInstanceOf(InvalidArgumentsException.class)).hasMessage("The provided username is empty.");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> users.assertValidUsername("")).isInstanceOf(InvalidArgumentsException.class)).hasMessage("The provided username is empty.");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> users.assertValidUsername(",")).isInstanceOf(InvalidArgumentsException.class)).hasMessage("Username ',' contains illegal characters. Use ascii characters that are not ',', ':' or whitespaces.");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> users.assertValidUsername("with space")).isInstanceOf(InvalidArgumentsException.class)).hasMessage("Username 'with space' contains illegal characters. Use ascii characters that are not ',', ':' or whitespaces.");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> users.assertValidUsername("with:colon")).isInstanceOf(InvalidArgumentsException.class)).hasMessage("Username 'with:colon' contains illegal characters. Use ascii characters that are not ',', ':' or whitespaces.");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> users.assertValidUsername("with\u00e5")).isInstanceOf(InvalidArgumentsException.class)).hasMessage("Username 'with\u00e5' contains illegal characters. Use ascii characters that are not ',', ':' or whitespaces.");
    }

    @Test
    void shouldRecoverIfCrashedDuringMove() throws Throwable {
        final IOException exception = new IOException("simulated IO Exception on create");
        DelegatingFileSystemAbstraction crashingFileSystem = new DelegatingFileSystemAbstraction(this.fs){

            public void renameFile(Path oldLocation, Path newLocation, CopyOption ... copyOptions) throws IOException {
                if (FileUserRepositoryTest.this.authFile.getFileName().equals(newLocation.getFileName())) {
                    throw exception;
                }
                super.renameFile(oldLocation, newLocation, copyOptions);
            }
        };
        FileUserRepository users = new FileUserRepository((FileSystemAbstraction)crashingFileSystem, this.authFile, this.logProvider);
        users.start();
        User user = new User.Builder("jake", (Credential)LegacyCredential.INACCESSIBLE).withRequiredPasswordChange(true).build();
        IOException e = (IOException)org.junit.jupiter.api.Assertions.assertThrows(IOException.class, () -> users.create(user));
        org.junit.jupiter.api.Assertions.assertSame((Object)exception, (Object)e);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)crashingFileSystem.fileExists(this.authFile));
        Assertions.assertThat((int)crashingFileSystem.listFiles(this.authFile.getParent()).length).isEqualTo(0);
    }

    @Test
    void shouldFailOnReadingInvalidEntries() throws Throwable {
        AssertableLogProvider logProvider = new AssertableLogProvider();
        this.fs.mkdir(this.authFile.getParent());
        FileRepositorySerializer.writeToFile((FileSystemAbstraction)this.fs, (Path)this.authFile, (byte[])UTF8.encode((String)"admin:SHA-256,A42E541F276CF17036DB7818F8B09B1C229AAD52A17F69F4029617F3A554640F,FB7E8AE08A6A7C741F678AD22217808F:\nneo4j:fc4c600b43ffe4d5857b4439c35df88f:SHA-256,A42E541F276CF17036DB7818F8B09B1C229AAD52A17F69F4029617F3A554640F,FB7E8AE08A6A7C741F678AD22217808F:\n"));
        FileUserRepository users = new FileUserRepository(this.fs, this.authFile, (LogProvider)logProvider);
        IllegalStateException e = (IllegalStateException)org.junit.jupiter.api.Assertions.assertThrows(IllegalStateException.class, () -> ((FileUserRepository)users).start());
        Assertions.assertThat((String)e.getMessage()).startsWith((CharSequence)"Failed to read authentication file: ");
        Assertions.assertThat((int)users.numberOfUsers()).isEqualTo(0);
        LogAssertions.assertThat((AssertableLogProvider)logProvider).forClass(FileUserRepository.class).forLevel(AssertableLogProvider.Level.ERROR).containsMessageWithArguments("Failed to read authentication file \"%s\" (%s)", new Object[]{this.authFile.toAbsolutePath(), "wrong number of line fields, expected 3, got 4 [line 2]"});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void shouldProvideUserByUsernameEvenIfMidSetUsers() throws Throwable {
        FileUserRepository users = new FileUserRepository(this.fs, this.authFile, this.logProvider);
        users.create(new User.Builder("oskar", (Credential)LegacyCredential.forPassword((String)"hidden")).build());
        DoubleLatch latch = new DoubleLatch(2);
        ExecutorService executor = Executors.newSingleThreadExecutor();
        try {
            Future<?> setUsers = executor.submit(() -> {
                try {
                    users.setUsers((ListSnapshot)new HangingListSnapshot(latch, 10L, Collections.emptyList()));
                }
                catch (InvalidArgumentsException e) {
                    throw new RuntimeException(e);
                }
            });
            latch.startAndWaitForAllToStart();
            org.junit.jupiter.api.Assertions.assertNotNull((Object)users.getUserByName("oskar"));
            latch.finish();
            setUsers.get();
        }
        finally {
            executor.shutdown();
        }
    }

    static class HangingListSnapshot
    extends ListSnapshot<User> {
        private final DoubleLatch latch;

        HangingListSnapshot(DoubleLatch latch, long timestamp, List<User> values) {
            super(timestamp, values);
            this.latch = latch;
        }

        public long timestamp() {
            this.latch.start();
            this.latch.finishAndWaitForAllToFinish();
            return super.timestamp();
        }
    }
}

