/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.server.security.auth;

import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Map;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;
import org.neo4j.cypher.internal.security.SecureHasher;
import org.neo4j.internal.helpers.collection.MapUtil;
import org.neo4j.internal.kernel.api.security.AuthenticationResult;
import org.neo4j.internal.kernel.api.security.LoginContext;
import org.neo4j.kernel.api.exceptions.InvalidArgumentsException;
import org.neo4j.kernel.api.security.AuthToken;
import org.neo4j.kernel.api.security.exception.InvalidAuthTokenException;
import org.neo4j.kernel.impl.security.Credential;
import org.neo4j.kernel.impl.security.User;
import org.neo4j.server.security.auth.AuthenticationStrategy;
import org.neo4j.server.security.auth.SecurityTestUtils;
import org.neo4j.server.security.systemgraph.BasicSystemGraphRealm;
import org.neo4j.server.security.systemgraph.SystemGraphRealmHelper;

public class BasicSystemGraphRealmTest {
    private AuthenticationStrategy authStrategy;
    private SystemGraphRealmHelper realmHelper;
    private BasicSystemGraphRealm realm;

    @BeforeEach
    void setUp() {
        this.authStrategy = (AuthenticationStrategy)Mockito.mock(AuthenticationStrategy.class);
        this.realmHelper = (SystemGraphRealmHelper)Mockito.spy((Object)new SystemGraphRealmHelper(null, new SecureHasher()));
        this.realm = new BasicSystemGraphRealm(this.realmHelper, this.authStrategy);
    }

    @Test
    void shouldFindAndAuthenticateUserSuccessfully() throws Throwable {
        User user = new User.Builder("jake", (Credential)SecurityTestUtils.credentialFor("abc123")).build();
        ((SystemGraphRealmHelper)Mockito.doReturn((Object)user).when((Object)this.realmHelper)).getUser("jake");
        this.setMockAuthenticationStrategyResult(user, "abc123", AuthenticationResult.SUCCESS);
        this.assertLoginGivesResult("jake", "abc123", AuthenticationResult.SUCCESS);
    }

    @Test
    void shouldFindAndAuthenticateUserAndReturnAuthStrategyResult() throws Throwable {
        User user = new User.Builder("jake", (Credential)SecurityTestUtils.credentialFor("abc123")).build();
        ((SystemGraphRealmHelper)Mockito.doReturn((Object)user).when((Object)this.realmHelper)).getUser("jake");
        this.setMockAuthenticationStrategyResult(user, "abc123", AuthenticationResult.TOO_MANY_ATTEMPTS);
        this.assertLoginGivesResult("jake", "abc123", AuthenticationResult.TOO_MANY_ATTEMPTS);
    }

    @Test
    void shouldFindAndAuthenticateUserAndReturnPasswordChangeIfRequired() throws Throwable {
        User user = new User.Builder("jake", (Credential)SecurityTestUtils.credentialFor("abc123")).withRequiredPasswordChange(true).build();
        ((SystemGraphRealmHelper)Mockito.doReturn((Object)user).when((Object)this.realmHelper)).getUser("jake");
        this.setMockAuthenticationStrategyResult(user, "abc123", AuthenticationResult.SUCCESS);
        this.assertLoginGivesResult("jake", "abc123", AuthenticationResult.PASSWORD_CHANGE_REQUIRED);
    }

    @Test
    void shouldFailAuthenticationIfUserIsNotFound() throws Throwable {
        ((SystemGraphRealmHelper)Mockito.doThrow((Throwable[])new Throwable[]{new InvalidArgumentsException("User 'unknown' does not exist.")}).when((Object)this.realmHelper)).getUser("unknown");
        this.assertLoginGivesResult("unknown", "abc123", AuthenticationResult.FAILURE);
    }

    @Test
    void shouldClearPasswordOnLogin() throws Throwable {
        Mockito.when((Object)this.authStrategy.authenticate((User)ArgumentMatchers.any(), (byte[])ArgumentMatchers.any())).thenReturn((Object)AuthenticationResult.SUCCESS);
        User user = new User.Builder("jake", (Credential)SecurityTestUtils.credentialFor("abc123")).withRequiredPasswordChange(true).build();
        ((SystemGraphRealmHelper)Mockito.doReturn((Object)user).when((Object)this.realmHelper)).getUser("jake");
        byte[] password = SecurityTestUtils.password("abc123");
        Map authToken = AuthToken.newBasicAuthToken((String)"jake", (byte[])password);
        this.realm.login(authToken);
        Assertions.assertThat((byte[])password).isEqualTo((Object)BasicSystemGraphRealmTest.clearedPasswordWithSameLengthAs("abc123"));
        Assertions.assertThat(authToken.get("credentials")).isEqualTo((Object)BasicSystemGraphRealmTest.clearedPasswordWithSameLengthAs("abc123"));
    }

    @Test
    void shouldClearPasswordOnInvalidAuthToken() {
        byte[] password = SecurityTestUtils.password("abc123");
        Map authToken = AuthToken.newBasicAuthToken((String)"jake", (byte[])password);
        authToken.put("scheme", null);
        try {
            this.realm.login(authToken);
            org.junit.jupiter.api.Assertions.fail((String)"exception expected");
        }
        catch (InvalidAuthTokenException invalidAuthTokenException) {
            // empty catch block
        }
        Assertions.assertThat((byte[])password).isEqualTo((Object)BasicSystemGraphRealmTest.clearedPasswordWithSameLengthAs("abc123"));
        Assertions.assertThat(authToken.get("credentials")).isEqualTo((Object)BasicSystemGraphRealmTest.clearedPasswordWithSameLengthAs("abc123"));
    }

    @Test
    void shouldFailWhenAuthTokenIsInvalid() {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.realm.login(MapUtil.map((Object[])new Object[]{"scheme", "supercool", "principal", "neo4j"}))).isInstanceOf(InvalidAuthTokenException.class)).hasMessage("Unsupported authentication token, scheme 'supercool' is not supported.");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.realm.login(MapUtil.map((Object[])new Object[]{"scheme", "none"}))).isInstanceOf(InvalidAuthTokenException.class)).hasMessage("Unsupported authentication token, scheme 'none' is only allowed when auth is disabled.");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.realm.login(MapUtil.map((Object[])new Object[]{"key", "value"}))).isInstanceOf(InvalidAuthTokenException.class)).hasMessage("Unsupported authentication token, missing key `scheme`");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.realm.login(MapUtil.map((Object[])new Object[]{"scheme", "basic", "principal", "neo4j"}))).isInstanceOf(InvalidAuthTokenException.class)).hasMessage("Unsupported authentication token, missing key `credentials`");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.realm.login(MapUtil.map((Object[])new Object[]{"scheme", "basic", "credentials", "very-secret"}))).isInstanceOf(InvalidAuthTokenException.class)).hasMessage("Unsupported authentication token, missing key `principal`");
    }

    private void assertLoginGivesResult(String username, String password, AuthenticationResult expectedResult) throws InvalidAuthTokenException {
        LoginContext securityContext = this.realm.login(SecurityTestUtils.authToken(username, password));
        Assertions.assertThat((Comparable)securityContext.subject().getAuthenticationResult()).isEqualTo((Object)expectedResult);
    }

    private void setMockAuthenticationStrategyResult(User user, String password, AuthenticationResult result) {
        Mockito.when((Object)this.authStrategy.authenticate(user, SecurityTestUtils.password(password))).thenReturn((Object)result);
    }

    public static byte[] clearedPasswordWithSameLengthAs(String passwordString) {
        byte[] password = passwordString.getBytes(StandardCharsets.UTF_8);
        Arrays.fill(password, (byte)0);
        return password;
    }
}

