/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.server.security.auth;

import java.io.File;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Future;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.neo4j.graphdb.mockfs.DelegatingFileSystemAbstraction;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.kernel.api.exceptions.InvalidArgumentsException;
import org.neo4j.kernel.impl.security.Credential;
import org.neo4j.kernel.impl.security.User;
import org.neo4j.logging.AssertableLogProvider;
import org.neo4j.logging.LogProvider;
import org.neo4j.logging.NullLogProvider;
import org.neo4j.server.security.auth.FileRepositorySerializer;
import org.neo4j.server.security.auth.FileUserRepository;
import org.neo4j.server.security.auth.LegacyCredential;
import org.neo4j.server.security.auth.ListSnapshot;
import org.neo4j.server.security.auth.exception.ConcurrentModificationException;
import org.neo4j.string.UTF8;
import org.neo4j.test.DoubleLatch;
import org.neo4j.test.rule.TestDirectory;
import org.neo4j.test.rule.concurrent.ThreadingRule;
import org.neo4j.test.rule.fs.DefaultFileSystemRule;

public class FileUserRepositoryTest {
    private File authFile;
    private final LogProvider logProvider = NullLogProvider.getInstance();
    private FileSystemAbstraction fs;
    @Rule
    public final TestDirectory testDirectory = TestDirectory.testDirectory();
    @Rule
    public final ExpectedException thrown = ExpectedException.none();
    @Rule
    public final ThreadingRule threading = new ThreadingRule();
    @Rule
    public final DefaultFileSystemRule fileSystemRule = new DefaultFileSystemRule();

    @Before
    public void setUp() {
        this.fs = this.fileSystemRule.get();
        this.authFile = new File(this.testDirectory.directory("dbms"), "auth");
    }

    @Test
    public void shouldStoreAndRetrieveUsersByName() throws Exception {
        FileUserRepository users = new FileUserRepository(this.fs, this.authFile, this.logProvider);
        User user = new User.Builder("jake", (Credential)LegacyCredential.INACCESSIBLE).withRequiredPasswordChange(true).build();
        users.create(user);
        User result = users.getUserByName(user.name());
        MatcherAssert.assertThat((Object)result, (Matcher)CoreMatchers.equalTo((Object)user));
    }

    @Test
    public void shouldPersistUsers() throws Throwable {
        FileUserRepository users = new FileUserRepository(this.fs, this.authFile, this.logProvider);
        User user = new User.Builder("jake", (Credential)LegacyCredential.INACCESSIBLE).withRequiredPasswordChange(true).build();
        users.create(user);
        users = new FileUserRepository(this.fs, this.authFile, this.logProvider);
        users.start();
        User resultByName = users.getUserByName(user.name());
        MatcherAssert.assertThat((Object)resultByName, (Matcher)CoreMatchers.equalTo((Object)user));
    }

    @Test
    public void shouldNotFindUserAfterDelete() throws Throwable {
        FileUserRepository users = new FileUserRepository(this.fs, this.authFile, this.logProvider);
        User user = new User.Builder("jake", (Credential)LegacyCredential.INACCESSIBLE).withRequiredPasswordChange(true).build();
        users.create(user);
        users.delete(user);
        MatcherAssert.assertThat((Object)users.getUserByName(user.name()), (Matcher)CoreMatchers.nullValue());
    }

    @Test
    public void shouldNotAllowComplexNames() throws Exception {
        FileUserRepository users = new FileUserRepository(this.fs, this.authFile, this.logProvider);
        users.assertValidUsername("neo4j");
        users.assertValidUsername("johnosbourne");
        users.assertValidUsername("john_osbourne");
        org.neo4j.test.assertion.Assert.assertException(() -> users.assertValidUsername(null), InvalidArgumentsException.class, (String)"The provided username is empty.");
        org.neo4j.test.assertion.Assert.assertException(() -> users.assertValidUsername(""), InvalidArgumentsException.class, (String)"The provided username is empty.");
        org.neo4j.test.assertion.Assert.assertException(() -> users.assertValidUsername(","), InvalidArgumentsException.class, (String)"Username ',' contains illegal characters. Use ascii characters that are not ',', ':' or whitespaces.");
        org.neo4j.test.assertion.Assert.assertException(() -> users.assertValidUsername("with space"), InvalidArgumentsException.class, (String)"Username 'with space' contains illegal characters. Use ascii characters that are not ',', ':' or whitespaces.");
        org.neo4j.test.assertion.Assert.assertException(() -> users.assertValidUsername("with:colon"), InvalidArgumentsException.class, (String)"Username 'with:colon' contains illegal characters. Use ascii characters that are not ',', ':' or whitespaces.");
        org.neo4j.test.assertion.Assert.assertException(() -> users.assertValidUsername("with\u00e5"), InvalidArgumentsException.class, (String)"Username 'with\u00e5' contains illegal characters. Use ascii characters that are not ',', ':' or whitespaces.");
    }

    @Test
    public void shouldRecoverIfCrashedDuringMove() throws Throwable {
        final IOException exception = new IOException("simulated IO Exception on create");
        DelegatingFileSystemAbstraction crashingFileSystem = new DelegatingFileSystemAbstraction(this.fs){

            public void renameFile(File oldLocation, File newLocation, CopyOption ... copyOptions) throws IOException {
                if (FileUserRepositoryTest.this.authFile.getName().equals(newLocation.getName())) {
                    throw exception;
                }
                super.renameFile(oldLocation, newLocation, copyOptions);
            }
        };
        FileUserRepository users = new FileUserRepository((FileSystemAbstraction)crashingFileSystem, this.authFile, this.logProvider);
        users.start();
        User user = new User.Builder("jake", (Credential)LegacyCredential.INACCESSIBLE).withRequiredPasswordChange(true).build();
        try {
            users.create(user);
            Assert.fail((String)"Expected an IOException");
        }
        catch (IOException e) {
            Assert.assertSame((Object)exception, (Object)e);
        }
        Assert.assertFalse((boolean)crashingFileSystem.fileExists(this.authFile));
        MatcherAssert.assertThat((Object)crashingFileSystem.listFiles(this.authFile.getParentFile()).length, (Matcher)CoreMatchers.equalTo((Object)0));
    }

    @Test
    public void shouldThrowIfUpdateChangesName() throws Throwable {
        FileUserRepository users = new FileUserRepository(this.fs, this.authFile, this.logProvider);
        User user = new User.Builder("jake", (Credential)LegacyCredential.INACCESSIBLE).withRequiredPasswordChange(true).build();
        users.create(user);
        User updatedUser = new User.Builder("john", (Credential)LegacyCredential.INACCESSIBLE).withRequiredPasswordChange(true).build();
        try {
            users.update(user, updatedUser);
            Assert.fail((String)"expected exception not thrown");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        MatcherAssert.assertThat((Object)users.getUserByName(user.name()), (Matcher)CoreMatchers.equalTo((Object)user));
    }

    @Test
    public void shouldThrowIfExistingUserDoesNotMatch() throws Throwable {
        FileUserRepository users = new FileUserRepository(this.fs, this.authFile, this.logProvider);
        User user = new User.Builder("jake", (Credential)LegacyCredential.INACCESSIBLE).withRequiredPasswordChange(true).build();
        users.create(user);
        User modifiedUser = user.augment().withCredentials((Credential)LegacyCredential.forPassword((String)"foo")).build();
        User updatedUser = user.augment().withCredentials((Credential)LegacyCredential.forPassword((String)"bar")).build();
        try {
            users.update(modifiedUser, updatedUser);
            Assert.fail((String)"expected exception not thrown");
        }
        catch (ConcurrentModificationException concurrentModificationException) {
            // empty catch block
        }
    }

    @Test
    public void shouldFailOnReadingInvalidEntries() throws Throwable {
        AssertableLogProvider logProvider = new AssertableLogProvider();
        this.fs.mkdir(this.authFile.getParentFile());
        FileRepositorySerializer.writeToFile((FileSystemAbstraction)this.fs, (File)this.authFile, (byte[])UTF8.encode((String)"admin:SHA-256,A42E541F276CF17036DB7818F8B09B1C229AAD52A17F69F4029617F3A554640F,FB7E8AE08A6A7C741F678AD22217808F:\nneo4j:fc4c600b43ffe4d5857b4439c35df88f:SHA-256,A42E541F276CF17036DB7818F8B09B1C229AAD52A17F69F4029617F3A554640F,FB7E8AE08A6A7C741F678AD22217808F:\n"));
        FileUserRepository users = new FileUserRepository(this.fs, this.authFile, (LogProvider)logProvider);
        this.thrown.expect(IllegalStateException.class);
        this.thrown.expectMessage(CoreMatchers.startsWith((String)"Failed to read authentication file: "));
        try {
            users.start();
        }
        catch (IllegalStateException e) {
            MatcherAssert.assertThat((Object)users.numberOfUsers(), (Matcher)CoreMatchers.equalTo((Object)0));
            logProvider.assertExactly(new AssertableLogProvider.LogMatcher[]{AssertableLogProvider.inLog(FileUserRepository.class).error("Failed to read authentication file \"%s\" (%s)", new Object[]{this.authFile.getAbsolutePath(), "wrong number of line fields, expected 3, got 4 [line 2]"})});
            throw e;
        }
    }

    @Test
    public void shouldProvideUserByUsernameEvenIfMidSetUsers() throws Throwable {
        FileUserRepository users = new FileUserRepository(this.fs, this.authFile, this.logProvider);
        users.create(new User.Builder("oskar", (Credential)LegacyCredential.forPassword((String)"hidden")).build());
        DoubleLatch latch = new DoubleLatch(2);
        Future setUsers = this.threading.execute(o -> {
            users.setUsers((ListSnapshot)new HangingListSnapshot(latch, 10L, Collections.emptyList()));
            return null;
        }, null);
        latch.startAndWaitForAllToStart();
        Assert.assertNotNull((Object)users.getUserByName("oskar"));
        latch.finish();
        setUsers.get();
    }

    class HangingListSnapshot
    extends ListSnapshot<User> {
        private final DoubleLatch latch;

        HangingListSnapshot(DoubleLatch latch, long timestamp, List<User> values) {
            super(timestamp, values, true);
            this.latch = latch;
        }

        public long timestamp() {
            this.latch.start();
            this.latch.finishAndWaitForAllToFinish();
            return super.timestamp();
        }
    }
}

