/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.server.security.auth;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.hamcrest.core.IsEqual;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.ResourceIterator;
import org.neo4j.graphdb.config.Setting;
import org.neo4j.graphdb.factory.GraphDatabaseSettings;
import org.neo4j.graphdb.mockfs.EphemeralFileSystemAbstraction;
import org.neo4j.helpers.collection.MapUtil;
import org.neo4j.internal.kernel.api.Transaction;
import org.neo4j.internal.kernel.api.security.AuthenticationResult;
import org.neo4j.internal.kernel.api.security.LoginContext;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.kernel.api.exceptions.InvalidArgumentsException;
import org.neo4j.kernel.api.security.exception.InvalidAuthTokenException;
import org.neo4j.kernel.impl.coreapi.InternalTransaction;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.server.security.auth.BasicAuthManager;
import org.neo4j.server.security.auth.SecurityTestUtils;
import org.neo4j.test.TestGraphDatabaseBuilder;
import org.neo4j.test.TestGraphDatabaseFactory;

public class AuthProceduresIT {
    private static final String PWD_CHANGE = AuthenticationResult.PASSWORD_CHANGE_REQUIRED.name().toLowerCase();
    protected GraphDatabaseAPI db;
    private EphemeralFileSystemAbstraction fs;
    private BasicAuthManager authManager;
    private LoginContext admin;

    @Before
    public void setup() throws InvalidAuthTokenException, IOException {
        this.fs = new EphemeralFileSystemAbstraction();
        this.db = (GraphDatabaseAPI)this.createGraphDatabase(this.fs);
        this.authManager = (BasicAuthManager)this.db.getDependencyResolver().resolveDependency(BasicAuthManager.class);
        this.admin = this.login("neo4j", "neo4j");
        this.admin.subject().setPasswordChangeNoLongerRequired();
    }

    @After
    public void cleanup() throws Exception {
        this.db.shutdown();
        this.fs.close();
    }

    @Test
    public void shouldChangePassword() throws Throwable {
        this.assertEmpty(this.admin, "CALL dbms.changePassword('abc')");
        assert (this.authManager.getUser("neo4j").credentials().matchesPassword("abc"));
    }

    @Test
    public void shouldNotChangeOwnPasswordIfNewPasswordInvalid() {
        this.assertFail(this.admin, "CALL dbms.changePassword( '' )", "A password cannot be empty.");
        this.assertFail(this.admin, "CALL dbms.changePassword( 'neo4j' )", "Old password and new password cannot be the same.");
    }

    @Test
    public void newUserShouldBeAbleToChangePassword() throws Throwable {
        this.authManager.newUser("andres", "banana", true);
        this.assertEmpty(this.login("andres", "banana"), "CALL dbms.changePassword('abc')");
    }

    @Test
    public void newUserShouldNotBeAbleToCallOtherProcedures() throws Throwable {
        this.authManager.newUser("andres", "banana", true);
        LoginContext user = this.login("andres", "banana");
        this.assertFail(user, "CALL dbms.procedures", "The credentials you provided were valid, but must be changed before you can use this instance.");
    }

    @Test
    public void shouldCreateUser() {
        this.assertEmpty(this.admin, "CALL dbms.security.createUser('andres', '123', true)");
        try {
            MatcherAssert.assertThat((Object)this.authManager.getUser("andres").passwordChangeRequired(), (Matcher)IsEqual.equalTo((Object)true));
        }
        catch (Throwable t) {
            Assert.fail((String)"Expected no exception!");
        }
    }

    @Test
    public void shouldCreateUserWithNoPasswordChange() {
        this.assertEmpty(this.admin, "CALL dbms.security.createUser('andres', '123', false)");
        try {
            MatcherAssert.assertThat((Object)this.authManager.getUser("andres").passwordChangeRequired(), (Matcher)IsEqual.equalTo((Object)false));
        }
        catch (Throwable t) {
            Assert.fail((String)"Expected no exception!");
        }
    }

    @Test
    public void shouldCreateUserWithDefault() {
        this.assertEmpty(this.admin, "CALL dbms.security.createUser('andres', '123')");
        try {
            MatcherAssert.assertThat((Object)this.authManager.getUser("andres").passwordChangeRequired(), (Matcher)IsEqual.equalTo((Object)true));
        }
        catch (Throwable t) {
            Assert.fail((String)"Expected no exception!");
        }
    }

    @Test
    public void shouldNotCreateUserIfInvalidUsername() {
        this.assertFail(this.admin, "CALL dbms.security.createUser('', '1234', true)", "The provided username is empty.");
        this.assertFail(this.admin, "CALL dbms.security.createUser(',!', '1234', true)", "Username ',!' contains illegal characters.");
        this.assertFail(this.admin, "CALL dbms.security.createUser(':ss!', '', true)", "Username ':ss!' contains illegal characters.");
    }

    @Test
    public void shouldNotCreateUserIfInvalidPassword() {
        this.assertFail(this.admin, "CALL dbms.security.createUser('andres', '', true)", "A password cannot be empty.");
    }

    @Test
    public void shouldNotCreateExistingUser() {
        this.assertFail(this.admin, "CALL dbms.security.createUser('neo4j', '1234', true)", "The specified user 'neo4j' already exists");
        this.assertFail(this.admin, "CALL dbms.security.createUser('neo4j', '', true)", "A password cannot be empty.");
    }

    @Test
    public void shouldDeleteUser() throws Exception {
        this.authManager.newUser("andres", "123", false);
        this.assertEmpty(this.admin, "CALL dbms.security.deleteUser('andres')");
        try {
            this.authManager.getUser("andres");
            Assert.fail((String)"Andres should no longer exist, expected exception.");
        }
        catch (InvalidArgumentsException e) {
            MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)Matchers.containsString((String)"User 'andres' does not exist."));
        }
        catch (Throwable t) {
            MatcherAssert.assertThat(t.getClass(), (Matcher)IsEqual.equalTo(InvalidArgumentsException.class));
        }
    }

    @Test
    public void shouldNotDeleteNonExistentUser() {
        this.assertFail(this.admin, "CALL dbms.security.deleteUser('nonExistentUser')", "User 'nonExistentUser' does not exist");
    }

    @Test
    public void shouldListUsers() throws Exception {
        this.authManager.newUser("andres", "123", false);
        this.assertSuccess(this.admin, "CALL dbms.security.listUsers() YIELD username", r -> this.assertKeyIs((ResourceIterator<Map<String, Object>>)r, "username", "neo4j", "andres"));
    }

    @Test
    public void shouldReturnUsersWithFlags() throws Exception {
        this.authManager.newUser("andres", "123", false);
        Map expected = MapUtil.map((Object[])new Object[]{"neo4j", this.listOf(PWD_CHANGE), "andres", this.listOf(new String[0])});
        this.assertSuccess(this.admin, "CALL dbms.security.listUsers()", r -> AuthProceduresIT.assertKeyIsMap((ResourceIterator<Map<String, Object>>)r, "username", "flags", expected));
    }

    @Test
    public void shouldShowCurrentUser() throws Exception {
        this.assertSuccess(this.admin, "CALL dbms.showCurrentUser()", r -> AuthProceduresIT.assertKeyIsMap((ResourceIterator<Map<String, Object>>)r, "username", "flags", MapUtil.map((Object[])new Object[]{"neo4j", this.listOf(PWD_CHANGE)})));
        this.authManager.newUser("andres", "123", false);
        LoginContext andres = this.login("andres", "123");
        this.assertSuccess(andres, "CALL dbms.showCurrentUser()", r -> AuthProceduresIT.assertKeyIsMap((ResourceIterator<Map<String, Object>>)r, "username", "flags", MapUtil.map((Object[])new Object[]{"andres", this.listOf(new String[0])})));
    }

    private GraphDatabaseService createGraphDatabase(EphemeralFileSystemAbstraction fs) throws IOException {
        this.removePreviousAuthFile();
        HashMap<Setting, String> settings = new HashMap<Setting, String>();
        settings.put(GraphDatabaseSettings.auth_enabled, "true");
        TestGraphDatabaseBuilder graphDatabaseFactory = (TestGraphDatabaseBuilder)new TestGraphDatabaseFactory().setFileSystem((FileSystemAbstraction)fs).newImpermanentDatabaseBuilder().setConfig(GraphDatabaseSettings.auth_enabled, "true");
        return graphDatabaseFactory.newGraphDatabase();
    }

    private void removePreviousAuthFile() throws IOException {
        Path file = Paths.get("target/test-data/impermanent-db/data/dbms/auth", new String[0]);
        if (Files.exists(file, new LinkOption[0])) {
            Files.delete(file);
        }
    }

    private LoginContext login(String username, String password) throws InvalidAuthTokenException {
        return this.authManager.login(SecurityTestUtils.authToken(username, password));
    }

    private void assertEmpty(LoginContext subject, String query) {
        MatcherAssert.assertThat((Object)this.execute(subject, query, r -> {
            assert (!r.hasNext());
        }), (Matcher)IsEqual.equalTo((Object)""));
    }

    private void assertFail(LoginContext subject, String query, String partOfErrorMsg) {
        MatcherAssert.assertThat((Object)this.execute(subject, query, r -> {
            assert (!r.hasNext());
        }), (Matcher)Matchers.containsString((String)partOfErrorMsg));
    }

    private void assertSuccess(LoginContext subject, String query, Consumer<ResourceIterator<Map<String, Object>>> resultConsumer) {
        MatcherAssert.assertThat((Object)this.execute(subject, query, resultConsumer), (Matcher)IsEqual.equalTo((Object)""));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private String execute(LoginContext subject, String query, Consumer<ResourceIterator<Map<String, Object>>> resultConsumer) {
        try (InternalTransaction tx = this.db.beginTransaction(Transaction.Type.implicit, subject);){
            resultConsumer.accept((ResourceIterator<Map<String, Object>>)this.db.execute(query));
            tx.success();
            String string = "";
            return string;
        }
        catch (Exception e) {
            return e.getMessage();
        }
    }

    private List<Object> getObjectsAsList(ResourceIterator<Map<String, Object>> r, String key) {
        return r.stream().map(s -> s.get(key)).collect(Collectors.toList());
    }

    private void assertKeyIs(ResourceIterator<Map<String, Object>> r, String key, String ... items) {
        this.assertKeyIsArray(r, key, items);
    }

    private void assertKeyIsArray(ResourceIterator<Map<String, Object>> r, String key, String[] items) {
        List<Object> results = this.getObjectsAsList(r, key);
        Assert.assertEquals((long)Arrays.asList(items).size(), (long)results.size());
        Assert.assertThat(results, (Matcher)Matchers.containsInAnyOrder((Object[])items));
    }

    protected String[] with(String[] strs, String ... moreStr) {
        return (String[])Stream.concat(Arrays.stream(strs), Arrays.stream(moreStr)).toArray(String[]::new);
    }

    private List<String> listOf(String ... values) {
        return Stream.of(values).collect(Collectors.toList());
    }

    public static void assertKeyIsMap(ResourceIterator<Map<String, Object>> r, String keyKey, String valueKey, Map<String, Object> expected) {
        List result = r.stream().collect(Collectors.toList());
        Assert.assertEquals((String)("Results for should have size " + expected.size() + " but was " + result.size()), (long)expected.size(), (long)result.size());
        for (Map row : result) {
            Object value;
            String key = (String)row.get(keyKey);
            Assert.assertTrue((String)("Unexpected key '" + key + "'"), (boolean)expected.containsKey(key));
            Assert.assertTrue((String)("Value key '" + valueKey + "' not found in results"), (boolean)row.containsKey(valueKey));
            Object objectValue = row.get(valueKey);
            if (objectValue instanceof List) {
                value = (List)objectValue;
                List expectedValues = (List)expected.get(key);
                Assert.assertEquals((String)("Results for '" + key + "' should have size " + expectedValues.size() + " but was " + value.size()), (long)value.size(), (long)expectedValues.size());
                MatcherAssert.assertThat((Object)value, (Matcher)Matchers.containsInAnyOrder((Object[])expectedValues.toArray()));
                continue;
            }
            value = objectValue.toString();
            String expectedValue = expected.get(key).toString();
            Assert.assertTrue((String)String.format("Wrong value for '%s', expected '%s', got '%s'", key, expectedValue, value), (boolean)((String)value).equals(expectedValue));
        }
    }
}

