/*
 * Copyright (c) 2002-2017 "Neo Technology,"
 * Network Engine for Objects in Lund AB [http://neotechnology.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.neo4j.kernel.api.impl.schema.populator;

import org.junit.Test;

import java.util.Arrays;
import java.util.List;

import org.neo4j.collection.primitive.PrimitiveLongCollections;
import org.neo4j.kernel.api.impl.schema.LuceneDocumentStructure;
import org.neo4j.kernel.api.impl.schema.SchemaIndex;
import org.neo4j.kernel.api.impl.schema.writer.LuceneIndexWriter;
import org.neo4j.kernel.api.index.PropertyAccessor;
import org.neo4j.kernel.api.schema_new.LabelSchemaDescriptor;
import org.neo4j.kernel.api.schema_new.SchemaDescriptorFactory;
import org.neo4j.kernel.impl.api.index.sampling.UniqueIndexSampler;
import org.neo4j.storageengine.api.schema.IndexSample;

import static org.hamcrest.Matchers.instanceOf;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyZeroInteractions;
import static org.neo4j.kernel.api.impl.schema.LuceneDocumentStructure.newTermForChangeOrRemove;
import static org.neo4j.kernel.api.index.IndexEntryUpdate.add;
import static org.neo4j.kernel.api.index.IndexEntryUpdate.change;
import static org.neo4j.kernel.api.index.IndexEntryUpdate.remove;

public class UniqueDatabaseIndexPopulatingUpdaterTest
{
    private static final LabelSchemaDescriptor descriptor = SchemaDescriptorFactory.forLabel( 1, 42 );

    @Test
    public void removeNotSupported()
    {
        UniqueLuceneIndexPopulatingUpdater updater = newUpdater();

        try
        {
            updater.remove( PrimitiveLongCollections.setOf( 1, 2, 3 ) );
            fail( "Exception expected" );
        }
        catch ( Exception e )
        {
            assertThat( e, instanceOf( UnsupportedOperationException.class ) );
        }
    }

    @Test
    public void closeVerifiesUniquenessOfAddedValues() throws Exception
    {
        SchemaIndex index = mock( SchemaIndex.class );
        UniqueLuceneIndexPopulatingUpdater updater = newUpdater( index );

        updater.process( add( 1, descriptor, "foo" ) );
        updater.process( add( 1, descriptor, "bar" ) );
        updater.process( add( 1, descriptor, "baz" ) );

        verifyZeroInteractions( index );

        updater.close();
        verify( index ).verifyUniqueness(
                any(), eq( descriptor.getPropertyIds() ), eq( Arrays.asList( "foo", "bar", "baz" ) ) );
    }

    @Test
    public void closeVerifiesUniquenessOfChangedValues() throws Exception
    {
        SchemaIndex index = mock( SchemaIndex.class );
        UniqueLuceneIndexPopulatingUpdater updater = newUpdater( index );

        updater.process( change( 1, descriptor, "foo1", "foo2" ) );
        updater.process( change( 1, descriptor, "bar1", "bar2" ) );
        updater.process( change( 1, descriptor, "baz1", "baz2" ) );

        verifyZeroInteractions( index );

        updater.close();
        verify( index ).verifyUniqueness(
                any(), eq( descriptor.getPropertyIds() ), eq( Arrays.asList( "foo2", "bar2", "baz2" ) ) );
    }

    @Test
    public void closeVerifiesUniquenessOfAddedAndChangedValues() throws Exception
    {
        SchemaIndex index = mock( SchemaIndex.class );
        UniqueLuceneIndexPopulatingUpdater updater = newUpdater( index );

        updater.process( add( 1, descriptor, "added1" ) );
        updater.process( add( 2, descriptor, "added2" ) );
        updater.process( change( 3, descriptor, "before1", "after1" ) );
        updater.process( change( 4, descriptor, "before2", "after2" ) );
        updater.process( remove( 5, descriptor, "removed1" ) );

        verifyZeroInteractions( index );

        updater.close();

        List<Object> toBeVerified = Arrays.asList( "added1", "added2", "after1", "after2" );
        verify( index ).verifyUniqueness(
                any(), eq( descriptor.getPropertyIds() ), eq( toBeVerified ) );
    }

    @Test
    public void addedNodePropertiesIncludedInSample() throws Exception
    {
        UniqueIndexSampler sampler = new UniqueIndexSampler();
        UniqueLuceneIndexPopulatingUpdater updater = newUpdater( sampler );

        updater.process( add( 1, descriptor, "foo" ) );
        updater.process( add( 2, descriptor, "bar" ) );
        updater.process( add( 3, descriptor, "baz" ) );
        updater.process( add( 4, descriptor, "qux" ) );

        verifySamplingResult( sampler, 4 );
    }

    @Test
    public void changedNodePropertiesDoNotInfluenceSample() throws Exception
    {
        UniqueIndexSampler sampler = new UniqueIndexSampler();
        UniqueLuceneIndexPopulatingUpdater updater = newUpdater( sampler );

        updater.process( change( 1, descriptor, "before1", "after1" ) );
        updater.process( change( 2, descriptor, "before2", "after2" ) );

        verifySamplingResult( sampler, 0 );
    }

    @Test
    public void removedNodePropertyIncludedInSample() throws Exception
    {
        long initialValue = 10;
        UniqueIndexSampler sampler = new UniqueIndexSampler();
        sampler.increment( initialValue );

        UniqueLuceneIndexPopulatingUpdater updater = newUpdater( sampler );

        updater.process( remove( 1, descriptor, "removed1" ) );
        updater.process( remove( 2, descriptor, "removed2" ) );

        verifySamplingResult( sampler, initialValue - 2 );
    }

    @Test
    public void nodePropertyUpdatesIncludedInSample() throws Exception
    {
        UniqueIndexSampler sampler = new UniqueIndexSampler();
        UniqueLuceneIndexPopulatingUpdater updater = newUpdater( sampler );

        updater.process( add( 1, descriptor, "foo" ) );
        updater.process( change( 1, descriptor, "foo", "bar" ) );
        updater.process( add( 2, descriptor, "baz" ) );
        updater.process( add( 3, descriptor, "qux" ) );
        updater.process( remove( 4, descriptor, "qux" ) );

        verifySamplingResult( sampler, 2 );
    }

    @Test
    public void additionsDeliveredToIndexWriter() throws Exception
    {
        LuceneIndexWriter writer = mock( LuceneIndexWriter.class );
        UniqueLuceneIndexPopulatingUpdater updater = newUpdater( writer );

        updater.process( add( 1, descriptor, "foo" ) );
        updater.process( add( 2, descriptor, "bar" ) );
        updater.process( add( 3, descriptor, "qux" ) );

        verify( writer ).updateDocument( newTermForChangeOrRemove( 1 ),
                LuceneDocumentStructure.documentRepresentingProperties( (long) 1, "foo" ) );
        verify( writer ).updateDocument( newTermForChangeOrRemove( 2 ),
                LuceneDocumentStructure.documentRepresentingProperties( (long) 2, "bar" ) );
        verify( writer ).updateDocument( newTermForChangeOrRemove( 3 ),
                LuceneDocumentStructure.documentRepresentingProperties( (long) 3, "qux" ) );
    }

    @Test
    public void changesDeliveredToIndexWriter() throws Exception
    {
        LuceneIndexWriter writer = mock( LuceneIndexWriter.class );
        UniqueLuceneIndexPopulatingUpdater updater = newUpdater( writer );

        updater.process( change( 1, descriptor, "before1", "after1" ) );
        updater.process( change( 2, descriptor, "before2", "after2" ) );

        verify( writer ).updateDocument( newTermForChangeOrRemove( 1 ),
                LuceneDocumentStructure.documentRepresentingProperties( (long) 1, "after1" ) );
        verify( writer ).updateDocument( newTermForChangeOrRemove( 2 ),
                LuceneDocumentStructure.documentRepresentingProperties( (long) 2, "after2" ) );
    }

    @Test
    public void removalsDeliveredToIndexWriter() throws Exception
    {
        LuceneIndexWriter writer = mock( LuceneIndexWriter.class );
        UniqueLuceneIndexPopulatingUpdater updater = newUpdater( writer );

        updater.process( remove( 1, descriptor, "foo" ) );
        updater.process( remove( 2, descriptor, "bar" ) );
        updater.process( remove( 3, descriptor, "baz" ) );

        verify( writer ).deleteDocuments( newTermForChangeOrRemove( 1 ) );
        verify( writer ).deleteDocuments( newTermForChangeOrRemove( 2 ) );
        verify( writer ).deleteDocuments( newTermForChangeOrRemove( 3 ) );
    }

    private static void verifySamplingResult( UniqueIndexSampler sampler, long expectedValue )
    {
        IndexSample sample = sampler.result();

        assertEquals( expectedValue, sample.indexSize() );
        assertEquals( expectedValue, sample.uniqueValues() );
        assertEquals( expectedValue, sample.sampleSize() );
    }

    private static UniqueLuceneIndexPopulatingUpdater newUpdater()
    {
        return newUpdater( new UniqueIndexSampler() );
    }

    private static UniqueLuceneIndexPopulatingUpdater newUpdater( SchemaIndex index )
    {
        return newUpdater( index, mock( LuceneIndexWriter.class ), new UniqueIndexSampler() );
    }

    private static UniqueLuceneIndexPopulatingUpdater newUpdater( LuceneIndexWriter writer )
    {
        return newUpdater( mock( SchemaIndex.class ), writer, new UniqueIndexSampler() );
    }

    private static UniqueLuceneIndexPopulatingUpdater newUpdater( UniqueIndexSampler sampler )
    {
        return newUpdater( mock( SchemaIndex.class ), mock( LuceneIndexWriter.class ), sampler );
    }

    private static UniqueLuceneIndexPopulatingUpdater newUpdater( SchemaIndex index, LuceneIndexWriter writer,
            UniqueIndexSampler sampler )
    {
        return new UniqueLuceneIndexPopulatingUpdater( writer, descriptor.getPropertyIds(), index,
                mock( PropertyAccessor.class ), sampler );
    }
}
