/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.graphdb.mockfs;

import java.io.File;
import java.io.IOException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.neo4j.graphdb.mockfs.CloseTrackingFileSystem;
import org.neo4j.graphdb.mockfs.EphemeralFileSystemAbstraction;
import org.neo4j.io.ByteUnit;
import org.neo4j.io.fs.OpenMode;
import org.neo4j.io.fs.StoreChannel;

class EphemeralFileSystemAbstractionTest {
    private EphemeralFileSystemAbstraction fs;

    EphemeralFileSystemAbstractionTest() {
    }

    @BeforeEach
    void setUp() {
        this.fs = new EphemeralFileSystemAbstraction();
    }

    @AfterEach
    void tearDown() throws IOException {
        this.fs.close();
    }

    @Test
    void allowStoreThatExceedDefaultSize() throws IOException {
        File aFile = new File("test");
        StoreChannel channel = this.fs.open(aFile, OpenMode.READ_WRITE);
        ByteBuffer buffer = ByteBuffer.allocate(8);
        int mebiBytes = (int)ByteUnit.mebiBytes((long)1L);
        for (int position = mebiBytes + 42; position < 10000000; position += mebiBytes) {
            buffer.putLong(1L);
            buffer.flip();
            channel.writeAll(buffer, (long)position);
            buffer.clear();
        }
        channel.close();
    }

    @Test
    void growEphemeralFileBuffer() {
        EphemeralFileSystemAbstraction.DynamicByteBuffer byteBuffer = new EphemeralFileSystemAbstraction.DynamicByteBuffer();
        byte[] testBytes = new byte[]{1, 2, 3, 4};
        int length = testBytes.length;
        byteBuffer.put(0, testBytes, 0, length);
        Assertions.assertEquals((int)((int)ByteUnit.kibiBytes((long)1L)), (int)byteBuffer.buf().capacity());
        byteBuffer.put((int)(ByteUnit.kibiBytes((long)1L) + 2L), testBytes, 0, length);
        Assertions.assertEquals((int)((int)ByteUnit.kibiBytes((long)2L)), (int)byteBuffer.buf().capacity());
        byteBuffer.put((int)(ByteUnit.kibiBytes((long)5L) + 2L), testBytes, 0, length);
        Assertions.assertEquals((int)((int)ByteUnit.kibiBytes((long)8L)), (int)byteBuffer.buf().capacity());
        byteBuffer.put((int)(ByteUnit.mebiBytes((long)2L) + 2L), testBytes, 0, length);
        Assertions.assertEquals((int)((int)ByteUnit.mebiBytes((long)4L)), (int)byteBuffer.buf().capacity());
    }

    @Test
    void shouldNotLoseDataForcedBeforeFileSystemCrashes() throws Exception {
        try (EphemeralFileSystemAbstraction fs = new EphemeralFileSystemAbstraction();){
            int numberOfBytesForced = 8;
            File aFile = new File("yo");
            StoreChannel channel = fs.open(aFile, OpenMode.READ_WRITE);
            this.writeLong(channel, 1111L);
            channel.force(true);
            this.writeLong(channel, 2222L);
            fs.crash();
            StoreChannel readChannel = fs.open(aFile, OpenMode.READ);
            Assertions.assertEquals((long)numberOfBytesForced, (long)readChannel.size());
            Assertions.assertEquals((long)1111L, (long)this.readLong(readChannel).getLong());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void shouldBeConsistentAfterConcurrentWritesAndCrashes() throws Exception {
        ExecutorService executorService = Executors.newCachedThreadPool();
        try (EphemeralFileSystemAbstraction fs = new EphemeralFileSystemAbstraction();){
            File aFile = new File("contendedFile");
            for (int attempt = 0; attempt < 100; ++attempt) {
                ArrayList<Callable<Void>> workers = new ArrayList<Callable<Void>>();
                for (int i = 0; i < 100; ++i) {
                    workers.add(() -> {
                        try {
                            StoreChannel channel = fs.open(aFile, OpenMode.READ_WRITE);
                            channel.position(0L);
                            this.writeLong(channel, 1L);
                        }
                        catch (IOException e) {
                            throw new RuntimeException(e);
                        }
                        return null;
                    });
                    workers.add(() -> {
                        fs.crash();
                        return null;
                    });
                }
                List futures = executorService.invokeAll(workers);
                for (Future future : futures) {
                    future.get();
                }
                this.verifyFileIsEitherEmptyOrContainsLongIntegerValueOne(fs.open(aFile, OpenMode.READ_WRITE));
            }
        }
        finally {
            executorService.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void shouldBeConsistentAfterConcurrentWritesAndForces() throws Exception {
        ExecutorService executorService = Executors.newCachedThreadPool();
        try {
            for (int attempt = 0; attempt < 100; ++attempt) {
                try (EphemeralFileSystemAbstraction fs = new EphemeralFileSystemAbstraction();){
                    File aFile = new File("contendedFile");
                    ArrayList<Callable<Void>> workers = new ArrayList<Callable<Void>>();
                    for (int i = 0; i < 100; ++i) {
                        workers.add(() -> {
                            try {
                                StoreChannel channel = fs.open(aFile, OpenMode.READ_WRITE);
                                channel.position(channel.size());
                                this.writeLong(channel, 1L);
                            }
                            catch (IOException e) {
                                throw new RuntimeException(e);
                            }
                            return null;
                        });
                        workers.add(() -> {
                            StoreChannel channel = fs.open(aFile, OpenMode.READ_WRITE);
                            channel.force(true);
                            return null;
                        });
                    }
                    List futures = executorService.invokeAll(workers);
                    for (Future future : futures) {
                        future.get();
                    }
                    fs.crash();
                    this.verifyFileIsFullOfLongIntegerOnes(fs.open(aFile, OpenMode.READ_WRITE));
                    continue;
                }
            }
        }
        finally {
            executorService.shutdown();
        }
    }

    @Test
    void releaseResourcesOnClose() throws IOException {
        try (EphemeralFileSystemAbstraction fileSystemAbstraction = new EphemeralFileSystemAbstraction();){
            CloseTrackingFileSystem closeTrackingFileSystem = new CloseTrackingFileSystem();
            fileSystemAbstraction.getOrCreateThirdPartyFileSystem(CloseTrackingFileSystem.class, closeTrackingFileSystemClass -> closeTrackingFileSystem);
            File testDir = new File("testDir");
            File testFile = new File("testFile");
            fileSystemAbstraction.mkdir(testDir);
            fileSystemAbstraction.create(testFile);
            Assertions.assertTrue((boolean)fileSystemAbstraction.fileExists(testFile));
            Assertions.assertTrue((boolean)fileSystemAbstraction.fileExists(testFile));
            Assertions.assertFalse((boolean)closeTrackingFileSystem.isClosed());
            fileSystemAbstraction.close();
            Assertions.assertTrue((boolean)closeTrackingFileSystem.isClosed());
            Assertions.assertTrue((boolean)fileSystemAbstraction.isClosed());
            Assertions.assertFalse((boolean)fileSystemAbstraction.fileExists(testFile));
            Assertions.assertFalse((boolean)fileSystemAbstraction.fileExists(testFile));
        }
    }

    private void verifyFileIsFullOfLongIntegerOnes(StoreChannel channel) {
        try {
            long claimedSize = channel.size();
            ByteBuffer buffer = ByteBuffer.allocate((int)claimedSize);
            channel.readAll(buffer);
            buffer.flip();
            int position = 0;
            while ((long)position < claimedSize) {
                long value = buffer.getLong(position);
                Assertions.assertEquals((long)1L, (long)value);
                position += 8;
            }
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private void verifyFileIsEitherEmptyOrContainsLongIntegerValueOne(StoreChannel channel) {
        try {
            long claimedSize = channel.size();
            ByteBuffer buffer = ByteBuffer.allocate(8);
            channel.read(buffer, 0L);
            buffer.flip();
            if (claimedSize == 8L) {
                Assertions.assertEquals((long)1L, (long)buffer.getLong());
            } else {
                Assertions.assertThrows(BufferUnderflowException.class, buffer::getLong, (String)"Should have thrown an exception");
            }
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private ByteBuffer readLong(StoreChannel readChannel) throws IOException {
        ByteBuffer readBuffer = ByteBuffer.allocate(8);
        readChannel.readAll(readBuffer);
        readBuffer.flip();
        return readBuffer;
    }

    private void writeLong(StoreChannel channel, long value) throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate(8);
        buffer.putLong(value);
        buffer.flip();
        channel.write(buffer);
    }
}

