/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.api.impl.fulltext;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.lucene.document.Document;
import org.neo4j.concurrent.BinaryLatch;
import org.neo4j.function.ThrowingAction;
import org.neo4j.graphdb.Entity;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.ResourceIterable;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.event.PropertyEntry;
import org.neo4j.kernel.AvailabilityGuard;
import org.neo4j.kernel.api.impl.fulltext.AsyncFulltextIndexOperation;
import org.neo4j.kernel.api.impl.fulltext.LuceneFulltextDocumentStructure;
import org.neo4j.kernel.api.impl.fulltext.WritableFulltext;
import org.neo4j.kernel.api.impl.schema.writer.PartitionedIndexWriter;
import org.neo4j.kernel.lifecycle.LifecycleAdapter;
import org.neo4j.logging.Log;
import org.neo4j.scheduler.JobScheduler;

class FulltextUpdateApplier
extends LifecycleAdapter {
    private static final FulltextIndexUpdate STOP_SIGNAL = new FulltextIndexUpdate(null, null);
    private static final int POPULATING_BATCH_SIZE = 10000;
    private static final JobScheduler.Group UPDATE_APPLIER = new JobScheduler.Group("FulltextIndexUpdateApplier");
    private static final String APPLIER_THREAD_NAME = "Fulltext Index Add-On Applier Thread";
    private final LinkedBlockingQueue<FulltextIndexUpdate> workQueue;
    private final Log log;
    private final AvailabilityGuard availabilityGuard;
    private final JobScheduler scheduler;
    private JobScheduler.JobHandle workerThread;

    FulltextUpdateApplier(Log log, AvailabilityGuard availabilityGuard, JobScheduler scheduler) {
        this.log = log;
        this.availabilityGuard = availabilityGuard;
        this.scheduler = scheduler;
        this.workQueue = new LinkedBlockingQueue();
    }

    <E extends Entity> AsyncFulltextIndexOperation updatePropertyData(Map<Long, Map<String, Object>> state, WritableFulltext index) throws IOException {
        FulltextIndexUpdate update = new FulltextIndexUpdate(index, () -> {
            PartitionedIndexWriter indexWriter = index.getIndexWriter();
            for (Map.Entry stateEntry : state.entrySet()) {
                Predicate<Map.Entry> relevantForIndex;
                Set<String> indexedProperties = index.getProperties();
                if (Collections.disjoint(indexedProperties, ((Map)stateEntry.getValue()).keySet())) continue;
                long entityId = (Long)stateEntry.getKey();
                Stream entryStream = ((Map)stateEntry.getValue()).entrySet().stream();
                Map<String, Object> allProperties = entryStream.filter(relevantForIndex = entry -> indexedProperties.contains(entry.getKey())).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
                if (allProperties.isEmpty()) continue;
                FulltextUpdateApplier.updateDocument(indexWriter, entityId, allProperties);
            }
        });
        this.enqueueUpdate(update);
        return update;
    }

    private static void updateDocument(PartitionedIndexWriter indexWriter, long entityId, Map<String, Object> properties) throws IOException {
        Document document = LuceneFulltextDocumentStructure.documentRepresentingProperties(entityId, properties);
        indexWriter.updateDocument(LuceneFulltextDocumentStructure.newTermForChangeOrRemove(entityId), document);
    }

    <E extends Entity> AsyncFulltextIndexOperation removePropertyData(Iterable<PropertyEntry<E>> propertyEntries, Map<Long, Map<String, Object>> state, WritableFulltext index) throws IOException {
        FulltextIndexUpdate update = new FulltextIndexUpdate(index, () -> {
            for (PropertyEntry propertyEntry : propertyEntries) {
                long entityId;
                Map allProperties;
                if (!index.getProperties().contains(propertyEntry.key()) || (allProperties = (Map)state.get(entityId = ((Entity)propertyEntry.entity()).getId())) != null && !allProperties.isEmpty()) continue;
                index.getIndexWriter().deleteDocuments(LuceneFulltextDocumentStructure.newTermForChangeOrRemove(entityId));
            }
        });
        this.enqueueUpdate(update);
        return update;
    }

    AsyncFulltextIndexOperation writeBarrier() throws IOException {
        FulltextIndexUpdate barrier = new FulltextIndexUpdate(null, ThrowingAction.noop());
        this.enqueueUpdate(barrier);
        return barrier;
    }

    AsyncFulltextIndexOperation populateNodes(WritableFulltext index, GraphDatabaseService db) throws IOException {
        return this.enqueuePopulateIndex(index, db, () -> ((GraphDatabaseService)db).getAllNodes());
    }

    AsyncFulltextIndexOperation populateRelationships(WritableFulltext index, GraphDatabaseService db) throws IOException {
        return this.enqueuePopulateIndex(index, db, () -> ((GraphDatabaseService)db).getAllRelationships());
    }

    private AsyncFulltextIndexOperation enqueuePopulateIndex(WritableFulltext index, GraphDatabaseService db, Supplier<ResourceIterable<? extends Entity>> entitySupplier) throws IOException {
        FulltextIndexUpdate population = new FulltextIndexUpdate(index, () -> {
            try {
                PartitionedIndexWriter indexWriter = index.getIndexWriter();
                String[] indexedPropertyKeys = index.getProperties().toArray(new String[0]);
                ArrayList<Supplier<Document>> documents = new ArrayList<Supplier<Document>>();
                try (Transaction ignore = db.beginTx(1L, TimeUnit.DAYS);){
                    ResourceIterable entities = (ResourceIterable)entitySupplier.get();
                    for (Entity entity : entities) {
                        long entityId = entity.getId();
                        Map properties = entity.getProperties(indexedPropertyKeys);
                        if (!properties.isEmpty()) {
                            documents.add(this.documentBuilder(entityId, properties));
                        }
                        if (documents.size() <= 10000) continue;
                        indexWriter.addDocuments(documents.size(), this.reifyDocuments(documents));
                        documents.clear();
                    }
                }
                indexWriter.addDocuments(documents.size(), this.reifyDocuments(documents));
                index.setPopulated();
            }
            catch (Throwable th) {
                if (index != null) {
                    index.setFailed();
                }
                throw th;
            }
        });
        this.enqueueUpdate(population);
        return population;
    }

    private Supplier<Document> documentBuilder(long entityId, Map<String, Object> properties) {
        return () -> LuceneFulltextDocumentStructure.documentRepresentingProperties(entityId, properties);
    }

    private Iterable<Document> reifyDocuments(ArrayList<Supplier<Document>> documents) {
        return () -> documents.stream().map(Supplier::get).iterator();
    }

    private void enqueueUpdate(FulltextIndexUpdate update) throws IOException {
        try {
            this.workQueue.put(update);
        }
        catch (InterruptedException e) {
            throw new IOException("Fulltext index update failed.", e);
        }
    }

    public void start() {
        if (this.workerThread != null) {
            throw new IllegalStateException("Fulltext Index Add-On Applier Thread already started.");
        }
        this.workerThread = this.scheduler.schedule(UPDATE_APPLIER, (Runnable)new ApplierWorker(this.workQueue, this.log, this.availabilityGuard));
    }

    public void stop() {
        boolean enqueued;
        while (!(enqueued = this.workQueue.offer(STOP_SIGNAL))) {
        }
        try {
            this.workerThread.waitTermination();
            this.workerThread = null;
        }
        catch (InterruptedException e) {
            this.log.error("Interrupted before Fulltext Index Add-On Applier Thread could shut down.", (Throwable)e);
        }
        catch (ExecutionException e) {
            this.log.error("Exception while waiting for Fulltext Index Add-On Applier Thread to shut down.", (Throwable)e);
        }
    }

    private static class ApplierWorker
    implements Runnable {
        private LinkedBlockingQueue<FulltextIndexUpdate> workQueue;
        private final Log log;
        private final AvailabilityGuard availabilityGuard;

        ApplierWorker(LinkedBlockingQueue<FulltextIndexUpdate> workQueue, Log log, AvailabilityGuard availabilityGuard) {
            this.workQueue = workQueue;
            this.log = log;
            this.availabilityGuard = availabilityGuard;
        }

        @Override
        public void run() {
            FulltextIndexUpdate update;
            Thread.currentThread().setName(FulltextUpdateApplier.APPLIER_THREAD_NAME);
            this.waitForDatabaseToBeAvailable();
            HashSet<WritableFulltext> refreshableSet = new HashSet<WritableFulltext>();
            ArrayList<BinaryLatch> latches = new ArrayList<BinaryLatch>();
            while ((update = this.getNextUpdate()) != STOP_SIGNAL) {
                update = this.drainQueueAndApplyUpdates(update, refreshableSet, latches);
                this.refreshAndClearIndexes(refreshableSet);
                this.releaseAndClearLatches(latches);
                if (update != STOP_SIGNAL) continue;
                return;
            }
        }

        private void waitForDatabaseToBeAvailable() {
            boolean isAvailable;
            while (!(isAvailable = this.availabilityGuard.isAvailable(100L)) && !this.availabilityGuard.isShutdown()) {
            }
        }

        private FulltextIndexUpdate drainQueueAndApplyUpdates(FulltextIndexUpdate update, Set<WritableFulltext> refreshableSet, List<BinaryLatch> latches) {
            do {
                this.applyUpdate(update, refreshableSet, latches);
            } while ((update = this.workQueue.poll()) != null && update != STOP_SIGNAL);
            return update;
        }

        private void refreshAndClearIndexes(Set<WritableFulltext> refreshableSet) {
            for (WritableFulltext index : refreshableSet) {
                this.refreshIndex(index);
            }
            refreshableSet.clear();
        }

        private void releaseAndClearLatches(List<BinaryLatch> latches) {
            for (BinaryLatch latch : latches) {
                latch.release();
            }
            latches.clear();
        }

        private FulltextIndexUpdate getNextUpdate() {
            FulltextIndexUpdate update = null;
            do {
                try {
                    update = this.workQueue.take();
                }
                catch (InterruptedException e) {
                    this.log.debug("Fulltext Index Add-On Applier Thread decided to ignore an interrupt.", (Throwable)e);
                }
            } while (update == null);
            return update;
        }

        private void applyUpdate(FulltextIndexUpdate update, Set<WritableFulltext> refreshableSet, List<BinaryLatch> latches) {
            latches.add(update);
            update.applyUpdate();
            refreshableSet.add(update.index);
        }

        private void refreshIndex(WritableFulltext index) {
            try {
                if (index != null) {
                    index.maybeRefreshBlocking();
                }
            }
            catch (Throwable e) {
                this.log.error("Failed to refresh fulltext after updates.", e);
            }
        }
    }

    private static class FulltextIndexUpdate
    extends BinaryLatch
    implements AsyncFulltextIndexOperation {
        private final WritableFulltext index;
        private final ThrowingAction<IOException> action;
        private volatile Throwable throwable;

        private FulltextIndexUpdate(WritableFulltext index, ThrowingAction<IOException> action) {
            this.index = index;
            this.action = action;
        }

        @Override
        public void awaitCompletion() throws ExecutionException {
            super.await();
            Throwable th = this.throwable;
            if (th != null) {
                throw new ExecutionException(th);
            }
        }

        void applyUpdate() {
            try {
                this.action.apply();
            }
            catch (Throwable e) {
                this.throwable = e;
            }
        }
    }
}

