/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.commandline.dbms;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.file.AccessDeniedException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileSystemException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import org.apache.commons.lang3.SystemUtils;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;
import org.neo4j.commandline.admin.AdminCommand;
import org.neo4j.commandline.admin.CommandFailed;
import org.neo4j.commandline.admin.CommandLocator;
import org.neo4j.commandline.admin.IncorrectUsage;
import org.neo4j.commandline.admin.Usage;
import org.neo4j.commandline.dbms.LoadCommand;
import org.neo4j.commandline.dbms.LoadCommandProvider;
import org.neo4j.dbms.archive.IncorrectFormat;
import org.neo4j.dbms.archive.Loader;
import org.neo4j.graphdb.config.Setting;
import org.neo4j.graphdb.factory.GraphDatabaseSettings;
import org.neo4j.helpers.ArrayUtil;
import org.neo4j.io.fs.DefaultFileSystemAbstraction;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.kernel.internal.locker.StoreLocker;
import org.neo4j.test.rule.TestDirectory;

public class LoadCommandTest {
    @Rule
    public TestDirectory testDirectory = TestDirectory.testDirectory();
    private Path homeDir;
    private Path configDir;
    private Path archive;
    private Loader loader;

    @Before
    public void setUp() {
        this.homeDir = this.testDirectory.directory("home-dir").toPath();
        this.configDir = this.testDirectory.directory("config-dir").toPath();
        this.archive = this.testDirectory.directory("some-archive.dump").toPath();
        this.loader = (Loader)Mockito.mock(Loader.class);
    }

    @Test
    public void shouldLoadTheDatabaseFromTheArchive() throws CommandFailed, IncorrectUsage, IOException, IncorrectFormat {
        this.execute("foo.db", new String[0]);
        ((Loader)Mockito.verify((Object)this.loader)).load(this.archive, this.homeDir.resolve("data/databases/foo.db"), this.homeDir.resolve("data/databases/foo.db"));
    }

    @Test
    public void shouldCalculateTheDatabaseDirectoryFromConfig() throws IOException, CommandFailed, IncorrectUsage, IncorrectFormat {
        Path dataDir = this.testDirectory.directory("some-other-path").toPath();
        Path databaseDir = dataDir.resolve("databases/foo.db");
        Files.createDirectories(databaseDir, new FileAttribute[0]);
        Files.write(this.configDir.resolve("neo4j.conf"), Arrays.asList(LoadCommandTest.formatProperty(GraphDatabaseSettings.data_directory, dataDir)), new OpenOption[0]);
        this.execute("foo.db", new String[0]);
        ((Loader)Mockito.verify((Object)this.loader)).load((Path)ArgumentMatchers.any(), (Path)ArgumentMatchers.eq((Object)databaseDir), (Path)ArgumentMatchers.eq((Object)databaseDir));
    }

    @Test
    public void shouldCalculateTheTxLogDirectoryFromConfig() throws Exception {
        Path dataDir = this.testDirectory.directory("some-other-path").toPath();
        Path txLogsDir = this.testDirectory.directory("txLogsPath").toPath();
        Path databaseDir = dataDir.resolve("databases/foo.db");
        Files.write(this.configDir.resolve("neo4j.conf"), Arrays.asList(LoadCommandTest.formatProperty(GraphDatabaseSettings.data_directory, dataDir), LoadCommandTest.formatProperty(GraphDatabaseSettings.logical_logs_location, txLogsDir)), new OpenOption[0]);
        this.execute("foo.db", new String[0]);
        ((Loader)Mockito.verify((Object)this.loader)).load((Path)ArgumentMatchers.any(), (Path)ArgumentMatchers.eq((Object)databaseDir), (Path)ArgumentMatchers.eq((Object)txLogsDir));
    }

    @Test
    public void shouldHandleSymlinkToDatabaseDir() throws IOException, CommandFailed, IncorrectUsage, IncorrectFormat {
        Assume.assumeFalse((String)"Can't reliably create symlinks on windows", (boolean)SystemUtils.IS_OS_WINDOWS);
        Path symDir = this.testDirectory.directory("path-to-links").toPath();
        Path realDatabaseDir = symDir.resolve("foo.db");
        Path dataDir = this.testDirectory.directory("some-other-path").toPath();
        Path databaseDir = dataDir.resolve("databases/foo.db");
        Files.createDirectories(realDatabaseDir, new FileAttribute[0]);
        Files.createDirectories(dataDir.resolve("databases"), new FileAttribute[0]);
        Files.createSymbolicLink(databaseDir, realDatabaseDir, new FileAttribute[0]);
        Files.write(this.configDir.resolve("neo4j.conf"), Arrays.asList(LoadCommandTest.formatProperty(GraphDatabaseSettings.data_directory, dataDir)), new OpenOption[0]);
        this.execute("foo.db", new String[0]);
        ((Loader)Mockito.verify((Object)this.loader)).load((Path)ArgumentMatchers.any(), (Path)ArgumentMatchers.eq((Object)realDatabaseDir), (Path)ArgumentMatchers.eq((Object)realDatabaseDir));
    }

    @Test
    public void shouldMakeFromCanonical() throws IOException, CommandFailed, IncorrectUsage, IncorrectFormat {
        Path dataDir = this.testDirectory.directory("some-other-path").toPath();
        Path databaseDir = dataDir.resolve("databases/foo.db");
        Files.createDirectories(databaseDir, new FileAttribute[0]);
        Files.write(this.configDir.resolve("neo4j.conf"), Arrays.asList(LoadCommandTest.formatProperty(GraphDatabaseSettings.data_directory, dataDir)), new OpenOption[0]);
        new LoadCommand(this.homeDir, this.configDir, this.loader).execute((String[])ArrayUtil.concat((Object[])new String[]{"--database=foo.db", "--from=foo.dump"}, (Object[])new String[0]));
        ((Loader)Mockito.verify((Object)this.loader)).load((Path)ArgumentMatchers.eq((Object)Paths.get(new File("foo.dump").getCanonicalPath(), new String[0])), (Path)ArgumentMatchers.any(), (Path)ArgumentMatchers.any());
    }

    @Test
    public void shouldDeleteTheOldDatabaseIfForceArgumentIsProvided() throws CommandFailed, IncorrectUsage, IOException, IncorrectFormat {
        Path databaseDirectory = this.homeDir.resolve("data/databases/foo.db");
        Files.createDirectories(databaseDirectory, new FileAttribute[0]);
        ((Loader)Mockito.doAnswer(ignored -> {
            Assert.assertThat((Object)Files.exists(databaseDirectory, new LinkOption[0]), (Matcher)Matchers.equalTo((Object)false));
            return null;
        }).when((Object)this.loader)).load((Path)ArgumentMatchers.any(), (Path)ArgumentMatchers.any(), (Path)ArgumentMatchers.any());
        this.execute("foo.db", "--force");
    }

    @Test
    public void shouldNotDeleteTheOldDatabaseIfForceArgumentIsNotProvided() throws CommandFailed, IncorrectUsage, IOException, IncorrectFormat {
        Path databaseDirectory = this.homeDir.resolve("data/databases/foo.db");
        Files.createDirectories(databaseDirectory, new FileAttribute[0]);
        ((Loader)Mockito.doAnswer(ignored -> {
            Assert.assertThat((Object)Files.exists(databaseDirectory, new LinkOption[0]), (Matcher)Matchers.equalTo((Object)true));
            return null;
        }).when((Object)this.loader)).load((Path)ArgumentMatchers.any(), (Path)ArgumentMatchers.any(), (Path)ArgumentMatchers.any());
        this.execute("foo.db", new String[0]);
    }

    @Test
    public void shouldRespectTheStoreLock() throws IOException, IncorrectUsage {
        Path databaseDirectory = this.homeDir.resolve("data/databases/foo.db");
        Files.createDirectories(databaseDirectory, new FileAttribute[0]);
        try (DefaultFileSystemAbstraction fileSystem = new DefaultFileSystemAbstraction();
             StoreLocker locker = new StoreLocker((FileSystemAbstraction)fileSystem, databaseDirectory.toFile());){
            locker.checkLock();
            this.execute("foo.db", "--force");
            Assert.fail((String)"expected exception");
        }
        catch (CommandFailed e) {
            Assert.assertThat((Object)e.getMessage(), (Matcher)Matchers.equalTo((Object)"the database is in use -- stop Neo4j and try again"));
        }
    }

    @Test
    public void shouldDefaultToGraphDb() throws Exception {
        Path databaseDir = this.homeDir.resolve("data/databases/graph.db");
        Files.createDirectories(databaseDir, new FileAttribute[0]);
        new LoadCommand(this.homeDir, this.configDir, this.loader).execute(new String[]{"--from=something"});
        ((Loader)Mockito.verify((Object)this.loader)).load((Path)ArgumentMatchers.any(), (Path)ArgumentMatchers.eq((Object)databaseDir), (Path)ArgumentMatchers.eq((Object)databaseDir));
    }

    @Test
    public void shouldObjectIfTheArchiveArgumentIsMissing() throws Exception {
        try {
            new LoadCommand(this.homeDir, this.configDir, this.loader).execute(new String[]{"--database=something"});
            Assert.fail((String)"expected exception");
        }
        catch (IllegalArgumentException e) {
            Assert.assertThat((Object)e.getMessage(), (Matcher)Matchers.equalTo((Object)"Missing argument 'from'"));
        }
    }

    @Test
    public void shouldGiveAClearMessageIfTheArchiveDoesntExist() throws IOException, IncorrectFormat, IncorrectUsage {
        ((Loader)Mockito.doThrow((Throwable[])new Throwable[]{new NoSuchFileException(this.archive.toString())}).when((Object)this.loader)).load((Path)ArgumentMatchers.any(), (Path)ArgumentMatchers.any(), (Path)ArgumentMatchers.any());
        try {
            this.execute(null, new String[0]);
            Assert.fail((String)"expected exception");
        }
        catch (CommandFailed e) {
            Assert.assertThat((Object)e.getMessage(), (Matcher)Matchers.equalTo((Object)("archive does not exist: " + this.archive)));
        }
    }

    @Test
    public void shouldGiveAClearMessageIfTheDatabaseAlreadyExists() throws IOException, IncorrectFormat, IncorrectUsage {
        ((Loader)Mockito.doThrow(FileAlreadyExistsException.class).when((Object)this.loader)).load((Path)ArgumentMatchers.any(), (Path)ArgumentMatchers.any(), (Path)ArgumentMatchers.any());
        try {
            this.execute("foo.db", new String[0]);
            Assert.fail((String)"expected exception");
        }
        catch (CommandFailed e) {
            Assert.assertThat((Object)e.getMessage(), (Matcher)Matchers.equalTo((Object)"database already exists: foo.db"));
        }
    }

    @Test
    public void shouldGiveAClearMessageIfTheDatabasesDirectoryIsNotWritable() throws IOException, IncorrectFormat, IncorrectUsage {
        ((Loader)Mockito.doThrow(AccessDeniedException.class).when((Object)this.loader)).load((Path)ArgumentMatchers.any(), (Path)ArgumentMatchers.any(), (Path)ArgumentMatchers.any());
        try {
            this.execute(null, new String[0]);
            Assert.fail((String)"expected exception");
        }
        catch (CommandFailed e) {
            Assert.assertThat((Object)e.getMessage(), (Matcher)Matchers.equalTo((Object)"you do not have permission to load a database -- is Neo4j running as a different user?"));
        }
    }

    @Test
    public void shouldWrapIOExceptionsCarefullyBecauseCriticalInformationIsOftenEncodedInTheirNameButMissingFromTheirMessage() throws IOException, IncorrectUsage, IncorrectFormat {
        ((Loader)Mockito.doThrow((Throwable[])new Throwable[]{new FileSystemException("the-message")}).when((Object)this.loader)).load((Path)ArgumentMatchers.any(), (Path)ArgumentMatchers.any(), (Path)ArgumentMatchers.any());
        try {
            this.execute(null, new String[0]);
            Assert.fail((String)"expected exception");
        }
        catch (CommandFailed e) {
            Assert.assertThat((Object)e.getMessage(), (Matcher)Matchers.equalTo((Object)"unable to load database: FileSystemException: the-message"));
        }
    }

    @Test
    public void shouldThrowIfTheArchiveFormatIsInvalid() throws IOException, IncorrectUsage, IncorrectFormat {
        ((Loader)Mockito.doThrow(IncorrectFormat.class).when((Object)this.loader)).load((Path)ArgumentMatchers.any(), (Path)ArgumentMatchers.any(), (Path)ArgumentMatchers.any());
        try {
            this.execute(null, new String[0]);
            Assert.fail((String)"expected exception");
        }
        catch (CommandFailed e) {
            Assert.assertThat((Object)e.getMessage(), (Matcher)Matchers.containsString((String)this.archive.toString()));
            Assert.assertThat((Object)e.getMessage(), (Matcher)Matchers.containsString((String)"valid Neo4j archive"));
        }
    }

    @Test
    public void shouldPrintNiceHelp() throws Throwable {
        try (ByteArrayOutputStream baos = new ByteArrayOutputStream();){
            PrintStream ps = new PrintStream(baos);
            Usage usage = new Usage("neo4j-admin", (CommandLocator)Mockito.mock(CommandLocator.class));
            usage.printUsageForCommand((AdminCommand.Provider)new LoadCommandProvider(), ps::println);
            Assert.assertEquals((Object)String.format("usage: neo4j-admin load --from=<archive-path> [--database=<name>]%n                        [--force[=<true|false>]]%n%nenvironment variables:%n    NEO4J_CONF    Path to directory which contains neo4j.conf.%n    NEO4J_DEBUG   Set to anything to enable debug output.%n    NEO4J_HOME    Neo4j home directory.%n    HEAP_SIZE     Set JVM maximum heap size during command execution.%n                  Takes a number and a unit, for example 512m.%n%nLoad a database from an archive. <archive-path> must be an archive created with%nthe dump command. <database> is the name of the database to create. Existing%ndatabases can be replaced by specifying --force. It is not possible to replace a%ndatabase that is mounted in a running Neo4j server.%n%noptions:%n  --from=<archive-path>   Path to archive created with the dump command.%n  --database=<name>       Name of database. [default:graph.db]%n  --force=<true|false>    If an existing database should be replaced.%n                          [default:false]%n", new Object[0]), (Object)baos.toString());
        }
    }

    private void execute(String database, String ... otherArgs) throws IncorrectUsage, CommandFailed {
        new LoadCommand(this.homeDir, this.configDir, this.loader).execute((String[])ArrayUtil.concat((Object[])new String[]{"--database=" + database, "--from=" + this.archive}, (Object[])otherArgs));
    }

    private static String formatProperty(Setting setting, Path path) {
        return String.format("%s=%s", setting.name(), path.toString().replace('\\', '/'));
    }
}

