/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.commandline.dbms;

import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.file.AccessDeniedException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.Collections;
import java.util.function.Predicate;
import org.apache.commons.lang3.SystemUtils;
import org.hamcrest.Matcher;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.mockito.Matchers;
import org.mockito.Mockito;
import org.neo4j.commandline.admin.AdminCommand;
import org.neo4j.commandline.admin.CommandFailed;
import org.neo4j.commandline.admin.CommandLocator;
import org.neo4j.commandline.admin.IncorrectUsage;
import org.neo4j.commandline.admin.Usage;
import org.neo4j.commandline.dbms.DumpCommand;
import org.neo4j.commandline.dbms.DumpCommandProvider;
import org.neo4j.dbms.DatabaseManagementSystemSettings;
import org.neo4j.dbms.archive.Dumper;
import org.neo4j.dbms.archive.TestUtils;
import org.neo4j.io.fs.DefaultFileSystemAbstraction;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.kernel.impl.storemigration.StoreFileType;
import org.neo4j.kernel.internal.StoreLocker;
import org.neo4j.test.rule.TestDirectory;

public class DumpCommandTest {
    @Rule
    public TestDirectory testDirectory = TestDirectory.testDirectory();
    @Rule
    public ExpectedException expected = ExpectedException.none();
    private Path homeDir;
    private Path configDir;
    private Path archive;
    private Dumper dumper;
    private Path databaseDirectory;

    @Before
    public void setUp() throws Exception {
        this.homeDir = this.testDirectory.directory("home-dir").toPath();
        this.configDir = this.testDirectory.directory("config-dir").toPath();
        this.archive = this.testDirectory.file("some-archive.dump").toPath();
        this.dumper = (Dumper)Mockito.mock(Dumper.class);
        this.putStoreInDirectory(this.homeDir.resolve("data/databases/foo.db"));
        this.databaseDirectory = this.homeDir.resolve("data/databases/foo.db");
    }

    @Test
    public void shouldDumpTheDatabaseToTheArchive() throws Exception {
        this.execute("foo.db");
        ((Dumper)Mockito.verify((Object)this.dumper)).dump((Path)Matchers.eq((Object)this.homeDir.resolve("data/databases/foo.db")), (Path)Matchers.eq((Object)this.archive), (Predicate)Matchers.any());
    }

    @Test
    public void shouldCalculateTheDatabaseDirectoryFromConfig() throws Exception {
        Path dataDir = this.testDirectory.directory("some-other-path").toPath();
        Path databaseDir = dataDir.resolve("databases/foo.db");
        this.putStoreInDirectory(databaseDir);
        Files.write(this.configDir.resolve("neo4j.conf"), Arrays.asList(String.format("%s=%s", DatabaseManagementSystemSettings.data_directory.name(), dataDir.toString().replace('\\', '/'))), new OpenOption[0]);
        this.execute("foo.db");
        ((Dumper)Mockito.verify((Object)this.dumper)).dump((Path)Matchers.eq((Object)databaseDir), (Path)Matchers.any(), (Predicate)Matchers.any());
    }

    @Test
    public void shouldHandleDatabaseSymlink() throws Exception {
        Assume.assumeFalse((String)"Can't reliably create symlinks on windows", (boolean)SystemUtils.IS_OS_WINDOWS);
        Path symDir = this.testDirectory.directory("path-to-links").toPath();
        Path realDatabaseDir = symDir.resolve("foo.db");
        Path dataDir = this.testDirectory.directory("some-other-path").toPath();
        Path databaseDir = dataDir.resolve("databases/foo.db");
        this.putStoreInDirectory(realDatabaseDir);
        Files.createDirectories(dataDir.resolve("databases"), new FileAttribute[0]);
        Files.createSymbolicLink(databaseDir, realDatabaseDir, new FileAttribute[0]);
        Files.write(this.configDir.resolve("neo4j.conf"), Arrays.asList(String.format("%s=%s", DatabaseManagementSystemSettings.data_directory.name(), dataDir.toString().replace('\\', '/'))), new OpenOption[0]);
        this.execute("foo.db");
        ((Dumper)Mockito.verify((Object)this.dumper)).dump((Path)Matchers.eq((Object)realDatabaseDir), (Path)Matchers.any(), (Predicate)Matchers.any());
    }

    @Test
    public void shouldCalculateTheArchiveNameIfPassedAnExistingDirectory() throws Exception {
        File to = this.testDirectory.directory("some-dir");
        new DumpCommand(this.homeDir, this.configDir, this.dumper).execute(new String[]{"--database=foo.db", "--to=" + to});
        ((Dumper)Mockito.verify((Object)this.dumper)).dump((Path)Matchers.any(Path.class), (Path)Matchers.eq((Object)to.toPath().resolve("foo.db.dump")), (Predicate)Matchers.any());
    }

    @Test
    public void shouldConvertToCanonicalPath() throws Exception {
        new DumpCommand(this.homeDir, this.configDir, this.dumper).execute(new String[]{"--database=foo.db", "--to=foo.dump"});
        ((Dumper)Mockito.verify((Object)this.dumper)).dump((Path)Matchers.any(Path.class), (Path)Matchers.eq((Object)Paths.get(new File("foo.dump").getCanonicalPath(), new String[0])), (Predicate)Matchers.any());
    }

    @Test
    public void shouldNotCalculateTheArchiveNameIfPassedAnExistingFile() throws Exception {
        Files.createFile(this.archive, new FileAttribute[0]);
        this.execute("foo.db");
        ((Dumper)Mockito.verify((Object)this.dumper)).dump((Path)Matchers.any(), (Path)Matchers.eq((Object)this.archive), (Predicate)Matchers.any());
    }

    @Test
    public void shouldRespectTheStoreLock() throws Exception {
        Path databaseDirectory = this.homeDir.resolve("data/databases/foo.db");
        try (DefaultFileSystemAbstraction fileSystem = new DefaultFileSystemAbstraction();
             StoreLocker storeLocker = new StoreLocker((FileSystemAbstraction)fileSystem);){
            storeLocker.checkLock(databaseDirectory.toFile());
            this.execute("foo.db");
            Assert.fail((String)"expected exception");
        }
        catch (CommandFailed e) {
            Assert.assertThat((Object)e.getMessage(), (Matcher)org.hamcrest.Matchers.equalTo((Object)"the database is in use -- stop Neo4j and try again"));
        }
    }

    @Test
    public void shouldReleaseTheStoreLockAfterDumping() throws Exception {
        this.execute("foo.db");
        this.assertCanLockStore(this.databaseDirectory);
    }

    @Test
    public void shouldReleaseTheStoreLockEvenIfThereIsAnError() throws Exception {
        ((Dumper)Mockito.doThrow(IOException.class).when((Object)this.dumper)).dump((Path)Matchers.any(), (Path)Matchers.any(), (Predicate)Matchers.any());
        try {
            this.execute("foo.db");
        }
        catch (CommandFailed commandFailed) {
            // empty catch block
        }
        this.assertCanLockStore(this.databaseDirectory);
    }

    @Test
    public void shouldNotAccidentallyCreateTheDatabaseDirectoryAsASideEffectOfStoreLocking() throws Exception {
        Path databaseDirectory = this.homeDir.resolve("data/databases/accident.db");
        ((Dumper)Mockito.doAnswer(ignored -> {
            Assert.assertThat((Object)Files.exists(databaseDirectory, new LinkOption[0]), (Matcher)org.hamcrest.Matchers.equalTo((Object)false));
            return null;
        }).when((Object)this.dumper)).dump((Path)Matchers.any(), (Path)Matchers.any(), (Predicate)Matchers.any());
        this.execute("foo.db");
    }

    @Test
    public void shouldReportAHelpfulErrorIfWeDontHaveWritePermissionsForLock() throws Exception {
        Assume.assumeFalse((String)"We haven't found a way to reliably tests permissions on Windows", (boolean)SystemUtils.IS_OS_WINDOWS);
        try (DefaultFileSystemAbstraction fileSystem = new DefaultFileSystemAbstraction();
             StoreLocker storeLocker = new StoreLocker((FileSystemAbstraction)fileSystem);){
            storeLocker.checkLock(this.databaseDirectory.toFile());
            try (Closeable ignored = TestUtils.withPermissions(this.databaseDirectory.resolve("store_lock"), Collections.emptySet());){
                this.execute("foo.db");
                Assert.fail((String)"expected exception");
            }
            catch (CommandFailed e) {
                Assert.assertThat((Object)e.getMessage(), (Matcher)org.hamcrest.Matchers.equalTo((Object)"you do not have permission to dump the database -- is Neo4j running as a different user?"));
            }
        }
    }

    @Test
    public void shouldExcludeTheStoreLockFromTheArchiveToAvoidProblemsWithReadingLockedFilesOnWindows() throws Exception {
        ((Dumper)Mockito.doAnswer(invocation -> {
            Predicate exclude = (Predicate)invocation.getArgumentAt(2, Predicate.class);
            Assert.assertThat((Object)exclude.test(Paths.get("store_lock", new String[0])), (Matcher)org.hamcrest.Matchers.is((Object)true));
            Assert.assertThat((Object)exclude.test(Paths.get("some-other-file", new String[0])), (Matcher)org.hamcrest.Matchers.is((Object)false));
            return null;
        }).when((Object)this.dumper)).dump((Path)Matchers.any(), (Path)Matchers.any(), (Predicate)Matchers.any());
        this.execute("foo.db");
    }

    @Test
    public void shouldDefaultToGraphDB() throws Exception {
        Path dataDir = this.testDirectory.directory("some-other-path").toPath();
        Path databaseDir = dataDir.resolve("databases/graph.db");
        this.putStoreInDirectory(databaseDir);
        Files.write(this.configDir.resolve("neo4j.conf"), Arrays.asList(String.format("%s=%s", DatabaseManagementSystemSettings.data_directory.name(), dataDir.toString().replace('\\', '/'))), new OpenOption[0]);
        new DumpCommand(this.homeDir, this.configDir, this.dumper).execute(new String[]{"--to=" + this.archive});
        ((Dumper)Mockito.verify((Object)this.dumper)).dump((Path)Matchers.eq((Object)databaseDir), (Path)Matchers.any(), (Predicate)Matchers.any());
    }

    @Test
    public void shouldObjectIfTheArchiveArgumentIsMissing() throws Exception {
        try {
            new DumpCommand(this.homeDir, this.configDir, null).execute(new String[]{"--database=something"});
            Assert.fail((String)"expected exception");
        }
        catch (IllegalArgumentException e) {
            Assert.assertThat((Object)e.getMessage(), (Matcher)org.hamcrest.Matchers.equalTo((Object)"Missing argument 'to'"));
        }
    }

    @Test
    public void shouldGiveAClearErrorIfTheArchiveAlreadyExists() throws Exception {
        ((Dumper)Mockito.doThrow((Throwable)new FileAlreadyExistsException("the-archive-path")).when((Object)this.dumper)).dump((Path)Matchers.any(), (Path)Matchers.any(), (Predicate)Matchers.any());
        try {
            this.execute("foo.db");
            Assert.fail((String)"expected exception");
        }
        catch (CommandFailed e) {
            Assert.assertThat((Object)e.getMessage(), (Matcher)org.hamcrest.Matchers.equalTo((Object)"archive already exists: the-archive-path"));
        }
    }

    @Test
    public void shouldGiveAClearMessageIfTheDatabaseDoesntExist() throws Exception {
        try {
            this.execute("bobo.db");
            Assert.fail((String)"expected exception");
        }
        catch (CommandFailed e) {
            Assert.assertThat((Object)e.getMessage(), (Matcher)org.hamcrest.Matchers.equalTo((Object)"database does not exist: bobo.db"));
        }
    }

    @Test
    public void shouldGiveAClearMessageIfTheArchivesParentDoesntExist() throws Exception {
        ((Dumper)Mockito.doThrow((Throwable)new NoSuchFileException(this.archive.getParent().toString())).when((Object)this.dumper)).dump((Path)Matchers.any(), (Path)Matchers.any(), (Predicate)Matchers.any());
        try {
            this.execute("foo.db");
            Assert.fail((String)"expected exception");
        }
        catch (CommandFailed e) {
            Assert.assertThat((Object)e.getMessage(), (Matcher)org.hamcrest.Matchers.equalTo((Object)("unable to dump database: NoSuchFileException: " + this.archive.getParent())));
        }
    }

    @Test
    public void shouldWrapIOExceptionsCarefulllyBecauseCriticalInformationIsOftenEncodedInTheirNameButMissingFromTheirMessage() throws Exception {
        ((Dumper)Mockito.doThrow((Throwable)new IOException("the-message")).when((Object)this.dumper)).dump((Path)Matchers.any(), (Path)Matchers.any(), (Predicate)Matchers.any());
        try {
            this.execute("foo.db");
            Assert.fail((String)"expected exception");
        }
        catch (CommandFailed e) {
            Assert.assertThat((Object)e.getMessage(), (Matcher)org.hamcrest.Matchers.equalTo((Object)"unable to dump database: IOException: the-message"));
        }
    }

    @Test
    public void shouldPrintNiceHelp() throws Exception {
        try (ByteArrayOutputStream baos = new ByteArrayOutputStream();){
            PrintStream ps = new PrintStream(baos);
            Usage usage = new Usage("neo4j-admin", (CommandLocator)Mockito.mock(CommandLocator.class));
            usage.printUsageForCommand((AdminCommand.Provider)new DumpCommandProvider(), ps::println);
            Assert.assertEquals((Object)String.format("usage: neo4j-admin dump [--database=<name>] --to=<destination-path>%n%nDump a database into a single-file archive. The archive can be used by the load%ncommand. <destination-path> can be a file or directory (in which case a file%ncalled <database>.dump will be created). It is not possible to dump a database%nthat is mounted in a running Neo4j server.%n%noptions:%n  --database=<name>         Name of database. [default:graph.db]%n  --to=<destination-path>   Destination (file or folder) of database dump.%n", new Object[0]), (Object)baos.toString());
        }
    }

    private void execute(String database) throws IncorrectUsage, CommandFailed, AccessDeniedException {
        new DumpCommand(this.homeDir, this.configDir, this.dumper).execute(new String[]{"--database=" + database, "--to=" + this.archive});
    }

    private void assertCanLockStore(Path databaseDirectory) throws IOException {
        try (DefaultFileSystemAbstraction fileSystem = new DefaultFileSystemAbstraction();
             StoreLocker storeLocker = new StoreLocker((FileSystemAbstraction)fileSystem);){
            storeLocker.checkLock(databaseDirectory.toFile());
        }
    }

    private void putStoreInDirectory(Path storeDir) throws IOException {
        Files.createDirectories(storeDir, new FileAttribute[0]);
        Path storeFile = storeDir.resolve(StoreFileType.STORE.augment("neostore"));
        Files.createFile(storeFile, new FileAttribute[0]);
    }
}

