/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.csv.reader;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.StringReader;
import java.io.Writer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collection;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.junit.Assert;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.neo4j.csv.reader.CharReadable;
import org.neo4j.csv.reader.Magic;
import org.neo4j.csv.reader.Readables;
import org.neo4j.csv.reader.SectionedCharBuffer;
import org.neo4j.test.rule.TestDirectory;

@RunWith(value=Parameterized.class)
public class ReadablesTest {
    @Rule
    public final TestDirectory directory = TestDirectory.testDirectory();
    @Parameterized.Parameter
    public ReadMethod readMethod;

    @Parameterized.Parameters
    public static Collection<ReadMethod> readMethods() {
        return Arrays.asList((readable, length) -> {
            SectionedCharBuffer readText = new SectionedCharBuffer(length);
            readable.read(readText, readText.front());
            return Arrays.copyOfRange(readText.array(), readText.pivot(), readText.front());
        }, (readable, length) -> {
            char[] result = new char[length];
            readable.read(result, 0, length);
            return result;
        });
    }

    @Test
    public void shouldReadTextCompressedInZipArchiveWithSingleFileIn() throws Exception {
        String text = "abcdefghijlkmnopqrstuvxyz";
        File compressed = this.compressWithZip(text, new String[0]);
        this.assertReadText(compressed, text);
    }

    @Test
    public void shouldReadTextCompressedInGZipFile() throws Exception {
        String text = "abcdefghijlkmnopqrstuvxyz";
        File compressed = this.compressWithGZip(text);
        this.assertReadText(compressed, text);
    }

    @Test
    public void shouldReadPlainTextFile() throws Exception {
        String text = "abcdefghijlkmnopqrstuvxyz";
        File plainText = this.write(text);
        this.assertReadText(plainText, text);
    }

    @Test
    public void shouldReadTheOnlyRealFileInThere() throws Exception {
        String text = "abcdefghijlkmnopqrstuvxyz";
        File compressed = this.compressWithZip(text, ".nothing", ".DS_Store", "__MACOSX/", "__MACOSX/file");
        this.assertReadText(compressed, text);
    }

    @Test
    public void shouldFailWhenThereAreMoreThanOneSuitableFileInThere() throws Exception {
        String text = "abcdefghijlkmnopqrstuvxyz";
        File compressed = this.compressWithZip(text, ".nothing", ".DS_Store", "somewhere/something");
        try {
            CharReadable readable = Readables.files((Charset)Charset.defaultCharset(), (File[])new File[]{compressed});
            Assert.fail((String)"Should fail since there are multiple suitable files in the zip archive");
        }
        catch (IOException e) {
            Assert.assertThat((Object)e.getMessage(), (Matcher)CoreMatchers.containsString((String)"Multiple"));
        }
    }

    @Test
    public void shouldTrackPosition() throws Exception {
        String data = "1234567890";
        CharReadable reader = Readables.wrap((Reader)new StringReader(data));
        SectionedCharBuffer buffer = new SectionedCharBuffer(4);
        int expected = 0;
        do {
            buffer = reader.read(buffer, buffer.front());
            Assert.assertEquals((long)(expected += buffer.available()), (long)reader.position());
        } while (buffer.hasAvailable());
        Assert.assertEquals((long)data.toCharArray().length, (long)expected);
    }

    @Test
    public void shouldComplyWithUtf8CharsetForExample() throws Exception {
        this.shouldComplyWithSpecifiedCharset(StandardCharsets.UTF_8);
    }

    @Test
    public void shouldComplyWithIso88591CharsetForExample() throws Exception {
        this.shouldComplyWithSpecifiedCharset(StandardCharsets.ISO_8859_1);
    }

    @Test
    public void shouldSkipBOM() throws Exception {
        String text = "abcdefghijklmnop";
        this.shouldReadTextFromFileWithBom(Magic.BOM_UTF_32_BE, text);
        this.shouldReadTextFromFileWithBom(Magic.BOM_UTF_32_LE, text);
        this.shouldReadTextFromFileWithBom(Magic.BOM_UTF_16_BE, text);
        this.shouldReadTextFromFileWithBom(Magic.BOM_UTF_16_LE, text);
        this.shouldReadTextFromFileWithBom(Magic.BOM_UTF_8, text);
    }

    @Test
    public void shouldReadTextFromWrappedInputStream() throws Exception {
        String text = "abcdefghijklmnop";
        File file = this.writeToFile(text, Charset.defaultCharset());
        this.assertReadTextAsInputStream(file, text);
    }

    @Test
    public void shouldSkipBomWhenWrappingInputStream() throws Exception {
        String text = "abcdefghijklmnop";
        this.shouldReadTextFromInputStreamWithBom(Magic.BOM_UTF_32_BE, text);
        this.shouldReadTextFromInputStreamWithBom(Magic.BOM_UTF_32_LE, text);
        this.shouldReadTextFromInputStreamWithBom(Magic.BOM_UTF_16_BE, text);
        this.shouldReadTextFromInputStreamWithBom(Magic.BOM_UTF_16_LE, text);
        this.shouldReadTextFromInputStreamWithBom(Magic.BOM_UTF_8, text);
    }

    private void shouldReadTextFromFileWithBom(Magic bom, String text) throws IOException {
        this.assertReadText(this.writeToFile(bom.bytes(), text, bom.encoding()), text);
    }

    private void shouldReadTextFromInputStreamWithBom(Magic bom, String text) throws IOException {
        this.assertReadTextAsInputStream(this.writeToFile(bom.bytes(), text, bom.encoding()), text);
    }

    private void shouldComplyWithSpecifiedCharset(Charset charset) throws Exception {
        String data = "abc\u00e5\u00e4\u00f6[]{}";
        File file = this.writeToFile(data, charset);
        CharReadable reader = Readables.files((Charset)charset, (File[])new File[]{file});
        SectionedCharBuffer buffer = new SectionedCharBuffer(100);
        buffer = reader.read(buffer, buffer.front());
        char[] expected = data.toCharArray();
        char[] array = buffer.array();
        Assert.assertEquals((long)expected.length, (long)buffer.available());
        for (int i = 0; i < expected.length; ++i) {
            Assert.assertEquals((long)expected[i], (long)array[buffer.pivot() + i]);
        }
    }

    private File writeToFile(String data, Charset charset) throws IOException {
        File file = new File(this.directory.directory(), "text-" + charset.name());
        try (OutputStreamWriter writer = new OutputStreamWriter((OutputStream)new FileOutputStream(file), charset);){
            ((Writer)writer).append(data);
        }
        return file;
    }

    private File writeToFile(byte[] header, String data, Charset charset) throws IOException {
        File file = new File(this.directory.directory(), "text-" + charset.name());
        try (FileOutputStream out = new FileOutputStream(file);
             OutputStreamWriter writer = new OutputStreamWriter((OutputStream)out, charset);){
            ((OutputStream)out).write(header);
            ((Writer)writer).append(data);
        }
        return file;
    }

    private File write(String text) throws IOException {
        File file = this.directory.file("plain-text");
        try (FileOutputStream out = new FileOutputStream(file);){
            ((OutputStream)out).write(text.getBytes());
        }
        return file;
    }

    private File compressWithZip(String text, String ... otherEntries) throws IOException {
        File file = this.directory.file("compressed");
        try (ZipOutputStream out = new ZipOutputStream(new FileOutputStream(file));){
            for (String otherEntry : otherEntries) {
                out.putNextEntry(new ZipEntry(otherEntry));
            }
            out.putNextEntry(new ZipEntry("file"));
            out.write(text.getBytes());
        }
        return file;
    }

    private File compressWithGZip(String text) throws IOException {
        File file = this.directory.file("compressed");
        try (GZIPOutputStream out = new GZIPOutputStream(new FileOutputStream(file));){
            out.write(text.getBytes());
        }
        return file;
    }

    private void assertReadText(File file, String text) throws IOException {
        this.assertReadText(Readables.files((Charset)Charset.defaultCharset(), (File[])new File[]{file}), text);
    }

    private void assertReadTextAsInputStream(File file, String text) throws IOException {
        try (FileInputStream stream = new FileInputStream(file);){
            this.assertReadText(Readables.wrap((InputStream)stream, (String)file.getPath(), (Charset)Charset.defaultCharset()), text);
        }
    }

    private void assertReadText(CharReadable readable, String text) throws IOException {
        char[] readText = this.readMethod.read(readable, text.toCharArray().length);
        Assert.assertArrayEquals((char[])readText, (char[])text.toCharArray());
    }

    static interface ReadMethod {
        public char[] read(CharReadable var1, int var2) throws IOException;
    }
}

