/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.collection.pool;

import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.LongSupplier;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.neo4j.collection.pool.LinkedQueuePool;

class LinkedQueuePoolTest {
    LinkedQueuePoolTest() {
    }

    @Test
    void shouldTimeoutGracefully() {
        FakeClock clock = new FakeClock();
        LinkedQueuePool.CheckStrategy.TimeoutCheckStrategy timeStrategy = new LinkedQueuePool.CheckStrategy.TimeoutCheckStrategy(100L, (LongSupplier)clock);
        while (clock.getAsLong() <= 100L) {
            Assertions.assertFalse((boolean)timeStrategy.shouldCheck());
            clock.forward(10L, TimeUnit.MILLISECONDS);
        }
        Assertions.assertTrue((boolean)timeStrategy.shouldCheck());
        clock.forward(1L, TimeUnit.MILLISECONDS);
        Assertions.assertFalse((boolean)timeStrategy.shouldCheck());
    }

    @Test
    void shouldBuildUpGracefullyUntilReachedMinPoolSize() {
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = new FakeClock();
        LinkedQueuePool<Object> pool = this.getLinkedQueuePool(stateMonitor, clock, 5);
        List<FlyweightHolder<Object>> flyweightHolders = this.acquireFromPool(pool, 5);
        for (FlyweightHolder<Object> flyweightHolder : flyweightHolders) {
            flyweightHolder.release();
        }
        Assertions.assertEquals((int)-1, (int)stateMonitor.currentPeakSize.get());
        Assertions.assertEquals((int)-1, (int)stateMonitor.targetSize.get());
        Assertions.assertEquals((int)0, (int)stateMonitor.disposed.get());
    }

    @Test
    void shouldBuildUpGracefullyWhilePassingMinPoolSizeBeforeTimerRings() {
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = new FakeClock();
        LinkedQueuePool<Object> pool = this.getLinkedQueuePool(stateMonitor, clock, 5);
        List<FlyweightHolder<Object>> flyweightHolders = this.acquireFromPool(pool, 15);
        for (FlyweightHolder<Object> flyweightHolder : flyweightHolders) {
            flyweightHolder.release();
        }
        Assertions.assertEquals((int)-1, (int)stateMonitor.currentPeakSize.get());
        Assertions.assertEquals((int)-1, (int)stateMonitor.targetSize.get());
        Assertions.assertEquals((int)15, (int)stateMonitor.created.get());
        Assertions.assertEquals((int)10, (int)stateMonitor.disposed.get());
    }

    @Test
    void shouldUpdateTargetSizeWhenSpikesOccur() {
        int MIN_SIZE = 5;
        int MAX_SIZE = 10;
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = new FakeClock();
        LinkedQueuePool<Object> pool = this.getLinkedQueuePool(stateMonitor, clock, 5);
        List<FlyweightHolder<Object>> holders = this.acquireFromPool(pool, 10);
        clock.forward(110L, TimeUnit.MILLISECONDS);
        holders.addAll(this.acquireFromPool(pool, 1));
        Assertions.assertEquals((int)11, (int)stateMonitor.currentPeakSize.get());
        Assertions.assertEquals((int)11, (int)stateMonitor.targetSize.get());
    }

    @Test
    void shouldKeepSmallPeakAndNeverDisposeIfAcquireAndReleaseContinuously() {
        boolean MIN_SIZE = true;
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = new FakeClock();
        LinkedQueuePool<Object> pool = this.getLinkedQueuePool(stateMonitor, clock, 1);
        for (int i = 0; i < 200; ++i) {
            List<FlyweightHolder<Object>> newOnes = this.acquireFromPool(pool, 1);
            for (FlyweightHolder<Object> newOne : newOnes) {
                newOne.release();
            }
        }
        Assertions.assertEquals((int)-1, (int)stateMonitor.currentPeakSize.get());
        Assertions.assertEquals((int)1, (int)stateMonitor.created.get());
        Assertions.assertEquals((int)0, (int)stateMonitor.disposed.get());
    }

    @Test
    void shouldSlowlyReduceTheNumberOfFlyweightsInThePoolWhenFlyweightsAreReleased() {
        int MIN_SIZE = 50;
        int MAX_SIZE = 200;
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = new FakeClock();
        LinkedQueuePool<Object> pool = this.getLinkedQueuePool(stateMonitor, clock, 50);
        LinkedList<FlyweightHolder<Object>> holders = new LinkedList<FlyweightHolder<Object>>();
        this.buildAPeakOfAcquiredFlyweightsAndTriggerAlarmWithSideEffects(200, clock, pool, holders);
        clock.forward(110L, TimeUnit.MILLISECONDS);
        for (int i = 0; i < 200; ++i) {
            this.acquireFromPool(pool, 1).get(0).release();
        }
        Assertions.assertEquals((int)1, (int)stateMonitor.currentPeakSize.get());
        Assertions.assertEquals((int)50, (int)stateMonitor.targetSize.get());
        Assertions.assertEquals((int)151, (int)stateMonitor.disposed.get());
    }

    @Test
    void shouldMaintainPoolAtHighWatermarkWhenConcurrentUsagePassesMinSize() {
        int MIN_SIZE = 50;
        int MAX_SIZE = 200;
        int MID_SIZE = 90;
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = new FakeClock();
        LinkedQueuePool<Object> pool = this.getLinkedQueuePool(stateMonitor, clock, 50);
        LinkedList<FlyweightHolder<Object>> holders = new LinkedList<FlyweightHolder<Object>>();
        this.buildAPeakOfAcquiredFlyweightsAndTriggerAlarmWithSideEffects(200, clock, pool, holders);
        Assertions.assertEquals((int)201, (int)stateMonitor.currentPeakSize.get());
        for (int i = 0; i < 2; ++i) {
            clock.forward(110L, TimeUnit.MILLISECONDS);
            for (FlyweightHolder<Object> holder : this.acquireFromPool(pool, 90)) {
                holder.release();
            }
            clock.forward(110L, TimeUnit.MILLISECONDS);
            for (FlyweightHolder<Object> holder : this.acquireFromPool(pool, 90)) {
                holder.release();
            }
        }
        Assertions.assertEquals((int)90, (int)stateMonitor.currentPeakSize.get());
        Assertions.assertEquals((int)90, (int)stateMonitor.targetSize.get());
        Assertions.assertEquals((int)111, (int)stateMonitor.disposed.get());
    }

    @Test
    void shouldReclaimAndRecreateWhenLullBetweenSpikesOccurs() {
        int MIN_SIZE = 50;
        int BELOW_MIN_SIZE = 10;
        int MAX_SIZE = 200;
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = new FakeClock();
        LinkedQueuePool<Object> pool = this.getLinkedQueuePool(stateMonitor, clock, 50);
        LinkedList<FlyweightHolder<Object>> holders = new LinkedList<FlyweightHolder<Object>>();
        this.buildAPeakOfAcquiredFlyweightsAndTriggerAlarmWithSideEffects(200, clock, pool, holders);
        clock.forward(110L, TimeUnit.MILLISECONDS);
        for (int i = 0; i < 30; ++i) {
            for (FlyweightHolder<Object> holder : this.acquireFromPool(pool, 10)) {
                holder.release();
            }
            clock.forward(110L, TimeUnit.MILLISECONDS);
        }
        Assertions.assertTrue((stateMonitor.currentPeakSize.get() <= 10 ? 1 : 0) != 0, (String)("Expected " + stateMonitor.currentPeakSize.get() + " <= " + 10));
        Assertions.assertEquals((int)50, (int)stateMonitor.targetSize.get());
        Assertions.assertEquals((int)151, (int)stateMonitor.disposed.get());
        stateMonitor.created.set(0);
        stateMonitor.disposed.set(0);
        holders.addAll(this.acquireFromPool(pool, 200));
        Assertions.assertEquals((int)150, (int)stateMonitor.created.get());
        Assertions.assertEquals((int)0, (int)stateMonitor.disposed.get());
    }

    private void buildAPeakOfAcquiredFlyweightsAndTriggerAlarmWithSideEffects(int MAX_SIZE, FakeClock clock, LinkedQueuePool<Object> pool, List<FlyweightHolder<Object>> holders) {
        holders.addAll(this.acquireFromPool(pool, MAX_SIZE));
        clock.forward(110L, TimeUnit.MILLISECONDS);
        holders.addAll(this.acquireFromPool(pool, 1));
        for (FlyweightHolder<Object> holder : holders) {
            holder.release();
        }
    }

    private LinkedQueuePool<Object> getLinkedQueuePool(StatefulMonitor stateMonitor, FakeClock clock, int minSize) {
        return new LinkedQueuePool(minSize, Object::new, (LinkedQueuePool.CheckStrategy)new LinkedQueuePool.CheckStrategy.TimeoutCheckStrategy(100L, (LongSupplier)clock), (LinkedQueuePool.Monitor)stateMonitor);
    }

    private <R> List<FlyweightHolder<R>> acquireFromPool(LinkedQueuePool<R> pool, int times) {
        LinkedList<FlyweightHolder<R>> acquirers = new LinkedList<FlyweightHolder<R>>();
        for (int i = 0; i < times; ++i) {
            FlyweightHolder holder = new FlyweightHolder(pool);
            acquirers.add(holder);
            holder.run();
        }
        return acquirers;
    }

    private static class FakeClock
    implements LongSupplier {
        private long time;

        private FakeClock() {
        }

        @Override
        public long getAsLong() {
            return this.time;
        }

        public void forward(long amount, TimeUnit timeUnit) {
            this.time += timeUnit.toMillis(amount);
        }
    }

    private static class StatefulMonitor
    implements LinkedQueuePool.Monitor<Object> {
        AtomicInteger currentPeakSize = new AtomicInteger(-1);
        AtomicInteger targetSize = new AtomicInteger(-1);
        AtomicInteger created = new AtomicInteger(0);
        AtomicInteger acquired = new AtomicInteger(0);
        AtomicInteger disposed = new AtomicInteger(0);

        private StatefulMonitor() {
        }

        public void updatedCurrentPeakSize(int currentPeakSize) {
            this.currentPeakSize.set(currentPeakSize);
        }

        public void updatedTargetSize(int targetSize) {
            this.targetSize.set(targetSize);
        }

        public void created(Object Object2) {
            this.created.incrementAndGet();
        }

        public void acquired(Object Object2) {
            this.acquired.incrementAndGet();
        }

        public void disposed(Object Object2) {
            this.disposed.incrementAndGet();
        }
    }

    private static class FlyweightHolder<R>
    implements Runnable {
        private final LinkedQueuePool<R> pool;
        private R resource;

        private FlyweightHolder(LinkedQueuePool<R> pool) {
            this.pool = pool;
        }

        @Override
        public void run() {
            this.resource = this.pool.acquire();
        }

        public void release() {
            this.pool.release(this.resource);
        }
    }
}

