/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.causalclustering.catchup.storecopy;

import io.netty.buffer.ByteBufAllocator;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Optional;
import java.util.Random;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.mockito.Matchers;
import org.mockito.Mockito;
import org.neo4j.adversaries.Adversary;
import org.neo4j.adversaries.RandomAdversary;
import org.neo4j.adversaries.fs.AdversarialFileSystemAbstraction;
import org.neo4j.causalclustering.catchup.storecopy.FileChunk;
import org.neo4j.causalclustering.catchup.storecopy.FileSender;
import org.neo4j.causalclustering.catchup.storecopy.StoreResource;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.StoreChannel;
import org.neo4j.io.pagecache.PageCache;
import org.neo4j.test.rule.TestDirectory;
import org.neo4j.test.rule.fs.EphemeralFileSystemRule;

public class FileSenderTest {
    private final Random random = new Random();
    @Rule
    public EphemeralFileSystemRule fsRule = new EphemeralFileSystemRule();
    private final FileSystemAbstraction fs = this.fsRule.get();
    @Rule
    public TestDirectory testDirectory = TestDirectory.testDirectory((FileSystemAbstraction)this.fsRule.get());
    private ByteBufAllocator allocator = (ByteBufAllocator)Mockito.mock(ByteBufAllocator.class);
    private PageCache pageCache = (PageCache)Mockito.mock(PageCache.class);

    @Before
    public void setup() throws IOException {
        Mockito.when((Object)this.pageCache.getExistingMapping((File)Matchers.any())).thenReturn(Optional.empty());
    }

    @Test
    public void sendEmptyFile() throws Exception {
        File emptyFile = this.testDirectory.file("emptyFile");
        this.fs.create(emptyFile).close();
        FileSender fileSender = new FileSender(new StoreResource(emptyFile, null, 16, this.pageCache, this.fs));
        Assert.assertFalse((boolean)fileSender.isEndOfInput());
        Assert.assertEquals((Object)FileChunk.create((byte[])new byte[0], (boolean)true), (Object)fileSender.readChunk(this.allocator));
        Assert.assertNull((Object)fileSender.readChunk(this.allocator));
        Assert.assertTrue((boolean)fileSender.isEndOfInput());
    }

    @Test
    public void sendSmallFile() throws Exception {
        byte[] bytes = new byte[10];
        this.random.nextBytes(bytes);
        File smallFile = this.testDirectory.file("smallFile");
        try (StoreChannel storeChannel = this.fs.create(smallFile);){
            storeChannel.write(ByteBuffer.wrap(bytes));
        }
        FileSender fileSender = new FileSender(new StoreResource(smallFile, null, 16, this.pageCache, this.fs));
        Assert.assertFalse((boolean)fileSender.isEndOfInput());
        Assert.assertEquals((Object)FileChunk.create((byte[])bytes, (boolean)true), (Object)fileSender.readChunk(this.allocator));
        Assert.assertNull((Object)fileSender.readChunk(this.allocator));
        Assert.assertTrue((boolean)fileSender.isEndOfInput());
    }

    @Test
    public void sendLargeFile() throws Exception {
        int dataSize = 12288;
        byte[] bytes = new byte[dataSize];
        this.random.nextBytes(bytes);
        File smallFile = this.testDirectory.file("smallFile");
        try (StoreChannel storeChannel = this.fs.create(smallFile);){
            storeChannel.write(ByteBuffer.wrap(bytes));
        }
        FileSender fileSender = new FileSender(new StoreResource(smallFile, null, 16, this.pageCache, this.fs));
        Assert.assertFalse((boolean)fileSender.isEndOfInput());
        Assert.assertEquals((Object)FileChunk.create((byte[])Arrays.copyOfRange(bytes, 0, 8192), (boolean)false), (Object)fileSender.readChunk(this.allocator));
        Assert.assertEquals((Object)FileChunk.create((byte[])Arrays.copyOfRange(bytes, 8192, bytes.length), (boolean)true), (Object)fileSender.readChunk(this.allocator));
        Assert.assertNull((Object)fileSender.readChunk(this.allocator));
        Assert.assertTrue((boolean)fileSender.isEndOfInput());
    }

    @Test
    public void sendLargeFileWithSizeMultipleOfTheChunkSize() throws Exception {
        byte[] bytes = new byte[24576];
        this.random.nextBytes(bytes);
        File smallFile = this.testDirectory.file("smallFile");
        try (StoreChannel storeChannel = this.fs.create(smallFile);){
            storeChannel.write(ByteBuffer.wrap(bytes));
        }
        FileSender fileSender = new FileSender(new StoreResource(smallFile, null, 16, this.pageCache, this.fs));
        Assert.assertFalse((boolean)fileSender.isEndOfInput());
        Assert.assertEquals((Object)FileChunk.create((byte[])Arrays.copyOfRange(bytes, 0, 8192), (boolean)false), (Object)fileSender.readChunk(this.allocator));
        Assert.assertEquals((Object)FileChunk.create((byte[])Arrays.copyOfRange(bytes, 8192, 16384), (boolean)false), (Object)fileSender.readChunk(this.allocator));
        Assert.assertEquals((Object)FileChunk.create((byte[])Arrays.copyOfRange(bytes, 16384, bytes.length), (boolean)true), (Object)fileSender.readChunk(this.allocator));
        Assert.assertNull((Object)fileSender.readChunk(this.allocator));
        Assert.assertTrue((boolean)fileSender.isEndOfInput());
    }

    @Test
    public void sendEmptyFileWhichGrowsBeforeSendCommences() throws Exception {
        File file = this.testDirectory.file("file");
        StoreChannel writer = this.fs.create(file);
        FileSender fileSender = new FileSender(new StoreResource(file, null, 16, this.pageCache, this.fs));
        byte[] bytes = this.writeRandomBytes(writer, 1024);
        Assert.assertFalse((boolean)fileSender.isEndOfInput());
        Assert.assertEquals((Object)FileChunk.create((byte[])bytes, (boolean)true), (Object)fileSender.readChunk(this.allocator));
        Assert.assertTrue((boolean)fileSender.isEndOfInput());
        Assert.assertNull((Object)fileSender.readChunk(this.allocator));
    }

    @Test
    public void sendEmptyFileWhichGrowsWithPartialChunkSizes() throws Exception {
        File file = this.testDirectory.file("file");
        StoreChannel writer = this.fs.create(file);
        FileSender fileSender = new FileSender(new StoreResource(file, null, 16, this.pageCache, this.fs));
        byte[] chunkA = this.writeRandomBytes(writer, 8192);
        byte[] chunkB = this.writeRandomBytes(writer, 4096);
        Assert.assertEquals((Object)FileChunk.create((byte[])chunkA, (boolean)false), (Object)fileSender.readChunk(this.allocator));
        Assert.assertFalse((boolean)fileSender.isEndOfInput());
        this.writeRandomBytes(writer, 4096);
        Assert.assertEquals((Object)FileChunk.create((byte[])chunkB, (boolean)true), (Object)fileSender.readChunk(this.allocator));
        Assert.assertTrue((boolean)fileSender.isEndOfInput());
        Assert.assertNull((Object)fileSender.readChunk(this.allocator));
    }

    @Test
    public void sendFileWhichGrowsAfterLastChunkWasSent() throws Exception {
        File file = this.testDirectory.file("file");
        StoreChannel writer = this.fs.create(file);
        FileSender fileSender = new FileSender(new StoreResource(file, null, 16, this.pageCache, this.fs));
        byte[] chunkA = this.writeRandomBytes(writer, 8192);
        FileChunk readChunkA = fileSender.readChunk(this.allocator);
        Assert.assertEquals((Object)FileChunk.create((byte[])chunkA, (boolean)true), (Object)readChunkA);
        Assert.assertTrue((boolean)fileSender.isEndOfInput());
        this.writeRandomBytes(writer, 8192);
        Assert.assertTrue((boolean)fileSender.isEndOfInput());
        Assert.assertNull((Object)fileSender.readChunk(this.allocator));
    }

    @Test
    public void sendLargerFileWhichGrows() throws Exception {
        File file = this.testDirectory.file("file");
        StoreChannel writer = this.fs.create(file);
        FileSender fileSender = new FileSender(new StoreResource(file, null, 16, this.pageCache, this.fs));
        byte[] chunkA = this.writeRandomBytes(writer, 8192);
        byte[] chunkB = this.writeRandomBytes(writer, 8192);
        FileChunk readChunkA = fileSender.readChunk(this.allocator);
        Assert.assertEquals((Object)FileChunk.create((byte[])chunkA, (boolean)false), (Object)readChunkA);
        Assert.assertFalse((boolean)fileSender.isEndOfInput());
        byte[] chunkC = this.writeRandomBytes(writer, 8192);
        FileChunk readChunkB = fileSender.readChunk(this.allocator);
        Assert.assertEquals((Object)FileChunk.create((byte[])chunkB, (boolean)false), (Object)readChunkB);
        Assert.assertFalse((boolean)fileSender.isEndOfInput());
        FileChunk readChunkC = fileSender.readChunk(this.allocator);
        Assert.assertEquals((Object)FileChunk.create((byte[])chunkC, (boolean)true), (Object)readChunkC);
        Assert.assertTrue((boolean)fileSender.isEndOfInput());
        Assert.assertNull((Object)fileSender.readChunk(this.allocator));
    }

    @Test
    public void sendLargeFileWithUnreliableReadBufferSize() throws Exception {
        byte[] bytes = new byte[24576];
        this.random.nextBytes(bytes);
        File smallFile = this.testDirectory.file("smallFile");
        try (StoreChannel storeChannel = this.fs.create(smallFile);){
            storeChannel.write(ByteBuffer.wrap(bytes));
        }
        RandomAdversary adversary = new RandomAdversary(0.9, 0.0, 0.0);
        AdversarialFileSystemAbstraction afs = new AdversarialFileSystemAbstraction((Adversary)adversary, this.fs);
        FileSender fileSender = new FileSender(new StoreResource(smallFile, null, 16, this.pageCache, (FileSystemAbstraction)afs));
        Assert.assertFalse((boolean)fileSender.isEndOfInput());
        Assert.assertEquals((Object)FileChunk.create((byte[])Arrays.copyOfRange(bytes, 0, 8192), (boolean)false), (Object)fileSender.readChunk(this.allocator));
        Assert.assertEquals((Object)FileChunk.create((byte[])Arrays.copyOfRange(bytes, 8192, 16384), (boolean)false), (Object)fileSender.readChunk(this.allocator));
        Assert.assertEquals((Object)FileChunk.create((byte[])Arrays.copyOfRange(bytes, 16384, bytes.length), (boolean)true), (Object)fileSender.readChunk(this.allocator));
        Assert.assertNull((Object)fileSender.readChunk(this.allocator));
        Assert.assertTrue((boolean)fileSender.isEndOfInput());
    }

    private byte[] writeRandomBytes(StoreChannel writer, int size) throws IOException {
        byte[] bytes = new byte[size];
        this.random.nextBytes(bytes);
        writer.writeAll(ByteBuffer.wrap(bytes));
        return bytes;
    }
}

