/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.causalclustering.readreplica;

import java.util.concurrent.locks.LockSupport;
import org.neo4j.causalclustering.catchup.storecopy.CopiedStoreRecovery;
import org.neo4j.causalclustering.catchup.storecopy.LocalDatabase;
import org.neo4j.causalclustering.catchup.storecopy.StoreCopyFailedException;
import org.neo4j.causalclustering.catchup.storecopy.StoreFetcher;
import org.neo4j.causalclustering.catchup.storecopy.StoreIdDownloadFailedException;
import org.neo4j.causalclustering.catchup.storecopy.StreamingTransactionsFailedException;
import org.neo4j.causalclustering.core.state.machines.tx.RetryStrategy;
import org.neo4j.causalclustering.identity.MemberId;
import org.neo4j.causalclustering.identity.StoreId;
import org.neo4j.causalclustering.messaging.routing.CoreMemberSelectionException;
import org.neo4j.causalclustering.messaging.routing.CoreMemberSelectionStrategy;
import org.neo4j.causalclustering.readreplica.CopyStoreSafely;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.kernel.lifecycle.Lifecycle;
import org.neo4j.logging.Log;
import org.neo4j.logging.LogProvider;

class ReadReplicaStartupProcess
implements Lifecycle {
    private final FileSystemAbstraction fs;
    private final StoreFetcher storeFetcher;
    private final LocalDatabase localDatabase;
    private final Lifecycle txPulling;
    private final CoreMemberSelectionStrategy connectionStrategy;
    private final Log log;
    private final RetryStrategy.Timeout timeout;
    private final CopiedStoreRecovery copiedStoreRecovery;

    ReadReplicaStartupProcess(FileSystemAbstraction fs, StoreFetcher storeFetcher, LocalDatabase localDatabase, Lifecycle txPulling, CoreMemberSelectionStrategy connectionStrategy, RetryStrategy retryStrategy, LogProvider logProvider, CopiedStoreRecovery copiedStoreRecovery) {
        this.fs = fs;
        this.storeFetcher = storeFetcher;
        this.localDatabase = localDatabase;
        this.txPulling = txPulling;
        this.connectionStrategy = connectionStrategy;
        this.copiedStoreRecovery = copiedStoreRecovery;
        this.timeout = retryStrategy.newTimeout();
        this.log = logProvider.getLog(this.getClass());
    }

    public void init() throws Throwable {
        this.localDatabase.init();
        this.txPulling.init();
    }

    public void start() throws Throwable {
        long retryInterval = 5000L;
        int attempts = 0;
        while (attempts++ < 5) {
            MemberId source = this.findCoreMemberToCopyFrom();
            try {
                this.tryToStart(source);
                return;
            }
            catch (StoreCopyFailedException e) {
                this.log.info("Attempt #%d to start read replica failed while copying store files from %s.", new Object[]{attempts, source});
            }
            catch (StreamingTransactionsFailedException e) {
                this.log.info("Attempt #%d to start read replica failed while streaming transactions from %s.", new Object[]{attempts, source});
            }
            catch (StoreIdDownloadFailedException e) {
                this.log.info("Attempt #%d to start read replica failed while getting store id from %s.", new Object[]{attempts, source});
            }
            try {
                Thread.sleep(retryInterval);
                retryInterval = Math.min(60000L, retryInterval * 2L);
            }
            catch (InterruptedException e) {
                Thread.interrupted();
                throw new RuntimeException("Interrupted while trying to start read replica.", e);
            }
        }
        throw new Exception("Failed to start read replica after " + (attempts - 1) + " attempts");
    }

    private void tryToStart(MemberId source) throws Throwable {
        if (this.localDatabase.isEmpty()) {
            this.log.info("Local database is empty, attempting to replace with copy from core server %s", new Object[]{source});
            this.log.info("Finding store id of core server %s", new Object[]{source});
            StoreId storeId = this.storeFetcher.getStoreIdOf(source);
            this.log.info("Copying store from core server %s", new Object[]{source});
            this.localDatabase.delete();
            new CopyStoreSafely(this.fs, this.localDatabase, this.copiedStoreRecovery, this.log).copyWholeStoreFrom(source, storeId, this.storeFetcher);
            this.log.info("Restarting local database after copy.", new Object[]{source});
        } else {
            this.ensureSameStoreIdAs(source);
        }
        this.localDatabase.start();
        this.txPulling.start();
    }

    private void ensureSameStoreIdAs(MemberId remoteCore) throws StoreIdDownloadFailedException {
        StoreId remoteStoreId;
        StoreId localStoreId = this.localDatabase.storeId();
        if (!localStoreId.equals(remoteStoreId = this.storeFetcher.getStoreIdOf(remoteCore))) {
            throw new IllegalStateException(String.format("This read replica cannot join the cluster. The local database is not empty and has a mismatching storeId: expected %s actual %s.", remoteStoreId, localStoreId));
        }
    }

    private MemberId findCoreMemberToCopyFrom() {
        while (true) {
            try {
                MemberId memberId = this.connectionStrategy.coreMember();
                this.log.info("Server starting, connecting to core server %s", new Object[]{memberId});
                return memberId;
            }
            catch (CoreMemberSelectionException ex) {
                this.log.info("Failed to connect to core server. Retrying in %d ms.", new Object[]{this.timeout.getMillis()});
                LockSupport.parkUntil(this.timeout.getMillis() + System.currentTimeMillis());
                this.timeout.increment();
                continue;
            }
            break;
        }
    }

    public void stop() throws Throwable {
        this.txPulling.stop();
        this.localDatabase.stop();
    }

    public void shutdown() throws Throwable {
        this.txPulling.shutdown();
        this.localDatabase.shutdown();
    }
}

