/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.restore;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;
import org.neo4j.commandline.admin.AdminCommand;
import org.neo4j.commandline.admin.CommandFailed;
import org.neo4j.commandline.admin.CommandLocator;
import org.neo4j.commandline.admin.Usage;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.PropertyContainer;
import org.neo4j.graphdb.Relationship;
import org.neo4j.graphdb.RelationshipType;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.config.Setting;
import org.neo4j.graphdb.factory.GraphDatabaseFactory;
import org.neo4j.graphdb.factory.GraphDatabaseSettings;
import org.neo4j.graphdb.index.Index;
import org.neo4j.graphdb.index.IndexHits;
import org.neo4j.graphdb.index.IndexManager;
import org.neo4j.graphdb.index.RelationshipIndex;
import org.neo4j.helpers.collection.Iterables;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.kernel.configuration.Config;
import org.neo4j.kernel.impl.enterprise.configuration.OnlineBackupSettings;
import org.neo4j.kernel.impl.transaction.log.files.LogFiles;
import org.neo4j.kernel.impl.transaction.log.files.LogFilesBuilder;
import org.neo4j.kernel.internal.locker.StoreLocker;
import org.neo4j.restore.RestoreDatabaseCliProvider;
import org.neo4j.restore.RestoreDatabaseCommand;
import org.neo4j.test.rule.TestDirectory;
import org.neo4j.test.rule.fs.DefaultFileSystemRule;

public class RestoreDatabaseCommandIT {
    @Rule
    public final TestDirectory directory = TestDirectory.testDirectory();
    @Rule
    public final DefaultFileSystemRule fileSystemRule = new DefaultFileSystemRule();
    @Rule
    public final ExpectedException expectedException = ExpectedException.none();
    private Config toConfig;
    private static final String databaseName = "to";
    private File fromStore;
    private File fromDatabase;
    private String fromLogicalLogsPath;
    private File toStore;
    private String toLogicalLogsPath;

    @Before
    public void setUp() throws Exception {
        this.toConfig = RestoreDatabaseCommandIT.configWith(databaseName, this.directory.absolutePath().getAbsolutePath());
        this.fromStore = this.directory.storeDir("from");
        this.fromDatabase = this.directory.databaseDir(this.fromStore);
        this.fromLogicalLogsPath = this.fromDatabase.getAbsolutePath();
        this.toStore = ((File)this.toConfig.get(GraphDatabaseSettings.database_path)).getParentFile();
        this.toLogicalLogsPath = this.toStore.getAbsolutePath();
    }

    @Test
    public void forceShouldRespectStoreLock() throws Exception {
        RestoreDatabaseCommandIT.createDbAt(this.fromStore, this.fromLogicalLogsPath, 10);
        RestoreDatabaseCommandIT.createDbAt(this.toStore, this.toLogicalLogsPath, 20, this.toConfig);
        this.expectedException.expect(CommandFailed.class);
        this.expectedException.expectMessage("the database is in use -- stop Neo4j and try again");
        FileSystemAbstraction fs = this.fileSystemRule.get();
        try (StoreLocker storeLocker = new StoreLocker(fs, this.toStore);){
            storeLocker.checkLock();
            new RestoreDatabaseCommand(fs, this.fromDatabase, this.toConfig, databaseName, true).execute();
        }
    }

    @Test
    public void shouldNotCopyOverAndExistingDatabase() throws Exception {
        RestoreDatabaseCommandIT.createDbAt(this.fromStore, this.fromLogicalLogsPath, 0);
        RestoreDatabaseCommandIT.createDbAt(this.toStore, this.toLogicalLogsPath, 0, this.toConfig);
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage("Database with name [to] already exists");
        new RestoreDatabaseCommand(this.fileSystemRule.get(), this.fromDatabase, this.toConfig, databaseName, false).execute();
    }

    @Test
    public void shouldThrowExceptionIfBackupDirectoryDoesNotExist() throws Exception {
        Assert.assertTrue((boolean)this.fromDatabase.delete());
        RestoreDatabaseCommandIT.createDbAt(this.toStore, this.toLogicalLogsPath, 0, this.toConfig);
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage("Source directory does not exist");
        new RestoreDatabaseCommand(this.fileSystemRule.get(), this.fromDatabase, this.toConfig, databaseName, false).execute();
    }

    @Test
    public void shouldThrowExceptionIfBackupDirectoryDoesNotHaveStoreFiles() throws Exception {
        this.expectedException.expect(IllegalArgumentException.class);
        this.expectedException.expectMessage("Source directory is not a database backup");
        new RestoreDatabaseCommand(this.fileSystemRule.get(), this.fromDatabase, this.toConfig, databaseName, false).execute();
    }

    @Test
    public void shouldAllowForcedCopyOverAnExistingDatabase() throws Exception {
        RestoreDatabaseCommandIT.createDbAt(this.fromStore, this.fromLogicalLogsPath, 10);
        RestoreDatabaseCommandIT.createDbAt(this.toStore, this.toLogicalLogsPath, 20, this.toConfig);
        new RestoreDatabaseCommand(this.fileSystemRule.get(), this.fromDatabase, this.toConfig, databaseName, true).execute();
        this.toConfig.augment(OnlineBackupSettings.online_backup_enabled, "false");
        GraphDatabaseService copiedDb = new GraphDatabaseFactory().newEmbeddedDatabaseBuilder(this.toStore).setConfig(this.toConfig.getRaw()).newGraphDatabase();
        try (Transaction ignored = copiedDb.beginTx();){
            Assert.assertEquals((long)10L, (long)Iterables.count((Iterable)copiedDb.getAllNodes()));
        }
        copiedDb.shutdown();
    }

    @Test
    public void restoreExplicitIndexesFromBackup() throws IOException, CommandFailed {
        String databaseName = "destination";
        Config toConfig = RestoreDatabaseCommandIT.configWith(databaseName, this.directory.absolutePath().getAbsolutePath());
        File toStorePath = (File)toConfig.get(GraphDatabaseSettings.database_path);
        RestoreDatabaseCommandIT.createDbWithExplicitIndexAt(this.fromStore, 100);
        new RestoreDatabaseCommand(this.fileSystemRule.get(), this.fromDatabase, toConfig, databaseName, true).execute();
        GraphDatabaseService restoredDatabase = RestoreDatabaseCommandIT.createDatabase(toStorePath, toStorePath.getAbsolutePath(), toConfig);
        try (Transaction transaction = restoredDatabase.beginTx();){
            IndexManager indexManager = restoredDatabase.index();
            String[] nodeIndexNames = indexManager.nodeIndexNames();
            String[] relationshipIndexNames = indexManager.relationshipIndexNames();
            for (String nodeIndexName : nodeIndexNames) {
                RestoreDatabaseCommandIT.countNodesByKeyValue(indexManager, nodeIndexName, "a", "b");
                RestoreDatabaseCommandIT.countNodesByKeyValue(indexManager, nodeIndexName, "c", "d");
            }
            for (String relationshipIndexName : relationshipIndexNames) {
                RestoreDatabaseCommandIT.countRelationshipByKeyValue(indexManager, relationshipIndexName, "x", "y");
            }
            transaction.success();
        }
        restoredDatabase.shutdown();
    }

    @Test
    public void restoreTransactionLogsInCustomDirectoryForTargetDatabaseWhenConfigured() throws IOException, CommandFailed {
        File customTxLogDirectory = this.directory.directory("customLogicalLog");
        String customTransactionLogDirectory = customTxLogDirectory.getAbsolutePath();
        RestoreDatabaseCommandIT.createDbAt(this.fromStore, this.fromLogicalLogsPath, 10);
        GraphDatabaseService db = RestoreDatabaseCommandIT.createDatabase(this.toStore, customTransactionLogDirectory, this.toConfig);
        RestoreDatabaseCommandIT.createTestData(20, db);
        db.shutdown();
        new RestoreDatabaseCommand(this.fileSystemRule.get(), this.fromDatabase, this.toConfig, databaseName, true).execute();
        LogFiles fromStoreLogFiles = LogFilesBuilder.logFilesBasedOnlyBuilder((File)this.fromDatabase, (FileSystemAbstraction)this.fileSystemRule.get()).build();
        LogFiles toStoreLogFiles = LogFilesBuilder.logFilesBasedOnlyBuilder((File)new File(this.toStore, databaseName), (FileSystemAbstraction)this.fileSystemRule.get()).build();
        LogFiles customLogLocationLogFiles = LogFilesBuilder.logFilesBasedOnlyBuilder((File)customTxLogDirectory, (FileSystemAbstraction)this.fileSystemRule.get()).build();
        Assert.assertThat((Object)toStoreLogFiles.logFiles(), (Matcher)Matchers.emptyArray());
        Assert.assertThat((Object)customLogLocationLogFiles.logFiles(), (Matcher)Matchers.arrayWithSize((int)1));
        Assert.assertEquals((long)fromStoreLogFiles.getLogFileForVersion(0L).length(), (long)customLogLocationLogFiles.getLogFileForVersion(0L).length());
    }

    @Test
    public void doNotRemoveRelativeTransactionDirectoryAgain() throws IOException, CommandFailed {
        FileSystemAbstraction fileSystem = (FileSystemAbstraction)Mockito.spy((Object)this.fileSystemRule.get());
        File storeDir = this.directory.directory();
        String testDatabase = "testDatabase";
        File databaseDirectory = new File(storeDir, testDatabase);
        File relativeLogDirectory = new File(databaseDirectory, "relativeDirectory");
        Config config = Config.defaults((Setting)GraphDatabaseSettings.database_path, (String)databaseDirectory.getAbsolutePath());
        config.augment(GraphDatabaseSettings.logical_logs_location, relativeLogDirectory.getAbsolutePath());
        RestoreDatabaseCommandIT.createDbAt(this.fromStore, this.fromLogicalLogsPath, 10);
        new RestoreDatabaseCommand(fileSystem, this.fromDatabase, config, testDatabase, true).execute();
        ((FileSystemAbstraction)Mockito.verify((Object)fileSystem)).deleteRecursively((File)ArgumentMatchers.eq((Object)databaseDirectory));
        ((FileSystemAbstraction)Mockito.verify((Object)fileSystem, (VerificationMode)Mockito.never())).deleteRecursively((File)ArgumentMatchers.eq((Object)relativeLogDirectory));
    }

    @Test
    public void shouldPrintNiceHelp() throws Throwable {
        try (ByteArrayOutputStream baos = new ByteArrayOutputStream();){
            PrintStream ps = new PrintStream(baos);
            Usage usage = new Usage("neo4j-admin", (CommandLocator)Mockito.mock(CommandLocator.class));
            usage.printUsageForCommand((AdminCommand.Provider)new RestoreDatabaseCliProvider(), ps::println);
            Assert.assertEquals((Object)String.format("usage: neo4j-admin restore --from=<backup-directory> [--database=<name>]%n                           [--force[=<true|false>]]%n%nenvironment variables:%n    NEO4J_CONF    Path to directory which contains neo4j.conf.%n    NEO4J_DEBUG   Set to anything to enable debug output.%n    NEO4J_HOME    Neo4j home directory.%n    HEAP_SIZE     Set JVM maximum heap size during command execution.%n                  Takes a number and a unit, for example 512m.%n%nRestore a backed up database.%n%noptions:%n  --from=<backup-directory>   Path to backup to restore from.%n  --database=<name>           Name of database. [default:graph.db]%n  --force=<true|false>        If an existing database should be replaced.%n                              [default:false]%n", new Object[0]), (Object)baos.toString());
        }
    }

    private static void countRelationshipByKeyValue(IndexManager indexManager, String indexName, String key, String value) {
        try (IndexHits nodes = indexManager.forRelationships(indexName).get(key, (Object)value);){
            Assert.assertEquals((long)50L, (long)nodes.size());
        }
    }

    private static void countNodesByKeyValue(IndexManager indexManager, String indexName, String key, String value) {
        try (IndexHits nodes = indexManager.forNodes(indexName).get(key, (Object)value);){
            Assert.assertEquals((long)50L, (long)nodes.size());
        }
    }

    private static Config configWith(String databaseName, String dataDirectory) {
        return Config.builder().withSetting(GraphDatabaseSettings.active_database.name(), databaseName).withSetting(GraphDatabaseSettings.data_directory.name(), dataDirectory).build();
    }

    private static void createDbAt(File fromPath, String logicalLogsDirectory, int nodesToCreate) {
        GraphDatabaseService db = RestoreDatabaseCommandIT.createDatabase(fromPath, logicalLogsDirectory, Config.defaults());
        RestoreDatabaseCommandIT.createTestData(nodesToCreate, db);
        db.shutdown();
    }

    private static void createDbAt(File fromPath, String logicalLogsDirectory, int nodesToCreate, Config config) {
        GraphDatabaseService db = RestoreDatabaseCommandIT.createDatabase(fromPath, logicalLogsDirectory, config);
        RestoreDatabaseCommandIT.createTestData(nodesToCreate, db);
        db.shutdown();
    }

    private static GraphDatabaseService createDatabase(File fromPath, String logicalLogsDirectory, Config config) {
        config.augment(OnlineBackupSettings.online_backup_enabled, "false");
        config.augment(GraphDatabaseSettings.logical_logs_location, logicalLogsDirectory);
        return new GraphDatabaseFactory().newEmbeddedDatabaseBuilder(fromPath).setConfig(config.getRaw()).newGraphDatabase();
    }

    private static void createDbWithExplicitIndexAt(File fromPath, int pairNumberOfNodesToCreate) {
        RelationshipIndex explicitRelationshipIndex;
        Index explicitNodeIndex;
        GraphDatabaseService db = RestoreDatabaseCommandIT.createDatabase(fromPath, fromPath.getAbsolutePath(), Config.defaults());
        try (Transaction transaction = db.beginTx();){
            explicitNodeIndex = db.index().forNodes("explicitNodeIndex");
            explicitRelationshipIndex = db.index().forRelationships("explicitRelationshipIndex");
            transaction.success();
        }
        var6_4 = null;
        try (Transaction tx = db.beginTx();){
            for (int i = 0; i < pairNumberOfNodesToCreate; i += 2) {
                Node node = db.createNode();
                Node otherNode = db.createNode();
                Relationship relationship = node.createRelationshipTo(otherNode, RelationshipType.withName((String)"rel"));
                explicitNodeIndex.add((PropertyContainer)node, "a", (Object)"b");
                explicitNodeIndex.add((PropertyContainer)otherNode, "c", (Object)"d");
                explicitRelationshipIndex.add((PropertyContainer)relationship, "x", (Object)"y");
            }
            tx.success();
        }
        catch (Throwable throwable) {
            var6_4 = throwable;
            throw throwable;
        }
        db.shutdown();
    }

    private static void createTestData(int nodesToCreate, GraphDatabaseService db) {
        try (Transaction tx = db.beginTx();){
            for (int i = 0; i < nodesToCreate; ++i) {
                db.createNode();
            }
            tx.success();
        }
    }
}

