/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.restore;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Map;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.junit.Assert;
import org.junit.Rule;
import org.junit.Test;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;
import org.neo4j.commandline.admin.AdminCommand;
import org.neo4j.commandline.admin.CommandFailed;
import org.neo4j.commandline.admin.CommandLocator;
import org.neo4j.commandline.admin.Usage;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.PropertyContainer;
import org.neo4j.graphdb.Relationship;
import org.neo4j.graphdb.RelationshipType;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.config.Setting;
import org.neo4j.graphdb.factory.GraphDatabaseFactory;
import org.neo4j.graphdb.factory.GraphDatabaseSettings;
import org.neo4j.graphdb.index.Index;
import org.neo4j.graphdb.index.IndexHits;
import org.neo4j.graphdb.index.IndexManager;
import org.neo4j.graphdb.index.RelationshipIndex;
import org.neo4j.helpers.collection.Iterables;
import org.neo4j.helpers.collection.MapUtil;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.kernel.configuration.Config;
import org.neo4j.kernel.impl.enterprise.configuration.OnlineBackupSettings;
import org.neo4j.kernel.impl.transaction.log.files.LogFiles;
import org.neo4j.kernel.impl.transaction.log.files.LogFilesBuilder;
import org.neo4j.kernel.internal.locker.StoreLocker;
import org.neo4j.restore.RestoreDatabaseCliProvider;
import org.neo4j.restore.RestoreDatabaseCommand;
import org.neo4j.test.rule.TestDirectory;
import org.neo4j.test.rule.fs.DefaultFileSystemRule;

public class RestoreDatabaseCommandIT {
    @Rule
    public final TestDirectory directory = TestDirectory.testDirectory();
    @Rule
    public final DefaultFileSystemRule fileSystemRule = new DefaultFileSystemRule();

    @Test
    public void forceShouldRespectStoreLock() {
        String databaseName = "to";
        Config config = RestoreDatabaseCommandIT.configWith(databaseName, this.directory.absolutePath().getAbsolutePath());
        File fromPath = new File(this.directory.absolutePath(), "from");
        File toPath = (File)config.get(GraphDatabaseSettings.database_path);
        int fromNodeCount = 10;
        int toNodeCount = 20;
        this.createDbAt(fromPath, fromNodeCount);
        this.createDbAt(toPath, toNodeCount);
        FileSystemAbstraction fs = this.fileSystemRule.get();
        try (StoreLocker storeLocker = new StoreLocker(fs, toPath);){
            storeLocker.checkLock();
            new RestoreDatabaseCommand(fs, fromPath, config, databaseName, true).execute();
            Assert.fail((String)"expected exception");
        }
        catch (Exception e) {
            Assert.assertThat((Object)e.getMessage(), (Matcher)Matchers.equalTo((Object)"the database is in use -- stop Neo4j and try again"));
        }
    }

    @Test
    public void shouldNotCopyOverAndExistingDatabase() throws Exception {
        String databaseName = "to";
        Config config = RestoreDatabaseCommandIT.configWith(databaseName, this.directory.absolutePath().getAbsolutePath());
        File fromPath = new File(this.directory.absolutePath(), "from");
        File toPath = (File)config.get(GraphDatabaseSettings.database_path);
        this.createDbAt(fromPath, 0);
        this.createDbAt(toPath, 0);
        try {
            new RestoreDatabaseCommand(this.fileSystemRule.get(), fromPath, config, databaseName, false).execute();
            Assert.fail((String)"Should have thrown exception");
        }
        catch (IllegalArgumentException exception) {
            Assert.assertTrue((String)exception.getMessage(), (boolean)exception.getMessage().contains("Database with name [to] already exists"));
        }
    }

    @Test
    public void shouldThrowExceptionIfBackupDirectoryDoesNotExist() throws Exception {
        String databaseName = "to";
        Config config = RestoreDatabaseCommandIT.configWith(databaseName, this.directory.absolutePath().getAbsolutePath());
        File fromPath = new File(this.directory.absolutePath(), "from");
        File toPath = (File)config.get(GraphDatabaseSettings.database_path);
        this.createDbAt(toPath, 0);
        try {
            new RestoreDatabaseCommand(this.fileSystemRule.get(), fromPath, config, databaseName, false).execute();
            Assert.fail((String)"Should have thrown exception");
        }
        catch (IllegalArgumentException exception) {
            Assert.assertTrue((String)exception.getMessage(), (boolean)exception.getMessage().contains("Source directory does not exist"));
        }
    }

    @Test
    public void shouldThrowExceptionIfBackupDirectoryDoesNotHaveStoreFiles() throws Exception {
        String databaseName = "to";
        Config config = RestoreDatabaseCommandIT.configWith(databaseName, this.directory.absolutePath().getAbsolutePath());
        File fromPath = new File(this.directory.absolutePath(), "from");
        Assert.assertTrue((boolean)fromPath.mkdirs());
        try {
            new RestoreDatabaseCommand(this.fileSystemRule.get(), fromPath, config, databaseName, false).execute();
            Assert.fail((String)"Should have thrown exception");
        }
        catch (IllegalArgumentException exception) {
            Assert.assertTrue((String)exception.getMessage(), (boolean)exception.getMessage().contains("Source directory is not a database backup"));
        }
    }

    @Test
    public void shouldAllowForcedCopyOverAnExistingDatabase() throws Exception {
        String databaseName = "to";
        Config config = RestoreDatabaseCommandIT.configWith(databaseName, this.directory.absolutePath().getAbsolutePath());
        File fromPath = new File(this.directory.absolutePath(), "from");
        File toPath = (File)config.get(GraphDatabaseSettings.database_path);
        int fromNodeCount = 10;
        int toNodeCount = 20;
        this.createDbAt(fromPath, fromNodeCount);
        this.createDbAt(toPath, toNodeCount);
        new RestoreDatabaseCommand(this.fileSystemRule.get(), fromPath, config, databaseName, true).execute();
        GraphDatabaseService copiedDb = new GraphDatabaseFactory().newEmbeddedDatabaseBuilder(toPath).setConfig(OnlineBackupSettings.online_backup_enabled, "false").newGraphDatabase();
        try (Transaction ignored = copiedDb.beginTx();){
            Assert.assertEquals((long)fromNodeCount, (long)Iterables.count((Iterable)copiedDb.getAllNodes()));
        }
        copiedDb.shutdown();
    }

    @Test
    public void restoreExplicitIndexesFromBackup() throws IOException, CommandFailed {
        String databaseName = "destination";
        Config config = RestoreDatabaseCommandIT.configWith(databaseName, this.directory.absolutePath().getAbsolutePath());
        File fromPath = new File(this.directory.absolutePath(), "from");
        File toPath = (File)config.get(GraphDatabaseSettings.database_path);
        this.createDbWithExplicitIndexAt(fromPath, 100);
        new RestoreDatabaseCommand(this.fileSystemRule.get(), fromPath, config, databaseName, true).execute();
        GraphDatabaseService restoredDatabase = this.createDatabase(toPath, toPath.getAbsolutePath());
        try (Transaction transaction = restoredDatabase.beginTx();){
            IndexManager indexManager = restoredDatabase.index();
            String[] nodeIndexNames = indexManager.nodeIndexNames();
            String[] relationshipIndexNames = indexManager.relationshipIndexNames();
            for (String nodeIndexName : nodeIndexNames) {
                RestoreDatabaseCommandIT.countNodesByKeyValue(indexManager, nodeIndexName, "a", "b");
                RestoreDatabaseCommandIT.countNodesByKeyValue(indexManager, nodeIndexName, "c", "d");
            }
            for (String relationshipIndexName : relationshipIndexNames) {
                RestoreDatabaseCommandIT.countRelationshipByKeyValue(indexManager, relationshipIndexName, "x", "y");
            }
        }
        restoredDatabase.shutdown();
    }

    @Test
    public void restoreTransactionLogsInCustomDirectoryForTargetDatabaseWhenConfigured() throws IOException, CommandFailed {
        String databaseName = "to";
        Config config = RestoreDatabaseCommandIT.configWith(databaseName, this.directory.absolutePath().getAbsolutePath());
        File customTxLogDirectory = this.directory.directory("customLogicalLog");
        String customTransactionLogDirectory = customTxLogDirectory.getAbsolutePath();
        config.augmentDefaults(GraphDatabaseSettings.logical_logs_location, customTransactionLogDirectory);
        File fromPath = new File(this.directory.absolutePath(), "from");
        File toPath = (File)config.get(GraphDatabaseSettings.database_path);
        int fromNodeCount = 10;
        int toNodeCount = 20;
        this.createDbAt(fromPath, fromNodeCount);
        GraphDatabaseService db = this.createDatabase(toPath, customTransactionLogDirectory);
        this.createTestData(toNodeCount, db);
        db.shutdown();
        new RestoreDatabaseCommand(this.fileSystemRule.get(), fromPath, config, databaseName, true).execute();
        LogFiles fromStoreLogFiles = LogFilesBuilder.logFilesBasedOnlyBuilder((File)fromPath, (FileSystemAbstraction)this.fileSystemRule.get()).build();
        LogFiles toStoreLogFiles = LogFilesBuilder.logFilesBasedOnlyBuilder((File)toPath, (FileSystemAbstraction)this.fileSystemRule.get()).build();
        LogFiles customLogLocationLogFiles = LogFilesBuilder.logFilesBasedOnlyBuilder((File)customTxLogDirectory, (FileSystemAbstraction)this.fileSystemRule.get()).build();
        Assert.assertThat((Object)toStoreLogFiles.logFiles(), (Matcher)Matchers.emptyArray());
        Assert.assertThat((Object)customLogLocationLogFiles.logFiles(), (Matcher)Matchers.arrayWithSize((int)1));
        Assert.assertEquals((long)fromStoreLogFiles.getLogFileForVersion(0L).length(), (long)customLogLocationLogFiles.getLogFileForVersion(0L).length());
    }

    @Test
    public void doNotRemoveRelativeTransactionDirectoryAgain() throws IOException, CommandFailed {
        FileSystemAbstraction fileSystem = (FileSystemAbstraction)Mockito.spy((Object)this.fileSystemRule.get());
        File fromPath = this.directory.directory("from");
        File databaseFile = this.directory.directory();
        File relativeLogDirectory = this.directory.directory("relativeDirectory");
        Config config = Config.defaults((Setting)GraphDatabaseSettings.database_path, (String)databaseFile.getAbsolutePath());
        config.augment(GraphDatabaseSettings.logical_logs_location, relativeLogDirectory.getAbsolutePath());
        this.createDbAt(fromPath, 10);
        new RestoreDatabaseCommand(fileSystem, fromPath, config, "testDatabase", true).execute();
        ((FileSystemAbstraction)Mockito.verify((Object)fileSystem)).deleteRecursively((File)ArgumentMatchers.eq((Object)databaseFile));
        ((FileSystemAbstraction)Mockito.verify((Object)fileSystem, (VerificationMode)Mockito.never())).deleteRecursively((File)ArgumentMatchers.eq((Object)relativeLogDirectory));
    }

    @Test
    public void shouldPrintNiceHelp() throws Throwable {
        try (ByteArrayOutputStream baos = new ByteArrayOutputStream();){
            PrintStream ps = new PrintStream(baos);
            Usage usage = new Usage("neo4j-admin", (CommandLocator)Mockito.mock(CommandLocator.class));
            usage.printUsageForCommand((AdminCommand.Provider)new RestoreDatabaseCliProvider(), ps::println);
            Assert.assertEquals((Object)String.format("usage: neo4j-admin restore --from=<backup-directory> [--database=<name>]%n                           [--force[=<true|false>]]%n%nenvironment variables:%n    NEO4J_CONF    Path to directory which contains neo4j.conf.%n    NEO4J_DEBUG   Set to anything to enable debug output.%n    NEO4J_HOME    Neo4j home directory.%n    HEAP_SIZE     Set JVM maximum heap size during command execution.%n                  Takes a number and a unit, for example 512m.%n%nRestore a backed up database.%n%noptions:%n  --from=<backup-directory>   Path to backup to restore from.%n  --database=<name>           Name of database. [default:graph.db]%n  --force=<true|false>        If an existing database should be replaced.%n                              [default:false]%n", new Object[0]), (Object)baos.toString());
        }
    }

    private static void countRelationshipByKeyValue(IndexManager indexManager, String indexName, String key, String value) {
        try (IndexHits nodes = indexManager.forRelationships(indexName).get(key, (Object)value);){
            Assert.assertEquals((long)50L, (long)nodes.size());
        }
    }

    private static void countNodesByKeyValue(IndexManager indexManager, String indexName, String key, String value) {
        try (IndexHits nodes = indexManager.forNodes(indexName).get(key, (Object)value);){
            Assert.assertEquals((long)50L, (long)nodes.size());
        }
    }

    private static Config configWith(String databaseName, String dataDirectory) {
        return Config.defaults((Map)MapUtil.stringMap((String[])new String[]{GraphDatabaseSettings.active_database.name(), databaseName, GraphDatabaseSettings.data_directory.name(), dataDirectory}));
    }

    private void createDbAt(File fromPath, int nodesToCreate) {
        GraphDatabaseService db = this.createDatabase(fromPath, fromPath.getAbsolutePath());
        this.createTestData(nodesToCreate, db);
        db.shutdown();
    }

    private GraphDatabaseService createDatabase(File fromPath, String absolutePath) {
        return new GraphDatabaseFactory().newEmbeddedDatabaseBuilder(fromPath).setConfig(OnlineBackupSettings.online_backup_enabled, "false").setConfig(GraphDatabaseSettings.logical_logs_location, absolutePath).newGraphDatabase();
    }

    private void createDbWithExplicitIndexAt(File fromPath, int pairNumberOfNodesToCreate) {
        RelationshipIndex explicitRelationshipIndex;
        Index explicitNodeIndex;
        GraphDatabaseService db = this.createDatabase(fromPath, fromPath.getAbsolutePath());
        try (Transaction transaction = db.beginTx();){
            explicitNodeIndex = db.index().forNodes("explicitNodeIndex");
            explicitRelationshipIndex = db.index().forRelationships("explicitRelationshipIndex");
            transaction.success();
        }
        var7_5 = null;
        try (Transaction tx = db.beginTx();){
            for (int i = 0; i < pairNumberOfNodesToCreate; i += 2) {
                Node node = db.createNode();
                Node otherNode = db.createNode();
                Relationship relationship = node.createRelationshipTo(otherNode, RelationshipType.withName((String)"rel"));
                explicitNodeIndex.add((PropertyContainer)node, "a", (Object)"b");
                explicitNodeIndex.add((PropertyContainer)otherNode, "c", (Object)"d");
                explicitRelationshipIndex.add((PropertyContainer)relationship, "x", (Object)"y");
            }
            tx.success();
        }
        catch (Throwable throwable) {
            var7_5 = throwable;
            throw throwable;
        }
        db.shutdown();
    }

    private void createTestData(int nodesToCreate, GraphDatabaseService db) {
        try (Transaction tx = db.beginTx();){
            for (int i = 0; i < nodesToCreate; ++i) {
                db.createNode();
            }
            tx.success();
        }
    }
}

