/*
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.neo4j.gds.core;

import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.neo4j.gds.core.ConfigKeyValidation.StringAndScore;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.TreeMap;

/**
 * Wrapper around configuration options map
 */
public final class CypherMapWrapper {

    private final Map<String, Object> config;

    private CypherMapWrapper(Map<String, Object> config) {
        this.config = config;
    }

    /**
     * Checks if the given key exists in the configuration.
     *
     * @param key key to look for
     * @return true, iff the key exists
     */
    public boolean containsKey(String key) {
        return this.config.containsKey(key);
    }

    public boolean isEmpty() {
        return config.isEmpty();
    }

    public Optional<String> getString(String key) {
        return Optional.ofNullable(getChecked(key, null, String.class));
    }

    public String requireString(String key) {
        return requireChecked(key, String.class);
    }

    @SuppressWarnings("unchecked")
    public Map<String, Object> getMap(String key) {
        return getChecked(key, Map.of(), Map.class);
    }

    @SuppressWarnings("unchecked")
    public List<String> getList(String key) {
        return getChecked(key, List.of(), List.class);
    }

    public <E> Optional<E> getOptional(String key, Class<E> clazz) {
        return Optional.ofNullable(getChecked(key, null, clazz));
    }

    @Contract("_, !null -> !null")
    public @Nullable String getString(String key, @Nullable String defaultValue) {
        return getChecked(key, defaultValue, String.class);
    }

    @Contract("_, _, !null -> !null")
    public @Nullable String getString(String key, String oldKey, @Nullable String defaultValue) {
        String value = getChecked(key, null, String.class);
        if (value != null) {
            return value;
        }
        return getChecked(oldKey, defaultValue, String.class);
    }

    public boolean getBool(String key, boolean defaultValue) {
        return getChecked(key, defaultValue, Boolean.class);
    }

    public boolean requireBool(String key) {
        return requireChecked(key, Boolean.class);
    }

    public Number getNumber(String key, Number defaultValue) {
        return getChecked(key, defaultValue, Number.class);
    }

    public Number requireNumber(String key) {
        return requireChecked(key, Number.class);
    }

    public Number getNumber(String key, String oldKey, Number defaultValue) {
        Number value = getChecked(key, null, Number.class);
        if (value != null) {
            return value;
        }
        return getChecked(oldKey, defaultValue, Number.class);
    }

    public long getLong(String key, long defaultValue) {
        return getChecked(key, defaultValue, Long.class);
    }

    public long requireLong(String key) {
        return requireChecked(key, Long.class);
    }

    public int getInt(String key, int defaultValue) {
        if (!containsKey(key)) {
            return defaultValue;
        }
        return getLongAsInt(key);
    }

    public int requireInt(String key) {
        if (!containsKey(key)) {
            throw missingValueFor(key);
        }
        return getLongAsInt(key);
    }

    private int getLongAsInt(String key) {
        Object value = config.get(key);
        // Cypher always uses longs, so we have to downcast them to ints
        if (value instanceof Long) {
            value = Math.toIntExact((Long) value);
        }
        return typedValue(key, Integer.class, value);
    }

    public double getDouble(String key, double defaultValue) {
        return getChecked(key, defaultValue, Double.class);
    }

    /**
     * Returns a copy of the internal Map.
     */
    public Map<String, Object> toMap() {
        return new HashMap<>(config);
    }

    public double requireDouble(String key) {
        return requireChecked(key, Double.class);
    }

    /**
     * Get and convert the value under the given key to the given type.
     *
     * @return the found value under the key - if it is of the provided type,
     *     or the provided default value if no entry for the key is found (or it's mapped to null).
     * @throws IllegalArgumentException if a value was found, but it is not of the expected type.
     */
    @Contract("_, !null, _ -> !null")
    public @Nullable <V> V getChecked(String key, @Nullable V defaultValue, Class<V> expectedType) {
        if (!containsKey(key)) {
            return defaultValue;
        }
        return typedValue(key, expectedType, config.get(key));
    }

    public <V> V requireChecked(String key, Class<V> expectedType) {
        if (!containsKey(key)) {
            throw missingValueFor(key);
        }
        return typedValue(key, expectedType, config.get(key));
    }

    public void requireOnlyKeysFrom(Collection<String> allowedKeys) {
        ConfigKeyValidation.requireOnlyKeysFrom(allowedKeys, config.keySet());
    }

    @SuppressWarnings("unchecked")
    @Contract("_, !null -> !null")
    @Deprecated
    public <V> @Nullable V get(String key, @Nullable V defaultValue) {
        Object value = config.get(key);
        if (null == value) {
            return defaultValue;
        }
        return (V) value;
    }

    @SuppressWarnings("unchecked")
    @Contract("_, _, !null -> !null")
    @Deprecated
    public <V> @Nullable V get(String newKey, String oldKey, @Nullable V defaultValue) {
        Object value = config.get(newKey);
        if (null == value) {
            value = config.get(oldKey);
        }
        return null == value ? defaultValue : (V) value;
    }

    public static <T> T failOnNull(String key, T value) {
        if (value == null) {
            throw missingValueFor(key, Collections.emptySet());
        }
        return value;
    }

    public static @NotNull String failOnBlank(String key, @Nullable String value) {
        if (value == null || value.trim().isEmpty()) {
            throw blankValueFor(key, value);
        }
        return value;
    }

    public static int validateIntegerRange(
        String key,
        int value,
        int min,
        int max,
        boolean minInclusive,
        boolean maxInclusive
    ) {
        boolean meetsLowerBound = minInclusive ? value >= min : value > min;
        boolean meetsUpperBound = maxInclusive ? value <= max : value < max;

        if (!meetsLowerBound || !meetsUpperBound) {
            throw outOfRangeError(key, value, Integer.toString(min), Integer.toString(max), minInclusive, maxInclusive);
        }

        return value;
    }

    public static long validateLongRange(
        String key,
        long value,
        long min,
        long max,
        boolean minInclusive,
        boolean maxInclusive
    ) {
        boolean meetsLowerBound = minInclusive ? value >= min : value > min;
        boolean meetsUpperBound = maxInclusive ? value <= max : value < max;

        if (!meetsLowerBound || !meetsUpperBound) {
            throw outOfRangeError(key, value, Long.toString(min), Long.toString(max), minInclusive, maxInclusive);
        }

        return value;
    }

    public static double validateDoubleRange(
        String key,
        double value,
        double min,
        double max,
        boolean minInclusive,
        boolean maxInclusive
    ) {
        boolean meetsLowerBound = minInclusive ? value >= min : value > min;
        boolean meetsUpperBound = maxInclusive ? value <= max : value < max;

        if (!meetsLowerBound || !meetsUpperBound) {
            throw outOfRangeError(
                key,
                value,
                String.format(Locale.ENGLISH, "%.2f", min),
                String.format(Locale.ENGLISH, "%.2f", max),
                minInclusive,
                maxInclusive
            );
        }

        return value;
    }

    private static <V> V typedValue(String key, Class<V> expectedType, @Nullable Object value) {
        if (Double.class.isAssignableFrom(expectedType) && value instanceof Number) {
            return expectedType.cast(((Number) value).doubleValue());
        } else if (expectedType.equals(Integer.class) && value instanceof Long) {
            return expectedType.cast(Math.toIntExact((Long) value));
        } else if (!expectedType.isInstance(value)) {
            String message = String.format(
                Locale.ENGLISH,
                "The value of `%s` must be of type `%s` but was `%s`.",
                key,
                expectedType.getSimpleName(),
                value == null ? "null" : value.getClass().getSimpleName()
            );
            throw new IllegalArgumentException(message);
        }
        return expectedType.cast(value);
    }

    private IllegalArgumentException missingValueFor(String key) {
        return missingValueFor(key, config.keySet());
    }

    private static IllegalArgumentException missingValueFor(String key, Collection<String> candidates) {
        return new IllegalArgumentException(missingValueForMessage(key, candidates));
    }

    private static String missingValueForMessage(String key, Collection<String> candidates) {
        List<String> suggestions = StringSimilarity.similarStringsIgnoreCase(key, candidates);
        return missingValueMessage(key, suggestions);
    }

    private static String missingValueMessage(String key, List<String> suggestions) {
        if (suggestions.isEmpty()) {
            return String.format(
                Locale.US,
                "No value specified for the mandatory configuration parameter `%s`",
                key
            );
        }
        if (suggestions.size() == 1) {
            return String.format(
                Locale.ENGLISH,
                "No value specified for the mandatory configuration parameter `%s` (a similar parameter exists: [%s])",
                key,
                suggestions.get(0)
            );
        }
        return String.format(
            Locale.ENGLISH,
            "No value specified for the mandatory configuration parameter `%s` (similar parameters exist: [%s])",
            key,
            String.join(", ", suggestions)
        );
    }

    public enum PairResult {
        FIRST_PAIR,
        SECOND_PAIR,
    }

    /**
     * Verifies that only one of two mutually exclusive pairs of configuration keys is present.
     *
     * More precisely, the following condition is checked:
     *  {@code (firstPairKeyOne AND firstPairKeyTwo) XOR (secondPairKeyOne AND secondPairKeyTwo)}
     * If the condition is verified, the return value will identify which one of the pairs is present.
     *
     * In the error case where the condition is violated, an {@link IllegalArgumentException} is thrown.
     * The message of that exception depends on which keys are present, possible mis-spelled, or absent.
     */
    public PairResult verifyMutuallyExclusivePairs(
        String firstPairKeyOne,
        String firstPairKeyTwo,
        String secondPairKeyOne,
        String secondPairKeyTwo,
        String errorPrefix
    ) throws IllegalArgumentException {
        boolean isValidFirstPair = checkMutuallyExclusivePairs(
            firstPairKeyOne, firstPairKeyTwo, secondPairKeyOne, secondPairKeyTwo
        );
        if (isValidFirstPair) {
            return PairResult.FIRST_PAIR;
        }

        boolean isValidSecondPair = checkMutuallyExclusivePairs(
            secondPairKeyOne, secondPairKeyTwo, firstPairKeyOne, firstPairKeyTwo
        );
        if (isValidSecondPair) {
            return PairResult.SECOND_PAIR;
        }

        String message = missingMutuallyExclusivePairMessage(firstPairKeyOne, firstPairKeyTwo, secondPairKeyOne, secondPairKeyTwo);
        throw new IllegalArgumentException(String.format(Locale.ENGLISH,"%s %s", errorPrefix, message));
    }

    private boolean checkMutuallyExclusivePairs(
        String firstPairKeyOne,
        String firstPairKeyTwo,
        String secondPairKeyOne,
        String secondPairKeyTwo
    ) throws IllegalArgumentException {
        if (config.containsKey(firstPairKeyOne) && config.containsKey(firstPairKeyTwo)) {
            boolean secondOneExists = config.containsKey(secondPairKeyOne);
            boolean secondTwoExists = config.containsKey(secondPairKeyTwo);
            if (secondOneExists && secondTwoExists) {
                throw new IllegalArgumentException(String.format(
                    Locale.ENGLISH,
                    "Invalid keys: [%s, %s]. Those keys cannot be used together with `%s` and `%s`.",
                    secondPairKeyOne,
                    secondPairKeyTwo,
                    firstPairKeyOne,
                    firstPairKeyTwo
                ));
            } else if (secondOneExists || secondTwoExists) {
                throw new IllegalArgumentException(String.format(
                    Locale.ENGLISH,
                    "Invalid key: [%s]. This key cannot be used together with `%s` and `%s`.",
                    secondOneExists ? secondPairKeyOne : secondPairKeyTwo,
                    firstPairKeyOne,
                    firstPairKeyTwo
                ));
            }
            return true;
        }
        return false;
    }

    private String missingMutuallyExclusivePairMessage(
        String firstPairKeyOne,
        String firstPairKeyTwo,
        String secondPairKeyOne,
        String secondPairKeyTwo
    ) {
        StringAndScore firstMessage = missingMutuallyExclusivePairs(firstPairKeyOne, firstPairKeyTwo, secondPairKeyOne, secondPairKeyTwo);
        StringAndScore secondMessage = missingMutuallyExclusivePairs(secondPairKeyOne, secondPairKeyTwo, firstPairKeyOne, firstPairKeyTwo);

        if (firstMessage != null && firstMessage.isBetterThan(secondMessage)) {
            // only return if the second message does not have a competitive score
            return firstMessage.string();
        }

        if (secondMessage != null && secondMessage.isBetterThan(firstMessage)) {
            // only return if the first message does not have a competitive score
            return secondMessage.string();
        }

        // either pairs have the same possibility score, we don't know which one we should use
        return String.format(
            Locale.ENGLISH,
            "Specify either `%s` and `%s` or `%s` and `%s`.",
            firstPairKeyOne,
            firstPairKeyTwo,
            secondPairKeyOne,
            secondPairKeyTwo
        );
    }

    private @Nullable StringAndScore missingMutuallyExclusivePairs(
        String keyOne,
        String keyTwo,
        String... forbiddenSuggestions
    ) {
        Collection<String> missingAndCandidates = new ArrayList<>();
        Collection<String> missingWithoutCandidates = new ArrayList<>();
        boolean hasAtLastOneKey = false;
        for (String key : List.of(keyOne, keyTwo)) {
            if (config.containsKey(key)) {
                hasAtLastOneKey = true;
            } else {
                List<String> candidates = StringSimilarity.similarStringsIgnoreCase(key, config.keySet());
                candidates.removeAll(List.of(forbiddenSuggestions));
                String message = missingValueMessage(key, candidates);
                (candidates.isEmpty()
                    ? missingWithoutCandidates
                    : missingAndCandidates
                ).add(message);
            }
        }
        // if one of the keys matches, we give it a full score,
        //   meaning "this is probably a pair that should be used"
        // if one of the keys is mis-spelled, we give it a half score,
        //   meaning "this could be that pair, but it might me something else"
        // If none of the keys are present or mis-spelled, we give it a zero score,
        //   meaning "This is not pair you are looking for" *waves hand*
        double score = hasAtLastOneKey ? 1.0 : !missingAndCandidates.isEmpty() ? 0.5 : 0.0;
        if (!missingAndCandidates.isEmpty()) {
            missingAndCandidates.addAll(missingWithoutCandidates);
            String message = String.join(". ", missingAndCandidates);
            return ImmutableStringAndScore.of(message, score);
        }
        if (hasAtLastOneKey && !missingWithoutCandidates.isEmpty()) {
            String message = String.join(". ", missingWithoutCandidates);
            return ImmutableStringAndScore.of(message, score);
        }
        // null here means, that there are no valid keys, but also no good error message
        // so it might be that this pair is not relevant for the error reporting
        return null;
    }

    private static IllegalArgumentException blankValueFor(String key, @Nullable String value) {
        return new IllegalArgumentException(String.format(
            Locale.ENGLISH,
            "`%s` can not be null or blank, but it was `%s`",
            key,
            value
        ));
    }

    private static IllegalArgumentException outOfRangeError(
        String key,
        Number value,
        String min,
        String max,
        boolean minInclusive,
        boolean maxInclusive
    ) {
        return new IllegalArgumentException(String.format(
            Locale.ENGLISH,
            "Value for `%s` was `%s`, but must be within the range %s%s, %s%s.",
            key,
            value,
            minInclusive ? "[" : "(",
            min,
            max,
            maxInclusive ? "]" : ")"
        ));
    }

    // FACTORIES

    public static CypherMapWrapper create(Map<String, Object> config) {
        if (config == null) {
            return empty();
        }
        Map<String, Object> configMap = new TreeMap<>(String.CASE_INSENSITIVE_ORDER);
        config.forEach((key, value) -> {
            if (value != null) {
                configMap.put(key, value);
            }
        });
        return new CypherMapWrapper(configMap);
    }

    public static CypherMapWrapper empty() {
        return new CypherMapWrapper(Map.of());
    }

    public CypherMapWrapper withString(String key, String value) {
        return withEntry(key, value);
    }

    public CypherMapWrapper withNumber(String key, Number value) {
        return withEntry(key, value);
    }

    public CypherMapWrapper withBoolean(String key, Boolean value) {
        return withEntry(key, value);
    }

    public CypherMapWrapper withEntry(String key, Object value) {
        Map<String, Object> newMap = copyValues();
        newMap.put(key, value);
        return new CypherMapWrapper(newMap);
    }

    public CypherMapWrapper withoutEntry(String key) {
        if (!containsKey(key)) {
            return this;
        }
        Map<String, Object> newMap = copyValues();
        newMap.remove(key);
        return new CypherMapWrapper(newMap);
    }

    public CypherMapWrapper withoutAny(Collection<String> keys) {
        Map<String, Object> newMap = copyValues();
        newMap.keySet().removeAll(keys);
        return new CypherMapWrapper(newMap);
    }

    private Map<String, Object> copyValues() {
        Map<String, Object> copiedMap = new TreeMap<>(String.CASE_INSENSITIVE_ORDER);
        copiedMap.putAll(config);
        return copiedMap;
    }
}
