/*
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.neo4j.gds.ml.nodemodels.pipeline.predict;

import org.neo4j.gds.NodeLabel;
import org.neo4j.gds.api.GraphStore;
import org.neo4j.gds.executor.validation.AfterLoadValidation;
import org.neo4j.gds.executor.validation.ValidationConfiguration;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import static org.neo4j.gds.utils.StringFormatting.formatWithLocale;

public final class NodeClassificationPipelineCompanion {

    static <CONFIG extends NodeClassificationPredictPipelineBaseConfig> ValidationConfiguration<CONFIG> getValidationConfig() {
        return new ValidationConfiguration<>() {
            @Override
            public List<AfterLoadValidation<CONFIG>> afterLoadValidations() {
                return List.of(
                    (graphStore, graphProjectConfig, config) -> {
                        if (config instanceof NodeClassificationPredictPipelineMutateConfig) {
                            var mutateConfig = (NodeClassificationPredictPipelineMutateConfig) config;

                            validatePredictedProbabilityPropertyDoesNotExist(
                                graphStore,
                                mutateConfig.nodeLabelIdentifiers(graphStore),
                                mutateConfig.predictedProbabilityProperty()
                            );

                            validateProperties(
                                mutateConfig.mutateProperty(),
                                mutateConfig.predictedProbabilityProperty()
                            );
                        }
                    }
                );
            }
        };
    }

    static void prepareTrainConfig(
        Object graphNameOrConfiguration,
        Map<String, Object> algoConfiguration
    ) {
        // TODO: this will go away once node property steps do not modify the graph store with the given graphName
        //  In the future it might operate on a shallow copy instead.
        if (graphNameOrConfiguration instanceof String) {
            algoConfiguration.put("graphName", graphNameOrConfiguration);
        } else {
            algoConfiguration.put("graphName", "__ANONYMOUS_GRAPH__");
        }
    }

    private static void validatePredictedProbabilityPropertyDoesNotExist(
        GraphStore graphStore,
        Collection<NodeLabel> filterLabels,
        Optional<String> maybePredictedProbabilityProperty
    ) {
        maybePredictedProbabilityProperty.ifPresent(predictedProbabilityProperty -> {
            if (graphStore.hasNodeProperty(filterLabels, predictedProbabilityProperty)) {
                throw new IllegalArgumentException(formatWithLocale(
                    "Node property `%s` already exists in the in-memory graph.",
                    predictedProbabilityProperty
                ));
            }
        });
    }

    private static void validateProperties(String property, Optional<String> maybePredictedProbabilityProperty) {
        maybePredictedProbabilityProperty.ifPresent(predictedProbabilityProperty -> {
            if (property.equals(predictedProbabilityProperty)) {
                throw new IllegalArgumentException(
                    formatWithLocale(
                        "Configuration parameters `%s` and `%s` must be different (both were `%s`)",
                        "mutateProperty",
                        "predictedProbabilityProperty",
                        predictedProbabilityProperty
                    )
                );
            }
        });
    }


    private NodeClassificationPipelineCompanion() {

    }
}
