package org.nekosoft.pdffer.explorer;

import org.nekosoft.pdffer.PdfferProducerBean;
import org.nekosoft.pdffer.exception.PdfferMailerException;
import org.nekosoft.pdffer.mail.PdfferMailerBean;
import org.nekosoft.pdffer.registry.PdfferRegistryBean;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.context.annotation.Profile;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import static org.nekosoft.pdffer.PdfferCoreConfiguration.PROFILE_EXPLORER;
import static org.springframework.http.MediaType.TEXT_HTML_VALUE;

/**
 * A controller that is made available when PDFfer mailing functionality is present. It holds a reference to the mailer
 * and provides a form for emailing a PDF generated with a template, as well as an endpoint to handle the form.
 */
@Controller
@RequestMapping("explorer")
@ConditionalOnBean(PdfferMailerBean.class)
@Profile(PROFILE_EXPLORER)
public class PdfferMailerExplorerController extends AbstractPdfferExplorerController {

    // The template resource for the /email page
    @Value("classpath:org/nekosoft/pdffer/explorer/email.html")
    private Resource emailHtmlTemplate;

    private final PdfferMailerBean mailer;

    public PdfferMailerExplorerController(PdfferProducerBean pdffer, PdfferRegistryBean registry, PdfferMailerBean mailer) {
        super(pdffer, registry);
        this.mailer = mailer;
    }

    /**
     * Controller method mapped to GET /email
     * @return the HTML content for the email form
     */
    @GetMapping("email")
    @ResponseBody
    public String emailForm() {
        return htmlTemplateAsString(emailHtmlTemplate, "");
    }

    /**
     * Processes an email form and sends an email with a PDF attachment.
     *
     * @param templatePath the path of the template to be used for the PDF
     * @param payload      the payload as a JSON string
     * @param sendEmails   who to send the email to (comma-separated list of email addresses)
     * @param mailSubject  the subject of the email
     * @param mailMessage  the body of the email message
     * @return the response entity with the HTML to display
     */
    @PostMapping(value = "email", produces = TEXT_HTML_VALUE)
    @ResponseBody
    public ResponseEntity<String> processEmailForm(@RequestParam("pdfTemplatePath") String templatePath, @RequestParam("pdfPayload") String payload, @RequestParam("sendEmails") String sendEmails, @RequestParam("mailSubject") String mailSubject, @RequestParam("mailMessage") String mailMessage) {
        try {
            byte[] pdfBytes = pdffer.generatePdfDocumentByPathFromJsonString(templatePath, payload);
            mailer.sendMessageWithPdfAttachment(sendEmails, mailSubject, mailMessage, pdfBytes, "pdffer.pdf", null, null);
            String body = htmlTemplateAsString(emailHtmlTemplate, "<p class=\"text-success\">Email with PDF was successfully sent to " + sendEmails + "</p>");
            return new ResponseEntity<>(body, HttpStatus.OK);
        } catch (PdfferMailerException e) {
            return new ResponseEntity<>(HttpStatus.INTERNAL_SERVER_ERROR);
        } catch (IllegalArgumentException e) {
            return new ResponseEntity<>(HttpStatus.BAD_REQUEST);
        }
    }

}
